use nanorand::tls::TlsWyRand;
use nanorand::RNG;

/// Factor for converting random u64 to uniformly distributed double.
const FACTOR: f64 = 5.421010862427522e-20; // 2.0_f64.powi(-64)

/// Generate a random floating point number between min and max.
pub fn random_float(rng: &mut TlsWyRand, min: f64, max: f64) -> f64 {
    // Casting to signed before casting to float can be faster depending on your hardware.
    // See Doornik, J.A.: Conversion of high-period random numbers to floating point,
    // CM transactions on modeling and computer simulation, 17(1) (2007)
    (rng.generate::<u64>() as i64 as f64 * FACTOR + 0.5) * (max - min) + min
}

/// A struct containing common index of refraction values as constants.
#[non_exhaustive]
pub struct IOR;

impl IOR {
    pub const ACRYLIC_GLASS: f64 = 1.491;
    pub const AIR: f64 = 1.0;
    pub const ALCOHOL: f64 = 1.36;
    pub const BEER: f64 = 1.345;
    pub const CRYSTAL: f64 = 2.0;
    pub const DIAMOND: f64 = 2.418;
    pub const EMERALD: f64 = 1.583;
    pub const GLASS: f64 = 1.5;
    pub const ICE: f64 = 1.309;
    pub const LUCITE: f64 = 1.495;
    pub const MERCURY: f64 = 1.62;
    pub const PEARL: f64 = 1.61;
    pub const VODKA: f64 = 1.363;
    pub const WATER: f64 = 1.325;
}
