use crate::color::{Color, Texture};
use crate::rendering::Camera;
use crate::shapes::Shape;

/// The scene, consisting of camera, objects and background.
pub struct Scene {
    camera: Camera,
    objects: Vec<Box<dyn Shape>>,
    background: Box<dyn Texture>,
    tile_size_x: usize,
    tile_size_y: usize,
}

impl Default for Scene {
    fn default() -> Self {
        Self::new()
    }
}

impl Scene {
    /// Create a new scene.
    pub fn new() -> Self {
        Scene {
            camera: Camera::new(),
            objects: vec![],
            background: Box::new(Color::new(0.8, 0.8, 0.8)),
            tile_size_x: 16,
            tile_size_y: 16,
        }
    }

    /// Set the camera of the scene and return self (builder pattern).
    pub fn with_camera(mut self, camera: Camera) -> Self {
        self.camera = camera;
        self
    }
    /// Return the camera of the scene.
    pub fn camera(&self) -> Camera {
        self.camera
    }
    /// Set the camera of the scene.
    pub fn set_camera(&mut self, camera: Camera) {
        self.camera = camera
    }

    /// Set the objects of the scene and return self (builder pattern).
    pub fn with_objects(mut self, objects: Vec<Box<dyn Shape>>) -> Self {
        self.objects = objects;
        self
    }
    /// Return the objects of the scene.
    pub fn objects(&self) -> &Vec<Box<dyn Shape>> {
        &self.objects
    }
    /// Set the objects of the scene.
    pub fn set_objects(&mut self, objects: Vec<Box<dyn Shape>>) {
        self.objects = objects;
    }

    /// Set the background of the scene and return self (builder pattern).
    pub fn with_background(mut self, background: Box<dyn Texture>) -> Self {
        self.background = background;
        self
    }
    /// Return the background of the scene.
    pub fn background(&self) -> &Box<dyn Texture> {
        &self.background
    }
    /// Set the background of the scene.
    pub fn set_background(&mut self, background: Box<dyn Texture>) {
        self.background = background;
    }

    /// Set the tile size of the scene in x direction and return self (builder pattern).
    pub fn with_tile_size_x(mut self, tile_size_x: usize) -> Self {
        self.tile_size_x = tile_size_x;
        self
    }
    /// Return the tile size of the scene in x direction.
    pub fn tile_size_x(&self) -> usize {
        self.tile_size_x
    }
    /// Set the tile size of the scene in x direction.
    pub fn set_tile_size_x(&mut self, tile_size_x: usize) {
        self.tile_size_x = tile_size_x;
    }

    /// Set the tile size in y direction of the scene and return self (builder pattern).
    pub fn with_tile_size_y(mut self, tile_size_y: usize) -> Self {
        self.tile_size_y = tile_size_y;
        self
    }
    /// Return the tile size of the scene in y direction.
    pub fn tile_size_y(&self) -> usize {
        self.tile_size_y
    }
    /// Set the tile size of the scene in y direction.
    pub fn set_tile_size_y(&mut self, tile_size_y: usize) {
        self.tile_size_y = tile_size_y;
    }
}
