use std::sync::{Arc, Mutex};
use std::time::Instant;

use indicatif::{ProgressBar, ProgressStyle};
use nanorand::tls::TlsWyRand;
use nanorand::tls_rng;
use rayon::prelude::*;

use crate::color::{Color, Image, ImageTile};
use crate::math::{Hit, Ray};
use crate::rendering::Scene;
use crate::shapes::{Shape, Sphere};
use crate::util::random_float;

/// The render engine.
#[derive(Copy, Clone, Debug, Eq, PartialEq, Hash)]
pub struct RenderEngine {
    samples: u32,
    max_bounces: usize,
}

impl RenderEngine {
    pub fn new(samples: u32, max_bounces: usize) -> Self {
        RenderEngine {
            samples,
            max_bounces,
        }
    }

    /// Return the samples of the render engine.
    pub fn samples(&self) -> u32 {
        self.samples
    }
    /// Set the samples of the render engine.
    pub fn set_samples(&mut self, samples: u32) {
        self.samples = samples;
    }

    /// Return the maximum amount of bounces of the render engine.
    pub fn max_bounces(&self) -> usize {
        self.max_bounces
    }
    /// Set the maximum amount of bounces of the render engine.
    pub fn set_max_bounces(&mut self, max_bounces: usize) {
        self.max_bounces = max_bounces;
    }

    /// Return the closest hit of a ray with objects of the scene, if any.
    fn get_closest_hit<'a>(&self, scene: &'a Scene, ray: Ray) -> Option<Hit<'a>> {
        let mut closest_hit: Option<Hit> = None;
        for object in scene.objects() {
            if let Some(hit) = object.intersects(ray) {
                match closest_hit {
                    None => {
                        closest_hit = Some(hit);
                    }
                    Some(old_closest_hit) => {
                        if old_closest_hit.distance() > hit.distance() {
                            closest_hit = Some(hit);
                        }
                    }
                }
            }
        }

        closest_hit
    }

    /// Trace a ray and return the resulting color.
    fn trace_tray(&self, scene: &Scene, ray: Ray, bounces: usize, rng: &mut TlsWyRand) -> Color {
        if bounces > self.max_bounces() {
            return Color::default();
        }

        let closest_hit = self.get_closest_hit(&scene, ray);

        return match closest_hit {
            None => {
                let (u, v) = Sphere::new().uv_map(ray.direction().normalize());

                scene.background().get_pixel(u, v)
            }
            Some(hit) => {
                let (u, v) = hit.uv();
                let color = hit.material().get_color(u, v);

                match hit.material().next_ray(ray, hit, rng) {
                    None => color,
                    Some(ray) => self.trace_tray(&scene, ray, bounces + 1, rng) * color,
                }
            }
        };
    }

    /// Render the give scene.
    pub fn render(&self, scene: &Scene) -> Image {
        let (width, height) = scene.camera().dimensions();

        println!(
            "Rendering {} objects with {} samples ({} bounces) to a {}x{} image.",
            scene.objects().len(),
            self.samples(),
            self.max_bounces(),
            width,
            height
        );

        let tile_size_x = scene.tile_size_x().min(width - 1);
        let tile_size_y = scene.tile_size_y().min(height - 1);

        let mut tiles = vec![];
        for x in 0..(width as f64 / tile_size_x as f64).ceil() as usize {
            for y in 0..(height as f64 / tile_size_y as f64).ceil() as usize {
                tiles.push(ImageTile::new(x, y, Image::new(tile_size_x, tile_size_y)))
            }
        }

        let progress_bar = ProgressBar::new((tiles.len()) as u64);
        progress_bar.set_style(
            ProgressStyle::default_bar()
                .template(
                    "[{elapsed_precise}] [{bar:60}] {percent:>5}% (ETA: ~{eta_precise}) {msg}",
                )
                .progress_chars("=> "),
        );
        progress_bar.set_position(0);
        let progress_bar = Arc::new(Mutex::new(progress_bar));

        let start_time = Instant::now();

        tiles.par_iter_mut().for_each(|tile| {
            let tile_x = tile.x();
            let tile_y = tile.y();

            tile.image_mut()
                .pixels_mut()
                .iter_mut()
                .enumerate()
                .for_each(|(y, row)| {
                    row.iter_mut().enumerate().for_each(|(x, pixel)| {
                        let x = x + scene.tile_size_x() * tile_x;
                        let y = y + scene.tile_size_y() * tile_y;

                        let mut rng = tls_rng();
                        let mut color = Color::default();

                        for _ in 0..self.samples() {
                            let u = (x as f64 + random_float(&mut rng, 0.0, 1.0))
                                / (width as f64 - 1.0);
                            let v = (height as f64 - (y as f64) + random_float(&mut rng, 0.0, 1.0))
                                / (height as f64 - 1.0);

                            let ray = scene.camera().get_ray(u, v, &mut rng);

                            color += self.trace_tray(&scene, ray, 0, &mut rng);
                        }

                        *pixel = color / self.samples() as f64;
                        *pixel = Color::new(
                            pixel.r().min(1.0).max(0.0),
                            pixel.g().min(1.0).max(0.0),
                            pixel.b().min(1.0).max(0.0),
                        );
                    });
                });

            progress_bar.lock().unwrap().inc(1);
        });

        progress_bar.lock().unwrap().finish_with_message(format!(
            "\nRendering finished after {:.2?}",
            Instant::now() - start_time
        ));

        Image::from_tiles(width, height, tiles)
    }
}
