use std::fmt::Debug;

use nanorand::tls::TlsWyRand;

use crate::color::{Color, Texture};
use crate::math::{Hit, Ray, Vec3};
use crate::util::IOR;

/// Characterize optical properties of a surface.
pub trait Material: Debug + Send + Sync {
    /// Generate the next ray to be traced.
    fn next_ray(&self, ray: Ray, hit: Hit, rng: &mut TlsWyRand) -> Option<Ray>;

    /// Return the color at the given coordinates.
    fn get_color(&self, u: f64, v: f64) -> Color;
}

/// A material that scatters diffusely according to Lambert's law.
#[derive(Copy, Clone, Debug)]
pub struct Diffuse<T: Texture> {
    texture: T,
}

impl<T: Texture> Material for Diffuse<T> {
    fn next_ray(&self, _ray: Ray, hit: Hit, rng: &mut TlsWyRand) -> Option<Ray> {
        let direction = hit.normal() + Vec3::random_unit_vector(rng);
        Some(Ray::new(hit.position(), direction, IOR::AIR))
    }

    fn get_color(&self, u: f64, v: f64) -> Color {
        self.texture.get_pixel(u, v)
    }
}

impl<T: Texture> Diffuse<T> {
    pub fn new(texture: T) -> Self {
        Diffuse { texture }
    }
}

/// A material that is rough and reflecting.
#[derive(Copy, Clone, Debug)]
pub struct Metallic<T: Texture> {
    texture: T,
    roughness: f64,
}

impl<T: Texture> Material for Metallic<T> {
    fn next_ray(&self, ray: Ray, hit: Hit, rng: &mut TlsWyRand) -> Option<Ray> {
        let direction = ray.direction().reflect(hit.normal());
        Some(Ray::new(
            hit.position(),
            direction + self.roughness * Vec3::random_in_unit_sphere(rng),
            IOR::AIR,
        ))
    }

    fn get_color(&self, u: f64, v: f64) -> Color {
        self.texture.get_pixel(u, v)
    }
}

impl<T: Texture> Metallic<T> {
    pub fn new(texture: T, roughness: f64) -> Self {
        Metallic { texture, roughness }
    }
}

/// A material that emits light.
#[derive(Copy, Clone, Debug)]
pub struct Emissive<T: Texture> {
    texture: T,
    intensity: f64,
}

impl<T: Texture> Material for Emissive<T> {
    fn next_ray(&self, _ray: Ray, _hit: Hit, _rng: &mut TlsWyRand) -> Option<Ray> {
        None
    }

    fn get_color(&self, u: f64, v: f64) -> Color {
        self.texture.get_pixel(u, v) * self.intensity
    }
}

impl<T: Texture> Emissive<T> {
    pub fn new(texture: T, intensity: f64) -> Self {
        Emissive { texture, intensity }
    }
}

/// A material that is rough and refracts and reflects light.
#[derive(Copy, Clone, Debug)]
pub struct Refractive<T: Texture> {
    texture: T,
    ior: f64,
    roughness: f64,
}

impl<T: Texture> Material for Refractive<T> {
    fn next_ray(&self, ray: Ray, hit: Hit, rng: &mut TlsWyRand) -> Option<Ray> {
        let (n1, n2) = (ray.last_ior(), self.ior);
        let direction = ray.direction().refract(hit.normal(), n1, n2, rng);

        Some(Ray::new(
            hit.position(),
            direction + self.roughness * Vec3::random_in_unit_sphere(rng),
            self.ior,
        ))
    }

    fn get_color(&self, u: f64, v: f64) -> Color {
        self.texture.get_pixel(u, v)
    }
}

impl<T: Texture> Refractive<T> {
    pub fn new(texture: T, ior: f64, roughness: f64) -> Self {
        Refractive {
            texture,
            ior,
            roughness,
        }
    }
}
