use crate::math::Vec3;

/// A ray (half-line).
#[derive(Copy, Clone, Debug, PartialEq)]
pub struct Ray {
    origin: Vec3,
    direction: Vec3,
    last_ior: f64,
}

impl Ray {
    /// Create a new ray.
    pub fn new(origin: Vec3, direction: Vec3, last_ior: f64) -> Self {
        let direction = direction.normalize();
        Ray {
            // Shift the intersection point slightly outwards in order to avoid problems due to finite floating point precision.
            origin: origin + direction * 1e-9,
            direction,
            last_ior,
        }
    }

    /// Return the origin of the ray.
    pub fn origin(&self) -> Vec3 {
        self.origin
    }

    /// Return the direction of the ray
    pub fn direction(&self) -> Vec3 {
        self.direction
    }

    /// Return the index of refraction of the material the ray came from.
    pub fn last_ior(&self) -> f64 {
        self.last_ior
    }

    /// Return the point on the ray at the given distance.
    pub fn get_point(&self, distance: f64) -> Vec3 {
        self.origin + self.direction * distance
    }
}

#[cfg(test)]
mod tests {
    use assert_approx_eq::assert_approx_eq;

    use super::{Ray, Vec3};

    #[test]
    fn ray_get_point() {
        assert_approx_eq!(
            Ray::new(Vec3::new(0.0, 0.0, 0.0), Vec3::new(1.0, 0.0, 0.0), 1.0).get_point(2.0),
            Vec3::new(2.0, 0.0, 0.0)
        );
        assert_approx_eq!(
            Ray::new(
                Vec3::new(5.72, 2.5, 8.824),
                Vec3::new(8.7, 5.987, 0.12),
                1.0
            )
            .get_point(2.5),
            Vec3::new(7.779336, 3.917155, 8.852405)
        );
    }
}
