use crate::math::Vec3;
use crate::rendering::Material;

/// The intersection of a ray with an object.
#[derive(Copy, Clone, Debug)]
pub struct Hit<'a> {
    position: Vec3,
    normal: Vec3,
    distance: f64,
    material: &'a Box<dyn Material>,
    uv: (f64, f64),
}

impl<'a> Hit<'a> {
    /// Create a new hit.
    pub fn new(
        position: Vec3,
        normal: Vec3,
        distance: f64,
        material: &'a Box<dyn Material>,
        uv: (f64, f64),
    ) -> Self {
        Hit {
            position,
            normal,
            distance,
            material,
            uv,
        }
    }

    /// Return the position of the intersection point.
    pub fn position(&self) -> Vec3 {
        self.position
    }

    /// Return the normal of the object at the intersection point.
    pub fn normal(&self) -> Vec3 {
        self.normal
    }

    /// Return the distance of the intersection point from the ray origin.
    pub fn distance(&self) -> f64 {
        self.distance
    }

    /// Return the object that is hit by the ray.
    pub fn material(&self) -> &Box<dyn Material> {
        &self.material
    }

    pub fn uv(&self) -> (f64, f64) {
        self.uv
    }
}
