use crate::color::Color;
use crate::math::{dot_product, Hit, Ray, Vec3};
use crate::rendering::Material;
use crate::shapes::Shape;

/// A geometrically perfect sphere.
#[derive(Copy, Clone, Debug)]
pub struct Sphere {
    radius: f64,
    center: Vec3,
    material: Material,
}

impl Shape for Sphere {
    fn intersects(&self, ray: Ray) -> Option<Hit> {
        let sphere_to_ray = ray.origin() - self.center;
        // If the ray origin is outside of the sphere,
        // tangent_squared is the square of the vector from the ray origin to a tangent point on the sphere.
        // If it is inside, tangent_squared will be < 0, but the formulas below still hold.
        let tangent_squared = dot_product(sphere_to_ray, sphere_to_ray) - self.radius.powi(2);
        let projection = dot_product(ray.direction(), sphere_to_ray);
        let discriminant = projection.powi(2) - tangent_squared;

        if discriminant <= 0.0 {
            return None;
        }

        let intersection_distance = if tangent_squared > 0.0 {
            // Ray origin is outside the sphere.
            -projection - discriminant.sqrt()
        } else {
            // Ray origin is inside the sphere.
            -projection + discriminant.sqrt()
        };

        if intersection_distance < 0.0 {
            return None;
        }

        let intersection_point = ray.get_point(intersection_distance);

        let mut normal = (intersection_point - self.center).normalize();

        if dot_product(ray.direction(), normal) > 0.0 {
            normal = -normal
        }

        Some(Hit::new(
            *self,
            intersection_point,
            normal,
            intersection_distance,
        ))
    }
}

impl Sphere {
    /// Create a new sphere.
    pub fn new() -> Self {
        Sphere {
            center: Vec3::new(0.0, 0.0, 0.0),
            radius: 1.0,
            material: Material::Diffuse {
                color: Color::new(0.8, 0.8, 0.8),
            },
        }
    }

    /// Set the center of the sphere and return self (builder pattern).
    pub fn with_center(mut self, center: Vec3) -> Self {
        self.center = center;
        self
    }
    /// Return the center of the sphere.
    pub fn center(&self) -> Vec3 {
        self.center
    }
    /// Set the center of the sphere.
    pub fn set_center(&mut self, center: Vec3) {
        self.center = center;
    }

    /// Set the radius of the sphere and return self (builder pattern).
    pub fn with_radius(mut self, radius: f64) -> Self {
        self.radius = radius;
        self
    }
    /// Return the radius of the sphere.
    pub fn radius(&self) -> f64 {
        self.radius
    }
    /// Set the radius of the sphere.
    pub fn set_radius(&mut self, radius: f64) {
        self.radius = radius;
    }

    /// Set the material of the sphere and return self (builder pattern).
    pub fn with_material(mut self, material: Material) -> Self {
        self.material = material;
        self
    }
    /// Return the material of the sphere.
    pub fn material(&self) -> Material {
        self.material
    }
    /// Set the material of the sphere.
    pub fn set_material(&mut self, material: Material) {
        self.material = material;
    }
}
