use crate::color::Color;
use crate::rendering::{BackgroundMaterial, Camera};
use crate::shapes::Sphere;

/// The scene, consisting of camera, objects and background.
#[derive(Clone, Debug)]
pub struct Scene {
    camera: Camera,
    objects: Vec<Sphere>,
    background: BackgroundMaterial,
}

impl Scene {
    /// Create a new scene.
    pub fn new() -> Self {
        Scene {
            camera: Camera::new(),
            objects: vec![],
            background: BackgroundMaterial::Color {
                color: Color::new(0.8, 0.8, 0.8),
            },
        }
    }

    /// Set the camera of the scene and return self (builder pattern).
    pub fn with_camera(mut self, camera: Camera) -> Self {
        self.camera = camera;
        self
    }
    /// Return the camera of the scene.
    pub fn camera(&self) -> Camera {
        self.camera
    }
    /// Set the camera of the scene.
    pub fn set_camera(&mut self, camera: Camera) {
        self.camera = camera
    }

    /// Set the objects of the scene and return self (builder pattern).
    pub fn with_objects(mut self, objects: Vec<Sphere>) -> Self {
        self.objects = objects;
        self
    }
    /// Return the objects of the scene.
    pub fn objects(&self) -> &Vec<Sphere> {
        &self.objects
    }
    /// Set the objects of the scene.
    pub fn set_objects(&mut self, objects: Vec<Sphere>) {
        self.objects = objects;
    }

    /// Set the background of the scene and return self (builder pattern).
    pub fn with_background(mut self, background: BackgroundMaterial) -> Self {
        self.background = background;
        self
    }
    /// Return the background of the scene.
    pub fn background(&self) -> &BackgroundMaterial {
        &self.background
    }
    /// Set the background of the scene.
    pub fn set_background(&mut self, background: BackgroundMaterial) {
        self.background = background;
    }
}
