use std::f64::consts::PI;
use std::sync::{Arc, Mutex};
use std::time::Instant;

use indicatif::{ProgressBar, ProgressStyle};
use random_fast_rng::{local_rng, Random};
use rayon::prelude::*;

use crate::color::{Color, Image};
use crate::math::{Hit, Ray, Vec3};
use crate::rendering::{BackgroundMaterial, Material, Scene};
use crate::shapes::Shape;

/// The render engine.
pub struct RenderEngine {
    samples: u32,
    max_bounces: usize,
}

impl RenderEngine {
    pub fn new(samples: u32, max_bounces: usize) -> Self {
        RenderEngine {
            samples,
            max_bounces,
        }
    }

    /// Return the samples of the render engine.
    pub fn samples(&self) -> u32 {
        self.samples
    }
    /// Set the samples of the render engine.
    pub fn set_samples(&mut self, samples: u32) {
        self.samples = samples;
    }

    /// Return the maximum amount of bounces of the render engine.
    pub fn max_bounces(&self) -> usize {
        self.max_bounces
    }
    /// Set the maximum amount of bounces of the render engine.
    pub fn set_max_bounces(&mut self, max_bounces: usize) {
        self.max_bounces = max_bounces;
    }
}

impl RenderEngine {
    /// Return the closest hit of a ray with objects of the scene, if any.
    fn get_closest_hit(&self, scene: &Scene, ray: Ray) -> Option<Hit> {
        let mut closest_hit: Option<Hit> = None;
        for object in scene.objects() {
            match object.intersects(ray) {
                None => {}
                Some(hit) => match closest_hit {
                    None => {
                        closest_hit = Some(hit);
                    }
                    Some(old_closest_hit) => {
                        if old_closest_hit.distance() > hit.distance() {
                            closest_hit = Some(hit);
                        }
                    }
                },
            }
        }

        closest_hit
    }

    /// Trace a ray and return the resulting color.
    fn trace_tray(&self, scene: &Scene, ray: Ray, bounces: usize) -> Color {
        if bounces > self.max_bounces() {
            return Color::default();
        }

        let closest_hit = self.get_closest_hit(&scene, ray);

        return match closest_hit {
            None => match scene.background() {
                BackgroundMaterial::Color { color } => *color,
                BackgroundMaterial::Texture { image, intensity } => {
                    let point = ray.direction().normalize();
                    let u = 0.5 + point.z().atan2(point.x()) / (2.0 * PI);
                    let v = 0.5 - point.y().asin() / PI;
                    let pixel =
                        image.get_pixel(image.width() as f64 * u, image.height() as f64 * v);

                    let color = Color::new(pixel.r(), pixel.g(), pixel.b());

                    color * *intensity
                }
            },
            Some(hit) => match hit.object().material() {
                Material::Diffuse { color } => {
                    let direction = hit.normal() + Vec3::random_unit_vector();
                    let ray = Ray::new(hit.position(), direction);
                    self.trace_tray(&scene, ray, bounces + 1) * color
                }
                Material::Metallic { color, roughness } => {
                    let direction = ray.direction().reflect(hit.normal());
                    let ray = Ray::new(
                        hit.position(),
                        direction + roughness * Vec3::random_in_unit_sphere(),
                    );
                    self.trace_tray(&scene, ray, bounces + 1) * color
                }
                Material::Emissive { color, intensity } => color * intensity,
                Material::Refractive {
                    color,
                    ior,
                    roughness,
                } => {
                    let object = hit.object();
                    let (n1, n2) = if (ray.origin() - object.center()).abs() > object.radius() {
                        (1.0, ior)
                    } else {
                        (ior, 1.0)
                    };

                    let direction = ray.direction().refract(hit.normal(), n1, n2);
                    let ray = Ray::new(
                        hit.position(),
                        direction + roughness * Vec3::random_in_unit_sphere(),
                    );
                    self.trace_tray(&scene, ray, bounces + 1) * color
                }
            },
        };
    }

    /// Render the give scene.
    pub fn render(&self, scene: &Scene) -> Image {
        let width = scene.camera().width();
        let height = scene.camera().height();

        let mut image = Image::new(width, height);

        println!(
            "Rendering {} objects with {} samples ({} bounces) to a {}x{} image.",
            scene.objects().len(),
            self.samples(),
            self.max_bounces(),
            width,
            height
        );

        let progress_bar = ProgressBar::new((height) as u64);
        progress_bar.set_style(
            ProgressStyle::default_bar()
                .template(
                    "[{elapsed_precise}] [{bar:60}] {percent:>5}% (ETA: ~{eta_precise}) {msg}",
                )
                .progress_chars("=> "),
        );
        progress_bar.set_position(0);
        let progress_bar = Arc::new(Mutex::new(progress_bar));

        let start_time = Instant::now();

        image
            .pixels_mut()
            .into_par_iter()
            .enumerate()
            .for_each(|(y, row)| {
                row.into_iter().enumerate().for_each(|(x, pixel)| {
                    let mut rng = local_rng();
                    let mut color = Color::default();
                    for _ in 0..self.samples() {
                        let u = (x as f64 + rng.gen::<f64>()) / (width as f64 - 1.0);
                        let v =
                            (height as f64 - (y as f64) + rng.gen::<f64>()) / (height as f64 - 1.0);

                        let ray = scene.camera().get_ray(u, v);

                        color += self.trace_tray(&scene, ray, 0);
                    }
                    *pixel = color / self.samples() as f64;
                    *pixel = Color::new(
                        pixel.r().min(1.0).max(0.0),
                        pixel.g().min(1.0).max(0.0),
                        pixel.b().min(1.0).max(0.0),
                    );
                });

                progress_bar.lock().unwrap().inc(1);
            });

        progress_bar.lock().unwrap().finish_with_message(format!(
            "\nRendering finished after {:.2?}",
            Instant::now() - start_time
        ));

        image
    }
}
