use std::f64::consts::PI;

use crate::math::{cross_product, Ray, Vec3};

/// A geometric representation of a camera including aperture and focus distance.
#[derive(Copy, Clone, Debug)]
pub struct Camera {
    position: Vec3,
    target: Vec3,
    view_up: Vec3,
    fov: f64,
    width: usize,
    height: usize,
    aperture: f64,
    focus_distance: f64,
}

impl Camera {
    pub fn new() -> Self {
        Camera {
            position: Vec3::new(0.0, 0.0, -5.0),
            target: Vec3::new(0.0, 0.0, 0.0),
            view_up: Vec3::new(0.0, 1.0, 0.0),
            fov: PI / 4.0,
            width: 1920,
            height: 1080,
            aperture: 0.0,
            focus_distance: 1.0,
        }
    }

    /// Set the position of the camera and return self (builder pattern).
    pub fn with_position(mut self, position: Vec3) -> Self {
        self.position = position;
        self
    }
    /// Return the position of the camera.
    pub fn position(&self) -> Vec3 {
        self.position
    }
    /// Set the position of the camera.
    pub fn set_position(&mut self, position: Vec3) {
        self.position = position;
    }

    /// Set the target of the camera and return self (builder pattern).
    pub fn with_target(mut self, target: Vec3) -> Self {
        self.target = target;
        self
    }
    /// Return the target point of the camera.
    /// Affects the rotation of the camera.
    pub fn target(&self) -> Vec3 {
        self.target
    }
    /// Set the target point of the camera.
    pub fn set_target(&mut self, target: Vec3) {
        self.target = target;
    }

    /// Return the up vector of the camera.
    pub fn view_up(&self) -> Vec3 {
        self.view_up
    }

    /// Set the field of view of the camera and return self (builder pattern).
    pub fn with_fov(mut self, fov: f64) -> Self {
        self.fov = fov;
        self
    }
    /// Return the field of view of the camera.
    pub fn fov(&self) -> f64 {
        self.fov
    }
    /// Set the field of view of the camera.
    pub fn set_fov(&mut self, fov: f64) {
        self.fov = fov;
    }

    /// Set the width of the camera and return self (builder pattern).
    pub fn with_width(mut self, width: usize) -> Self {
        self.width = width;
        self
    }
    /// Return the width of the camera image.
    pub fn width(&self) -> usize {
        self.width
    }
    /// Set the width of the camera.
    pub fn set_width(&mut self, width: usize) {
        self.width = width;
    }

    /// Set the height of the camera and return self (builder pattern).
    pub fn with_height(mut self, height: usize) -> Self {
        self.height = height;
        self
    }
    /// Return the height of the camera image.
    pub fn height(&self) -> usize {
        self.height
    }
    /// Set the height of the camera.
    pub fn set_height(&mut self, height: usize) {
        self.height = height;
    }

    /// Set the aperture of the camera and return self (builder pattern).
    pub fn with_aperture(mut self, aperture: f64) -> Self {
        self.aperture = aperture;
        self
    }
    /// Return the aperture of the camera.
    pub fn aperture(&self) -> f64 {
        self.aperture
    }
    /// Set the aperture of the camera.
    pub fn set_aperture(&mut self, aperture: f64) {
        self.aperture = aperture;
    }

    /// Set the focus distance of the camera and return self (builder pattern).
    pub fn with_focus_distance(mut self, focus_distance: f64) -> Self {
        self.focus_distance = focus_distance;
        self
    }
    /// Return the focus distance of the camera.
    pub fn focus_distance(&self) -> f64 {
        self.focus_distance
    }
    /// Set the focus distance of the camera.
    pub fn set_focus_distance(&mut self, focus_distance: f64) {
        self.focus_distance = focus_distance;
    }

    /// Return the ray from the camera's origin in the direction of the pixel `(x, y)`.
    pub fn get_ray(&self, x: f64, y: f64) -> Ray {
        let viewport_height = 2.0 * (self.fov / 2.0).tan();
        let viewport_width = (self.width as f64 / self.height as f64) * viewport_height;

        let w = (self.position - self.target).normalize();
        let u = cross_product(self.view_up, w).normalize();
        let v = cross_product(w, u);

        let horizontal = self.focus_distance * viewport_width * u;
        let vertical = self.focus_distance * viewport_height * v;
        let lower_left_corner =
            self.position - horizontal / 2.0 - vertical / 2.0 - self.focus_distance * w;

        let lens_radius = self.aperture / 2.0;
        let random_point = lens_radius * Vec3::random_in_unit_disk();
        let offset = u * random_point.x() + v * random_point.y();

        Ray::new(
            self.position + offset,
            lower_left_corner + x * horizontal + y * vertical - self.position - offset,
        )
    }
}
