use std::fs::File;
use std::io::BufReader;
use std::path::Path;

use image::codecs::hdr::HdrDecoder;
use image::{GenericImageView, ImageBuffer, ImageResult, Rgb};

use crate::color::Color;

/// An image of fixed size represented as a 2-dimensional `Vec` of `Color`.
#[derive(Clone, Debug)]
pub struct Image {
    width: usize,
    height: usize,
    pixels: Vec<Vec<Color>>,
}

impl Image {
    /// Create a new image.
    pub fn new(width: usize, height: usize) -> Self {
        Image {
            width,
            height,
            pixels: vec![vec![Color::new(f64::NAN, f64::NAN, f64::NAN); width]; height],
        }
    }

    /// Load an image from the file at the provided path.
    /// This *can* be used to load HDRIs, however good results are most likely using [from_hdr_file](Self::from_hdr_file).
    pub fn from_file(path: &Path) -> ImageResult<Image> {
        let image = image::open(path)?;
        let mut pixels = vec![
            vec![Color::new(f64::NAN, f64::NAN, f64::NAN); image.width() as usize];
            image.height() as usize
        ];
        pixels.iter_mut().enumerate().for_each(|(y, row)| {
            row.iter_mut().enumerate().for_each(|(x, color)| {
                let pixel = image.get_pixel(x as u32, y as u32);
                *color = Color::new(
                    pixel.0[0] as f64 / 255.0,
                    pixel.0[1] as f64 / 255.0,
                    pixel.0[2] as f64 / 255.0,
                );
            });
        });

        Ok(Image {
            width: image.width() as usize,
            height: image.height() as usize,
            pixels,
        })
    }

    /// Load an HDRI from the file at the provided path.
    /// Conversion into RGB space is according to the formula
    /// v' = (v * scale)<sup>gamma</sup>
    pub fn from_hdr_file(path: &Path, scale: f32, gamma: f32) -> ImageResult<Image> {
        let decoder = HdrDecoder::new(BufReader::new(
            File::open(path).expect("Error opening HDRI"),
        ))?;

        let width = decoder.metadata().width as usize;
        let height = decoder.metadata().height as usize;

        let image = decoder.read_image_native()?;

        let mut pixels = vec![vec![Color::new(f64::NAN, f64::NAN, f64::NAN); width]; height];
        pixels.iter_mut().enumerate().for_each(|(y, row)| {
            row.iter_mut().enumerate().for_each(|(x, color)| {
                let pixel: Rgb<u8> = image[x + y * width].to_ldr_scale_gamma(scale, gamma);
                *color = Color::new(
                    pixel.0[0] as f64 / 255.0,
                    pixel.0[1] as f64 / 255.0,
                    pixel.0[2] as f64 / 255.0,
                );
            });
        });

        Ok(Image {
            width: width as usize,
            height: height as usize,
            pixels,
        })
    }

    /// Return the width of the image.
    pub fn width(&self) -> usize {
        self.width
    }

    /// Return the height of the image.
    pub fn height(&self) -> usize {
        self.height
    }

    /// Return a reference to the grid of pixels.
    pub fn pixels(&self) -> &Vec<Vec<Color>> {
        &self.pixels
    }
    /// Return a mutable reference to the grid of pixels.
    pub fn pixels_mut(&mut self) -> &mut Vec<Vec<Color>> {
        &mut self.pixels
    }

    /// Return the color value of the pixel `(x, y)`.
    /// The image is considered to wrap periodically and is thus virtually infinite.
    pub fn get_pixel(&self, x: f64, y: f64) -> Color {
        self.bilinear_interpolate(x, y)
    }

    /// Return the bilinearly interpolated color value at the position `(x, y)`.
    fn bilinear_interpolate(&self, x: f64, y: f64) -> Color {
        let x1 = x.floor();
        let x2 = x.ceil() % self.width as f64;
        let y1 = y.floor();
        let y2 = y.ceil() % self.height as f64;

        let weight1 = (x2 - x) / (x2 - x1);
        let weight2 = (x - x1) / (x2 - x1);

        let color1 = self.pixels[y1 as usize][x1 as usize] * weight1
            + self.pixels[y1 as usize][x2 as usize] * weight2;
        let color2 = self.pixels[y2 as usize][x1 as usize] * weight1
            + self.pixels[y2 as usize][x2 as usize] * weight2;
        (color1 * (y2 - y) + color2 * (y - y1)) / (y2 - y1)
    }

    /// Set the color value of the pixel `(x, y)`.
    /// Return an error if the pixel is out of bounds.
    pub fn set_pixel(&mut self, x: usize, y: usize, color: Color) -> Result<(), String> {
        if y + 1 > self.height || x + 1 > self.width {
            return Err(format!(
                "Coordinates ({}, {}) are out of image bounds ({}, {})",
                x, y, self.width, self.height
            ));
        }

        self.pixels[y][x] = color;
        Ok(())
    }

    /// Write the image to a file at the provided path.
    pub fn write(&self, path: &Path) -> std::io::Result<()> {
        let mut image_buffer = ImageBuffer::new(self.width as u32, self.height as u32);
        image_buffer
            .enumerate_pixels_mut()
            .for_each(|(x, y, pixel)| {
                let color = self.pixels[(y as usize) as usize][x as usize];
                *pixel = Rgb([
                    ((color.r() * 255.0).round() as u8).min(255).max(0),
                    ((color.g() * 255.0).round() as u8).min(255).max(0),
                    ((color.b() * 255.0).round() as u8).min(255).max(0),
                ])
            });

        image_buffer.save(path).unwrap();

        println!("Saved image to {:?}", path);

        Ok(())
    }

    /// Denoise the image using the OIDN library.
    #[cfg_attr(doc, doc(cfg(feature = "oidn")))]
    #[cfg(feature = "oidn")]
    pub fn denoise(&mut self) -> Self {
        use oidn::{Device, RayTracing};
        use std::time::Instant;

        let width = self.width;
        let height = self.height;

        let mut input_img = vec![0.0f32; 3 * width * height];

        for y in 0..height {
            for x in 0..width {
                let pixel = self.pixels()[y][x];
                for c in 0..3 {
                    input_img[3 * (y * width + x) + c] = match c {
                        0 => pixel.r() as f32,
                        1 => pixel.g() as f32,
                        2 => pixel.b() as f32,
                        _ => 0.0,
                    };
                }
            }
        }

        let mut filter_output = vec![0.0f32; input_img.len()];

        println!("Denoising...");

        let start_time = Instant::now();

        let device = Device::new();
        let mut filter = RayTracing::new(&device);
        filter.srgb(true).image_dimensions(width, height);
        filter
            .filter(&input_img[..], &mut filter_output[..])
            .expect("Error denoising image");

        if let Err(e) = device.get_error() {
            println!("Error denoising image: {}", e.1);
        }

        println!(
            "Denoising finished after {:.2?}",
            Instant::now() - start_time
        );

        for i in (0..filter_output.len()).step_by(3) {
            let pixel = i / 3;

            let x = pixel % width;
            let y = pixel / width;

            let color = Color::new(
                filter_output[i] as f64,
                filter_output[i + 1] as f64,
                filter_output[i + 2] as f64,
            );

            self.set_pixel(x, y, color).unwrap();
        }

        self.clone()
    }
}
