use std::ops::{Add, AddAssign, Div, DivAssign, Mul, MulAssign, Sub, SubAssign};

use crate::math::Vec3;

/// An RGB color represented by 3 float values.
/// `(0, 0, 0)` is black and `(1, 1, 1)` is white.
/// This struct supports elementwise arithmetic operations (+, -, *, /).
#[derive(Copy, Clone, Default, Debug)]
pub struct Color {
    vec3: Vec3,
}

impl Add<Color> for Color {
    type Output = Color;

    fn add(self, other: Color) -> Color {
        Color {
            vec3: self.vec3 + other.vec3,
        }
    }
}

impl AddAssign<Color> for Color {
    fn add_assign(&mut self, other: Color) {
        *self = Color {
            vec3: self.vec3 + other.vec3,
        }
    }
}

impl Sub<Color> for Color {
    type Output = Color;

    fn sub(self, other: Color) -> Color {
        Color {
            vec3: self.vec3 - other.vec3,
        }
    }
}

impl SubAssign<Color> for Color {
    fn sub_assign(&mut self, other: Color) {
        *self = Color {
            vec3: self.vec3 - other.vec3,
        }
    }
}

impl Mul<Color> for Color {
    type Output = Color;

    fn mul(self, other: Color) -> Color {
        Color {
            vec3: Vec3::new(
                self.vec3.x() * other.vec3.x(),
                self.vec3.y() * other.vec3.y(),
                self.vec3.z() * other.vec3.z(),
            ),
        }
    }
}

impl MulAssign<Color> for Color {
    fn mul_assign(&mut self, other: Color) {
        *self = *self * other
    }
}

impl Mul<f64> for Color {
    type Output = Color;

    fn mul(self, other: f64) -> Color {
        Color {
            vec3: self.vec3 * other,
        }
    }
}

impl MulAssign<f64> for Color {
    fn mul_assign(&mut self, other: f64) {
        *self = *self * other;
    }
}

impl Mul<Color> for f64 {
    type Output = Color;

    fn mul(self, other: Color) -> Color {
        other * self
    }
}

impl Div<Color> for Color {
    type Output = Color;

    fn div(self, other: Color) -> Color {
        Color {
            vec3: Vec3::new(
                self.vec3.x() / other.vec3.x(),
                self.vec3.y() / other.vec3.y(),
                self.vec3.z() / other.vec3.z(),
            ),
        }
    }
}

impl DivAssign<Color> for Color {
    fn div_assign(&mut self, other: Color) {
        *self = *self / other
    }
}

impl Div<f64> for Color {
    type Output = Color;

    fn div(self, other: f64) -> Color {
        Color {
            vec3: self.vec3 / other,
        }
    }
}

impl DivAssign<f64> for Color {
    fn div_assign(&mut self, other: f64) {
        *self = *self / other;
    }
}

impl Color {
    /// Create a new Color from float values representing RGB, where `(1, 1, 1)` is white.
    pub fn new(r: f64, g: f64, b: f64) -> Self {
        Color {
            vec3: Vec3::new(r, g, b),
        }
    }

    /// Return the value of the red channel.
    pub fn r(&self) -> f64 {
        self.vec3.x()
    }

    /// Return the value of the green channel.
    pub fn g(&self) -> f64 {
        self.vec3.y()
    }

    /// Return the value of the blue channel.
    pub fn b(&self) -> f64 {
        self.vec3.z()
    }

    /// Required for testing: naive difference between colors.
    fn abs(&self) -> f64 {
        self.r().abs() + self.g().abs() + self.b().abs()
    }
}

#[cfg(test)]
mod tests {
    use assert_approx_eq::assert_approx_eq;

    use super::Color;

    const EPSILON: f64 = f64::EPSILON * 100.0;

    #[test]
    fn color_add() {
        assert_approx_eq!(
            Color::new(2.0, 1.0, 0.0) + Color::new(1.0, 1.0, 1.0),
            Color::new(3.0, 2.0, 1.0),
            EPSILON
        );
        assert_approx_eq!(
            Color::new(5.72, 2.5, 8.824) + Color::new(8.7, 5.987, 0.12),
            Color::new(14.42, 8.487, 8.944),
            EPSILON
        );
        let mut color_a = Color::new(7.0, 2.5, 3.2);
        color_a += Color::new(1.2, 9.23, 6.2);
        assert_approx_eq!(color_a, Color::new(8.2, 11.73, 9.4), EPSILON)
    }

    #[test]
    fn color_sub() {
        assert_approx_eq!(
            Color::new(2.0, 1.0, 0.0) - Color::new(1.0, 1.0, 1.0),
            Color::new(1.0, 0.0, -1.0),
            EPSILON
        );
        assert_approx_eq!(
            Color::new(5.72, 2.5, 8.824) - Color::new(8.7, 5.987, 0.12),
            Color::new(-2.98, -3.487, 8.704),
            EPSILON
        );
        let mut color_a = Color::new(7.0, 2.5, 3.2);
        color_a -= Color::new(1.2, 9.23, 6.2);
        assert_approx_eq!(color_a, Color::new(5.8, -6.73, -3.0), EPSILON)
    }

    #[test]
    fn color_mul() {
        assert_approx_eq!(
            Color::new(2.0, 1.0, 0.0) * 2.0,
            Color::new(4.0, 2.0, 0.0),
            EPSILON
        );
        assert_approx_eq!(
            2.5 * Color::new(8.7, 5.987, 0.12),
            Color::new(21.75, 14.9675, 0.3),
            EPSILON
        );
        let mut color_a = Color::new(7.0, 2.5, 3.2);
        color_a *= -2.0;
        assert_approx_eq!(color_a, Color::new(-14.0, -5.0, -6.4), EPSILON);
    }

    #[test]
    fn color_div() {
        assert_approx_eq!(
            Color::new(2.0, 1.0, 0.0) / 2.0,
            Color::new(1.0, 0.5, 0.0),
            EPSILON
        );
        assert_approx_eq!(
            Color::new(8.7, 5.987, 0.12) / 2.5,
            Color::new(3.48, 2.3948, 0.048),
            EPSILON
        );
        let mut color_a = Color::new(7.0, 2.5, 3.2);
        color_a /= -2.0;
        assert_approx_eq!(color_a, Color::new(-3.5, -1.25, -1.6), EPSILON);
    }
}
