use crate::WSClient;
use std::collections::HashMap;
use std::sync::mpsc::Sender;

use super::{
    utils::CHANNEL_PAIR_DELIMITER,
    ws_client_internal::{MiscMessage, WSClientInternal},
    Candlestick, Level3OrderBook, OrderBook, OrderBookTopK, Ticker, Trade, BBO,
};

use log::*;
use serde_json::Value;

pub(super) const EXCHANGE_NAME: &str = "bithumb";

const WEBSOCKET_URL: &str = "wss://global-api.bithumb.pro/message/realtime";

const CLIENT_PING_INTERVAL_AND_MSG: (u64, &str) = (60, r#"{"cmd":"ping"}"#);

/// The WebSocket client for Bithumb.
///
/// Bithumb has only Spot market.
///
///   * WebSocket API doc: <https://github.com/bithumb-pro/bithumb.pro-official-api-docs/blob/master/ws-api.md>
///   * Trading at: <https://en.bithumb.com/trade/order/BTC_KRW>
pub struct BithumbWSClient {
    client: WSClientInternal,
}

fn channels_to_commands(channels: &[String], subscribe: bool) -> Vec<String> {
    let mut all_commands: Vec<String> = channels
        .iter()
        .filter(|ch| ch.starts_with('{'))
        .map(|s| s.to_string())
        .collect();

    let raw_channels: Vec<&String> = channels.iter().filter(|ch| !ch.starts_with('{')).collect();
    if !raw_channels.is_empty() {
        all_commands.append(&mut vec![format!(
            r#"{{"cmd":"{}","args":{}}}"#,
            if subscribe {
                "subscribe"
            } else {
                "unsubscribe"
            },
            serde_json::to_string(&raw_channels).unwrap()
        )])
    };

    all_commands
}

fn on_misc_msg(msg: &str) -> MiscMessage {
    let obj = serde_json::from_str::<HashMap<String, Value>>(msg).unwrap();
    let code = obj.get("code").unwrap().as_str().unwrap();
    let code = code.parse::<i64>().unwrap();
    if code < 10000 {
        match code {
            0 => MiscMessage::Pong,
            6 => {
                let arr = obj.get("data").unwrap().as_array();
                if arr != None && arr.unwrap().is_empty() {
                    // ignore empty data
                    MiscMessage::Misc
                } else {
                    MiscMessage::Normal
                }
            }
            7 => MiscMessage::Normal,
            _ => {
                debug!("Received {} from {}", msg, EXCHANGE_NAME);
                MiscMessage::Misc
            }
        }
    } else {
        error!("Received {} from {}", msg, EXCHANGE_NAME);
        panic!("Received {} from {}", msg, EXCHANGE_NAME);
    }
}

fn to_raw_channel(channel: &str, pair: &str) -> String {
    format!("{}{}{}", channel, CHANNEL_PAIR_DELIMITER, pair)
}

#[rustfmt::skip]
impl_trait!(Trade, BithumbWSClient, subscribe_trade, "TRADE", to_raw_channel);
#[rustfmt::skip]
impl_trait!(Ticker, BithumbWSClient, subscribe_ticker, "TICKER", to_raw_channel);
#[rustfmt::skip]
impl_trait!(OrderBook, BithumbWSClient, subscribe_orderbook, "ORDERBOOK", to_raw_channel);

impl BBO for BithumbWSClient {
    fn subscribe_bbo(&self, _pairs: &[String]) {
        panic!("bithumb WebSocket does NOT have BBO channel");
    }
}

impl OrderBookTopK for BithumbWSClient {
    fn subscribe_orderbook_topk(&self, _pairs: &[String]) {
        panic!("bithumb does NOT have orderbook snapshot channel");
    }
}

impl Candlestick for BithumbWSClient {
    fn subscribe_candlestick(&self, _symbol_interval_list: &[(String, usize)]) {
        panic!("bithumb does NOT have candlestick channel");
    }
}

panic_l3_orderbook!(BithumbWSClient);

impl_new_constructor!(
    BithumbWSClient,
    EXCHANGE_NAME,
    WEBSOCKET_URL,
    channels_to_commands,
    on_misc_msg,
    Some(CLIENT_PING_INTERVAL_AND_MSG),
    None
);
impl_ws_client_trait!(BithumbWSClient);
