use crypto_market_type::MarketType;

pub(crate) fn normalize_currency(currency: &str) -> String {
    let uppercase = currency.to_uppercase();
    let mut currency = uppercase.as_str();
    // https://support.kraken.com/hc/en-us/articles/360001185506-How-to-interpret-asset-codes
    if currency.len() > 3 && (currency.starts_with('X') || currency.starts_with('Z')) {
        currency = &currency[1..]
    }

    if currency == "XBT" {
        "BTC"
    } else if currency == "XDG" {
        "DOGE"
    } else {
        currency
    }
    .to_string()
}

pub(crate) fn normalize_pair(symbol: &str) -> Option<String> {
    if symbol.contains('/') {
        // Spot
        let (base, quote) = {
            let v: Vec<&str> = symbol.split('/').collect();
            (v[0].to_string(), v[1].to_string())
        };

        Some(format!(
            "{}/{}",
            normalize_currency(&base),
            normalize_currency(&quote)
        ))
    } else if symbol.starts_with("pi_") || symbol.starts_with("fi_") {
        let pos = symbol.find("usd").unwrap();
        let base = (&symbol[3..pos]).to_uppercase();
        Some(format!("{}/USD", normalize_currency(&base),))
    } else {
        None
    }
}

pub(crate) fn get_market_type(symbol: &str) -> MarketType {
    if symbol.contains('/') {
        MarketType::Spot
    } else if symbol.starts_with("pi_") {
        MarketType::InverseSwap
    } else if symbol.starts_with("fi_") {
        MarketType::InverseFuture
    } else {
        MarketType::Unknown
    }
}
