//! Tif frame reader.
//!
//! Example usage:
//! ```rust
//! use cryiorust::tif::Tif;
//! use cryiorust::frame::Frame;
//! use std::path::Path;
//!
//! fn test_tif<P: AsRef<Path>>(path: P) {
//!     let testfile = path;
//!     let frame = Tif::read_file(testfile).unwrap();
//!     assert_eq!(frame.dim1(), 1475, "dim1 != 1475");
//!     assert_eq!(frame.dim2(), 1679, "dim2 != 1679");
//!     assert_eq!(frame.sum(), 820029., "frame sum != 820029");
//! }
//! ```
//!
use crate::frame::{Array, Frame, FrameError, Header};
use byteorder::NativeEndian;
use cryiorust_derive::Frame;
use reinterpret::reinterpret_slice;
use std::path::Path;
use std::{fs, io};
use tiff::decoder::{Decoder, DecodingResult};
use tiff::TiffError;

/// Main exported struct.
#[derive(Frame)]
pub struct Tif {
    header: Header,
    array: Array,
}

impl Tif {
    /// Reads a Tif frame from a file.
    pub fn read_file<P: AsRef<Path>>(path: P) -> Result<Tif, FrameError> {
        let frame = Tif {
            header: Default::default(),
            array: Tif::read(fs::File::open(path)?)?,
        };
        Ok(frame)
    }

    fn read<P: io::Read + io::Seek>(reader: P) -> Result<Array, FrameError> {
        let mut dec = Decoder::new(reader)?;
        let dims = dec.dimensions()?;
        let dim1 = dims.0 as usize;
        let dim2 = dims.1 as usize;
        let array = match dec.read_image()? {
            DecodingResult::I8(v) => {
                Array::from_slice_i8(dim1, dim2, unsafe { reinterpret_slice::<i8, u8>(&v) })?
            }
            DecodingResult::I16(v) => Array::from_slice_i16::<NativeEndian>(dim1, dim2, unsafe {
                reinterpret_slice::<i16, u8>(&v)
            })?,
            DecodingResult::I32(v) => Array::from_slice_i32::<NativeEndian>(dim1, dim2, unsafe {
                reinterpret_slice::<i32, u8>(&v)
            })?,
            DecodingResult::I64(v) => Array::from_slice_i64::<NativeEndian>(dim1, dim2, unsafe {
                reinterpret_slice::<i64, u8>(&v)
            })?,
            DecodingResult::U8(v) => Array::from_slice_u8(dim1, dim2, &v)?,
            DecodingResult::U16(v) => Array::from_slice_u16::<NativeEndian>(dim1, dim2, unsafe {
                reinterpret_slice::<u16, u8>(&v)
            })?,
            DecodingResult::U32(v) => Array::from_slice_u32::<NativeEndian>(dim1, dim2, unsafe {
                reinterpret_slice::<u32, u8>(&v)
            })?,
            DecodingResult::U64(v) => Array::from_slice_u64::<NativeEndian>(dim1, dim2, unsafe {
                reinterpret_slice::<u64, u8>(&v)
            })?,
            DecodingResult::F32(v) => Array::from_slice_f32::<NativeEndian>(dim1, dim2, unsafe {
                reinterpret_slice::<f32, u8>(&v)
            })?,
            DecodingResult::F64(v) => Array::from_slice_f64::<NativeEndian>(dim1, dim2, unsafe {
                reinterpret_slice::<f64, u8>(&v)
            })?,
        };
        Ok(array)
    }
}

impl From<TiffError> for FrameError {
    fn from(error: TiffError) -> Self {
        match error {
            TiffError::FormatError(e) => {
                FrameError::FormatError(format!("tiff format error {}", e).into())
            }
            TiffError::UnsupportedError(e) => {
                FrameError::FormatError(format!("unsupported tiff: error {}", e).into())
            }
            TiffError::IoError(e) => FrameError::IoError(e),
            TiffError::LimitsExceeded => FrameError::FormatError("tiff limits exceeded".into()),
            TiffError::IntSizeError => FrameError::FormatError("tiff endian error".into()),
            TiffError::UsageError(_) => FrameError::FormatError("tiff incompatible".into()),
        }
    }
}

#[cfg(test)]
mod tests {
    extern crate utilities;
    use crate::frame::Frame;
    use crate::tif::Tif;

    #[test]
    fn test_tif() {
        let testfile = utilities::download_test_file(
            "lab6.tif",
            "lab6.tif.tar.bz2",
            "49c454bf346439642324406c2a347a9b",
        )
        .unwrap();
        let frame = Tif::read_file(testfile).unwrap();
        assert_eq!(frame.dim1(), 2880, "dim1 != 2880");
        assert_eq!(frame.dim2(), 2880, "dim2 != 2880");
        assert_eq!(frame.sum(), 18092058285., "frame sum != 18092058285");
        assert_eq!(frame.array[2880 * 5 + 6], 1232., "frame[5, 6] != 1232");
    }

    #[test]
    fn test_tif_i32() {
        let testfile = utilities::download_test_file(
            "tiff_i32.tif",
            "tiff_i32.tar.bz2",
            "e0d7c21f026332d459bbb04aad2847c2",
        )
        .unwrap();
        let frame = Tif::read_file(testfile).unwrap();
        assert_eq!(frame.dim1(), 1475, "dim1 != 1475");
        assert_eq!(frame.dim2(), 1679, "dim2 != 1679");
        assert_eq!(frame.sum(), 820029., "frame sum != 820029");
        assert_eq!(
            frame.array[1475 * 89 + 857],
            50011.,
            "frame[89, 857] != 50011"
        );
    }
}
