//! Mar345 format reader.
//!
//! Only non-spiral compressed frames are supported.
//!
//! Example usage:
//! ```rust
//! use cryiorust::mar::Mar;
//! use cryiorust::frame::Frame;
//! use std::path::Path;
//!
//! fn test_mar_read<P: AsRef<Path>>(path: P) {
//!     let testfile = path;
//!     let frame = Mar::read_file(testfile).unwrap();
//!     assert_eq!(frame.dim1(), 32, "dim1 != 32");
//!     assert_eq!(frame.dim2(), 32, "dim2 != 32");
//!     assert_eq!(frame.sum(), 17347273., "frame sum != 17347273.");
//!     assert_eq!(frame.max(), 10000000., "frame max != 10000000.");
//!     assert_eq!(frame.min(), 9., "frame min != 9.");
//! }
//! ```
use crate::frame::FrameError::{FormatError, IoError};
use crate::frame::{Array, Frame, FrameResult, Header};
use byteorder::{BigEndian, ByteOrder, LittleEndian, ReadBytesExt};
use cryiorust_derive::Frame;
use std::collections::HashMap;
use std::fs::File;
use std::io;
use std::io::Read;
use std::path::Path;

/// Main exported struct.
#[derive(Frame)]
pub struct Mar {
    header: Header,
    array: Array,
    dim1: usize,
    dim2: usize,
    n_pixels: usize,
    n_overflow: usize,
    data_start: usize,
    overflow_start: usize,
    overflow_end: usize,
}

impl Mar {
    /// Creates a new empty Mar struct.
    pub fn new() -> Mar {
        Mar {
            header: HashMap::new(),
            array: Array::new(),
            dim1: 0,
            dim2: 0,
            n_pixels: 0,
            n_overflow: 0,
            data_start: 0,
            overflow_start: HEADER_SIZE,
            overflow_end: 0,
        }
    }

    /// Reads a Mar frame from a file.
    pub fn read_file<P: AsRef<Path>>(path: P) -> FrameResult<Mar> {
        let mut frame = Mar::new();
        frame.read(path)?;
        Ok(frame)
    }

    fn read<P: AsRef<Path>>(&mut self, path: P) -> FrameResult<()> {
        let mut reader = File::open(path)?;
        let size = reader.metadata()?.len() as usize;
        if size < HEADER_SIZE {
            return Err(IoError(io::Error::new(
                io::ErrorKind::UnexpectedEof,
                "file is too short",
            )));
        }
        let mut data: Vec<u8> = Vec::with_capacity(size);
        unsafe { data.set_len(size) };
        reader.read_exact(&mut data)?;
        let header: BinHeader = unsafe { std::ptr::read(data.as_ptr() as *const _) };
        match header.format {
            FORMAT_COMPRESSED => match header.magic_number {
                MAGIC_NUMBER_LE => {
                    self.dim1 = header.dim1.to_le() as usize;
                    self.n_pixels = header.n_pixels.to_le() as usize;
                    self.dim2 = self.n_pixels / self.dim1;
                    self.n_overflow = header.n_overflow.to_le() as usize;
                    self.decompress(&data)?;
                    self.parse_overflow::<LittleEndian>(&data)
                }
                MAGIC_NUMBER_BE => {
                    self.dim1 = header.dim1.to_be() as usize;
                    self.n_pixels = header.n_pixels.to_be() as usize;
                    self.dim2 = self.n_pixels / self.dim1;
                    self.n_overflow = header.n_overflow.to_be() as usize;
                    self.decompress(&data)?;
                    self.parse_overflow::<BigEndian>(&data)
                }
                _ => {
                    return Err(FormatError(
                        "could not parse magic number in the header".into(),
                    ));
                }
            },
            FORMAT_SPIRAL => return Err(FormatError("spiral format is not supported".into())),
            _ => {
                return Err(FormatError(
                    format!("file format is not recognized: {}", header.format).into(),
                ));
            }
        }
    }

    fn decompress(&mut self, data: &[u8]) -> FrameResult<()> {
        let magic_string = format!(
            "CCP4 packed image, X: {:04}, Y: {:04}\n",
            self.dim1, self.dim2
        );
        let magic_bytes = magic_string.as_bytes();
        for i in HEADER_SIZE..data.len() {
            if data[i] != magic_bytes[0] {
                continue;
            }
            if data.len() < i + magic_bytes.len() {
                return Err(IoError(io::Error::new(
                    io::ErrorKind::UnexpectedEof,
                    "not enough data in file",
                )));
            }
            let mut found = true;
            for (c1, c2) in data[i..i + magic_bytes.len()].iter().zip(magic_bytes) {
                if *c1 != *c2 {
                    found = false;
                    break;
                }
            }
            if found {
                self.overflow_end = i;
                self.data_start = i + magic_bytes.len();
                break;
            }
        }
        if self.data_start == 0 {
            return Err(FormatError(
                format!("could not find magic string: {}", magic_string).into(),
            ));
        }
        if self.overflow_end - self.overflow_start < self.n_overflow * 2 {
            return Err(FormatError("bad value for overflow".into()));
        }
        self.unpack(&data[self.data_start..])?;
        Ok(())
    }

    fn unpack(&mut self, data: &[u8]) -> FrameResult<()> {
        let mut unpacked: Vec<f64> = Vec::with_capacity(self.n_pixels);
        let mut array: Vec<u32> = Vec::with_capacity(self.n_pixels);
        unsafe {
            array.set_len(self.n_pixels);
            unpacked.set_len(self.n_pixels);
        }
        let mut i = 0;
        let mut j = 0;
        let mut num_error: u32 = 0;
        let mut bit_offset: u32 = 0;
        let mut num_bits: u32 = 0;
        let mut t1 = data[j];
        j += 1;
        while i < self.n_pixels {
            if num_error == 0 {
                if bit_offset >= 8 - CCP4_PCK_BLOCK_HEADER_LENGTH {
                    if j > data.len() {
                        return Err(IoError(io::Error::new(
                            io::ErrorKind::UnexpectedEof,
                            "not enough data in file",
                        )));
                    }
                    let t2 = data[j];
                    j += 1;
                    t1 = (t1 >> bit_offset as u8) + (t2 << (8 - bit_offset) as u8);
                    num_error = CCP4_PCK_ERR_COUNT[(t1 & CCP4_PCK_MASK[3]) as usize];
                    num_bits = CCP4_PCK_BIT_COUNT[((t1 >> 3) & CCP4_PCK_MASK[3]) as usize];
                    bit_offset = CCP4_PCK_BLOCK_HEADER_LENGTH + bit_offset - 8;
                    t1 = t2;
                } else {
                    num_error =
                        CCP4_PCK_ERR_COUNT[((t1 >> bit_offset as u8) & CCP4_PCK_MASK[3]) as usize];
                    num_bits = CCP4_PCK_BIT_COUNT
                        [((t1 >> (3 + bit_offset as u8)) & CCP4_PCK_MASK[3]) as usize];
                    bit_offset += CCP4_PCK_BLOCK_HEADER_LENGTH;
                }
            } else {
                while num_error > 0 {
                    let mut err_val: i32 = 0;
                    let mut read_bits: u32 = 0;
                    while read_bits < num_bits {
                        if bit_offset + (num_bits - read_bits) >= 8 {
                            let conv =
                                (t1 >> bit_offset as u8) & CCP4_PCK_MASK[(8 - bit_offset) as usize];
                            err_val |= ((conv as u32) << read_bits) as i32;
                            read_bits += 8 - bit_offset;
                            bit_offset = 0;
                            if j > data.len() {
                                return Err(IoError(io::Error::new(
                                    io::ErrorKind::UnexpectedEof,
                                    "not enough data in file",
                                )));
                            }
                            t1 = data[j];
                            j += 1;
                        } else {
                            let conv = (t1 >> bit_offset as u8)
                                & CCP4_PCK_MASK[(num_bits - read_bits) as usize];
                            err_val |= ((conv as i32) << (read_bits as i32)) as i32;
                            bit_offset += num_bits - read_bits;
                            read_bits = num_bits;
                        }
                    }
                    if err_val & 1i32.wrapping_shl(num_bits.wrapping_sub(1)) != 0 {
                        err_val |= -1i32.wrapping_shl(num_bits.wrapping_sub(1));
                    }
                    array[i] = if i > self.dim1 {
                        let x4 = (array[i - 1] as i16) as i32;
                        let x3 = (array[i - self.dim1 + 1] as i16) as i32;
                        let x2 = (array[i - self.dim1] as i16) as i32;
                        let x1 = (array[i - self.dim1 - 1] as i16) as i32;
                        ((err_val + (x4 + x3 + x2 + x1 + 2) / 4) as u16) as u32
                    } else if i != 0 {
                        ((err_val + array[i - 1] as i32) as u16) as u32
                    } else {
                        (err_val as u16) as u32
                    };
                    i += 1;
                    num_error -= 1;
                }
            }
        }
        for i in 0..self.n_pixels {
            unpacked[i] = array[i] as f64;
        }
        self.array = Array::with_data(self.dim1 as usize, self.dim2 as usize, unpacked);
        Ok(())
    }

    /// overflown pixels are stored as two u32 numbers for address and value
    /// {address value address value address value ...}
    /// and the size of u32 is four bytes
    fn parse_overflow<F: ByteOrder>(&mut self, data: &[u8]) -> FrameResult<()> {
        let mut cursor = io::Cursor::new(&data[self.overflow_start..self.overflow_end]);
        while self.n_overflow > 0 {
            let address = cursor.read_u32::<F>()? as usize;
            if address > 0 && address <= self.n_pixels {
                self.array[address - 1] = cursor.read_u32::<F>()? as f64;
            }
            self.n_overflow -= 1;
        }
        Ok(())
    }
}

#[repr(C)]
struct BinHeader {
    magic_number: i32,
    dim1: i32,
    n_overflow: i32,
    format: i32,
    mode: i32,
    n_pixels: i32,
    pixel_length: i32,
    pixel_height: i32,
    wavelength: i32,
    distance: i32,
    start_phi: i32,
    end_phi: i32,
    start_omega: i32,
    end_omega: i32,
    chi: i32,
    two_theta: i32,
}

const MAGIC_NUMBER: i32 = 1234;
const MAGIC_NUMBER_LE: i32 = MAGIC_NUMBER.to_le();
const MAGIC_NUMBER_BE: i32 = MAGIC_NUMBER.to_be();
const FORMAT_COMPRESSED: i32 = 1;
const FORMAT_SPIRAL: i32 = 2;
const HEADER_SIZE: usize = 4096;
const CCP4_PCK_BLOCK_HEADER_LENGTH: u32 = 6;
const CCP4_PCK_ERR_COUNT: [u32; 8] = [1, 2, 4, 8, 16, 32, 64, 128];
const CCP4_PCK_MASK: [u8; 9] = [0x00, 0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x7F, 0xFF];
const CCP4_PCK_BIT_COUNT: [u32; 8] = [0, 4, 5, 6, 7, 8, 16, 32];

#[cfg(test)]
mod tests {
    use crate::frame::Frame;
    use crate::mar::Mar;

    #[test]
    fn test_mar_read() {
        let frame = Mar::read_file("testdata/test.mar3450").unwrap();
        assert_eq!(frame.dim1(), 32, "dim1 != 32");
        assert_eq!(frame.dim2(), 32, "dim2 != 32");
        assert_eq!(frame.sum(), 17347273., "frame sum != 17347273.");
        assert_eq!(frame.max(), 10000000., "frame max != 10000000.");
        assert_eq!(frame.min(), 9., "frame min != 9.");
    }

    #[test]
    fn test_real_mar() {
        let testfile = utilities::download_test_file(
            "LR104_000.mar3450",
            "LR104_000.mar3450.tar.bz2",
            "85ce193ed808f2a2ff95f992691f33cc",
        )
        .unwrap();
        let frame = Mar::read_file(testfile).unwrap();
        assert_eq!(frame.dim1(), 3450, "dim1 != 3450");
        assert_eq!(frame.dim2(), 3450, "dim2 != 3450");
        assert_eq!(frame.sum(), 17617644051., "frame sum != 17617644051.");
        assert_eq!(frame.max(), 36834., "frame max != 36834.");
        assert_eq!(frame.min(), 0., "frame min != 0.");
        assert_eq!(
            frame.array[frame.dim2() * 1111 + 2222],
            944.,
            "frame[1111, 2222] != 944."
        );
    }
}
