//! Rust implementation of IO routines to read and write images produced by various
//! 2D X-ray detectors.
//!
//! # Supported formats
//! * Bruker
//! * Dectris Eiger HDF5 with bitshuffle plugin
//! * Dectris MiniCBF
//! * Agilent (Oxford) Crysalis Esperanto compressed bitfield
//! * ESRF data format EDF
//! * Mar345
//! * Tif (including int32 tifs produced by Dectis Pilatus)
//!
//! Usage example:
//! ```rust
//! use std::{io, fmt};
//! use cryiorust::frame::{self, Frame};
//! use std::path::Path;
//!
//! fn test_cbf<P: AsRef<Path> + fmt::Debug>(path: P) -> io::Result<Box<dyn Frame>> {
//!     let testfile = path;
//!     let cbf: Box<dyn Frame> = frame::open(testfile)?;
//!     println!("Frame size: {}x{}", cbf.dim1(), cbf.dim2());
//!     println!("Frame sum: {}", cbf.sum());
//!     Ok(cbf)
//! }
//! ```
//!

pub mod bruker;
pub mod cbf;
pub mod edf;
pub mod eiger;
pub mod esperanto;
pub mod frame;
pub mod mar;
pub mod tif;
