//! Array structure and Frame trait which encapsulate working with arrays for different
//! frame formats.
use crate::bruker::Bruker;
use crate::cbf::Cbf;
use crate::edf::Edf;
use crate::eiger::Eiger;
use crate::esperanto::Esperanto;
use crate::frame::FrameError::HeaderError;
use crate::mar::Mar;
use crate::tif::Tif;
use byteorder::{ByteOrder, ReadBytesExt};
use std::collections::HashMap;
use std::fmt::{Debug, Formatter};
use std::io::Cursor;
use std::num::ParseIntError;
use std::path::Path;
use std::str::Utf8Error;
use std::{cmp, default, fmt, io, mem, ops};

/// Frame Header possible values.
#[derive(PartialEq, Debug)]
pub enum HeaderEntry {
    Empty,
    Number(i64),
    Float(f64),
    String(String),
    Pixels([f64; 2]),
}

/// Header type as a [HashMap] with [String] key and [HeaderEntry] value.
pub type Header = HashMap<String, HeaderEntry>;

/// Basic struct for 2D arrays. It just contains dim1 and dim2 for dimensions and data
/// casted to [f64].
///
pub struct Array {
    dim1: usize,
    dim2: usize,
    data: Vec<f64>,
}

impl ops::Index<usize> for Array {
    type Output = f64;

    fn index(&self, index: usize) -> &Self::Output {
        self.data.index(index)
    }
}

impl ops::IndexMut<usize> for Array {
    fn index_mut(&mut self, index: usize) -> &mut Self::Output {
        self.data.index_mut(index)
    }
}

impl cmp::PartialEq for Array {
    fn eq(&self, other: &Self) -> bool {
        self.dim2 == other.dim2 && self.dim1 == other.dim1 && self.data.len() == other.data.len()
    }
}

impl cmp::PartialEq for dyn Frame {
    fn eq(&self, other: &Self) -> bool {
        self.array() == other.array()
    }
}

impl ops::SubAssign<&Array> for &mut Array {
    fn sub_assign(&mut self, rhs: &Array) {
        for (s, r) in self.data.iter_mut().zip(rhs.data.iter()) {
            *s -= *r;
        }
    }
}

impl ops::SubAssign<f64> for &mut Array {
    fn sub_assign(&mut self, rhs: f64) {
        for s in &mut self.data {
            *s -= rhs;
        }
    }
}

impl ops::AddAssign<&Array> for &mut Array {
    fn add_assign(&mut self, rhs: &Array) {
        for (s, r) in self.data.iter_mut().zip(rhs.data.iter()) {
            *s += *r;
        }
    }
}

impl ops::DivAssign<&Array> for &mut Array {
    fn div_assign(&mut self, rhs: &Array) {
        for (s, r) in self.data.iter_mut().zip(rhs.data.iter()) {
            *s /= *r;
        }
    }
}

impl ops::DivAssign<f64> for &mut Array {
    fn div_assign(&mut self, rhs: f64) {
        for s in &mut self.data {
            *s /= rhs;
        }
    }
}

impl ops::MulAssign<f64> for &mut Array {
    fn mul_assign(&mut self, rhs: f64) {
        for s in &mut self.data {
            *s *= rhs;
        }
    }
}

impl default::Default for Array {
    fn default() -> Self {
        Array::new()
    }
}

macro_rules! copy_from_byte_slice_impl {
    ($(#[$attr:meta])* => $Name:ident, $Read:ident) => {
        $(#[$attr])*
        pub fn $Name(dim1: usize, dim2: usize, src: &[u8]) -> FrameResult<Array> {
            let size = dim1 * dim2;
            let mut vec = Vec::with_capacity(size);
            unsafe { vec.set_len(size) };
            let mut cursor = Cursor::new(src);
            for value in &mut vec {
                *value = cursor.$Read()? as f64;
            }
            Ok(Array::with_data(dim1, dim2, vec))
        }
    };
}

macro_rules! copy_from_slice_impl {
    ($(#[$attr:meta])* => $Name:ident, $Read:ident) => {
        $(#[$attr])*
        pub fn $Name<F: ByteOrder>(dim1: usize, dim2: usize, src: &[u8]) -> FrameResult<Array> {
            let size = dim1 * dim2;
            let mut vec = Vec::with_capacity(size);
            unsafe { vec.set_len(size) };
            let mut cursor = Cursor::new(src);
            for value in &mut vec {
                *value = cursor.$Read::<F>()? as f64;
            }
            Ok(Array::with_data(dim1, dim2, vec))
        }
    };
}

impl Array {
    /// Creates a new empty [Array].
    pub fn new() -> Array {
        Array {
            dim1: 0,
            dim2: 0,
            data: vec![],
        }
    }

    /// Checks whether [Array] is empty.
    pub fn is_empty(&self) -> bool {
        self.data.is_empty()
    }

    /// Swaps [Array]'s dimensions.
    pub fn swap_dims(&mut self) {
        mem::swap(&mut self.dim1, &mut self.dim2);
    }

    /// Clears [Array] and frees its memory.
    pub fn clear(&mut self) {
        self.dim1 = 0;
        self.dim2 = 0;
        self.data.clear();
        self.data.shrink_to_fit();
    }

    /// Returns internal [Vec<f64>] consuming the [Array].
    pub fn consume(self) -> Vec<f64> {
        self.data
    }

    /// Creates [Array] with dims and allocates the needed memory for the data.
    pub fn with_dims(dim1: usize, dim2: usize) -> Array {
        Array {
            dim1,
            dim2,
            data: Vec::with_capacity(dim1 * dim2),
        }
    }

    /// Sets the length of the [Array].
    pub unsafe fn set_len(&mut self) {
        self.data.set_len(self.dim1 * self.dim2);
    }

    /// Creates an [Array] from the given data.
    pub fn with_data(dim1: usize, dim2: usize, data: Vec<f64>) -> Array {
        Array { dim1, dim2, data }
    }

    copy_from_byte_slice_impl!(
        /// Creates an [Array] copying a slice of [u8] casting the values to [f64].
        => from_slice_u8, read_u8
    );

    copy_from_byte_slice_impl!(
        /// Creates an [Array] copying a slice of [i8] casting the values to [f64].
        => from_slice_i8, read_i8
    );

    copy_from_slice_impl!(
        /// Creates an [Array] copying a slice of [u16] casting the values to [f64] taking into account
        /// the [ByteOrder].
        => from_slice_u16, read_u16
    );

    copy_from_slice_impl!(
        /// Creates an [Array] copying a slice of [i16] casting the values to [f64] taking into account
        /// the [ByteOrder].
        => from_slice_i16, read_i16
    );

    copy_from_slice_impl!(
        /// Creates an [Array] copying a slice of [i32] casting the values to [f64] taking into account
        /// the [ByteOrder].
        => from_slice_i32, read_i32
    );

    copy_from_slice_impl!(
        /// Creates an [Array] copying a slice of [u32] casting the values to [f64] taking into account
        /// the [ByteOrder].
        => from_slice_u32, read_u32
    );

    copy_from_slice_impl!(
        /// Creates an [Array] copying a slice of [u64] casting the values to [f64] taking into account
        /// the [ByteOrder].
        => from_slice_u64, read_u64
    );

    copy_from_slice_impl!(
        /// Creates an [Array] copying a slice of [i64] casting the values to [f64] taking into account
        /// the [ByteOrder].
        => from_slice_i64, read_i64
    );

    copy_from_slice_impl!(
        /// Creates an [Array] copying a slice of [f32] casting the values to [f64] taking into account
        /// the [ByteOrder].
        => from_slice_f32, read_f32
    );

    copy_from_slice_impl!(
        /// Creates an [Array] copying a slice of [f64] casting the values to [f64] taking into account
        /// the [ByteOrder].
        => from_slice_f64, read_f64
    );

    /// Returns length of the [Array].
    pub fn len(&self) -> usize {
        self.data.len()
    }

    /// Returns a reference to the [Vec<f64>] in the [Array].
    pub fn data(&self) -> &Vec<f64> {
        &self.data
    }

    /// Returns a mutable reference to the [Vec<f64>] in the [Array].
    pub fn data_mut(&mut self) -> &mut Vec<f64> {
        &mut self.data
    }

    /// Returns first dimension of the [Array].
    pub fn dim1(&self) -> usize {
        self.dim1
    }

    /// Returns second dimension of the [Array].
    pub fn dim2(&self) -> usize {
        self.dim2
    }

    /// Returns a tuple of the [Array] dimensions.
    pub fn dims(&self) -> (usize, usize) {
        (self.dim1, self.dim2)
    }

    /// Returns the [Array] capacity.
    pub fn capacity(&self) -> usize {
        self.data.capacity()
    }

    /// Pushes value into the array data.
    pub fn push(&mut self, value: f64) {
        self.data.push(value);
    }

    /// Gets a cell value from the [Array].
    pub fn cell(&self, i: usize, j: usize) -> f64 {
        self.data[i * self.dim2 + j]
    }

    /// Sets a cell value of the [Array].
    pub fn set_cell(&mut self, i: usize, j: usize, value: f64) {
        self.data[i * self.dim2 + j] = value;
    }

    /// Calculates the sum of the [Array] values.
    pub fn sum(&self) -> f64 {
        let mut sum = 0.;
        for value in &self.data {
            sum += *value;
        }
        sum
    }

    /// Gets the minimum value of the [Array].
    pub fn min(&self) -> f64 {
        let mut min = f64::MAX;
        for value in &self.data {
            if min > *value {
                min = *value;
            }
        }
        min
    }

    /// Gets the maximum value of the [Array].
    pub fn max(&self) -> f64 {
        let mut max = f64::MIN;
        for value in &self.data {
            if max < *value {
                max = *value;
            }
        }
        max
    }
}

/// Main trait which implemented by all the crate frame structures for dynamic dispatching.
pub trait Frame: Sync + Send {
    /// Reimplemented method returning a reference of the [Array].
    fn array(&self) -> &Array;

    /// Reimplemented method returning a reference of the [Header].
    fn header(&self) -> &Header;

    /// Reimplemented method returning a mutable reference of the [Header].
    fn header_mut(&mut self) -> &mut Header;

    /// Reimplemented method returning a mutable reference of the [Array].
    fn array_mut(&mut self) -> &mut Array;

    /// Reimplemented method which changes [Array] in the [Frame].
    fn set_array(&mut self, array: Array);

    /// Reimplemented method which returns [Array] consuming the [Frame].
    fn consume_array(self: Box<Self>) -> Array;

    /// Reimplemented method which returns [Array] consuming the [Frame].
    fn take_array(&mut self) -> Array;

    /// Switches [Frame] to the next sub-frame.
    fn next_frame(&mut self) -> FrameResult<usize> {
        Err(FrameError::NoMoreFrames)
    }

    /// Returns total number of sub-frames.
    fn total_frames(&self) -> usize {
        1
    }

    /// Returns the current frame number.
    fn current_frame(&self) -> usize {
        0
    }

    /// Checks whether this [Frame] contains sub-frames.
    fn is_multi(&self) -> bool {
        false
    }

    /// Returns the sum of the [Frame]'s [Array].
    fn sum(&self) -> f64 {
        self.array().sum()
    }

    /// Returns the minimum value of the [Frame]'s [Array].
    fn min(&self) -> f64 {
        self.array().min()
    }

    /// Returns the maximum value of the [Frame]'s [Array].
    fn max(&self) -> f64 {
        self.array().max()
    }

    /// Returns the first dimension of the [Frame]'s [Array].
    fn dim1(&self) -> usize {
        self.array().dim1
    }

    /// Returns the second dimension of the [Frame]'s [Array].
    fn dim2(&self) -> usize {
        self.array().dim2
    }

    /// Returns an [i64] value from the [Frame]'s [Header].
    fn get_header_i64(&self, key: &str) -> FrameResult<i64> {
        match self.header().get(key) {
            Some(val) => match val {
                HeaderEntry::Number(val) => Ok(*val),
                HeaderEntry::Float(val) => Ok(*val as i64),
                HeaderEntry::String(_) => self.get_header_str_as_i64(key),
                _ => Err(HeaderError(format!("bad header entry for {}", key).into())),
            },
            None => Err(HeaderError(format!("could not find {}", key).into())),
        }
    }

    /// Returns an [&str] value from the [Frame]'s [Header] or empty [&str] if the key does not
    /// exist.
    fn get_header_str_or_empty(&self, key: &str) -> &str {
        match self.header().get(key) {
            Some(entry) => match entry {
                HeaderEntry::String(value) => value,
                _ => "",
            },
            None => "",
        }
    }

    /// Returns an [f64] value from the [Frame]'s [Header] or zero if the key does not
    /// exist.
    fn get_header_float(&self, key: &str) -> f64 {
        match self.header().get(key) {
            Some(entry) => match entry {
                HeaderEntry::Float(value) => *value,
                HeaderEntry::Empty => 0.,
                HeaderEntry::Number(value) => *value as f64,
                HeaderEntry::Pixels(_) => 0.,
                HeaderEntry::String(value) => match value.parse::<f64>() {
                    Ok(value) => value,
                    Err(_) => 0.,
                },
            },
            None => 0.,
        }
    }

    /// Returns an [&str] value from the [Frame]'s [Header].
    fn get_header_str(&self, key: &str) -> FrameResult<&str> {
        match self.header().get(key) {
            Some(entry) => {
                return match entry {
                    HeaderEntry::String(value) => Ok(value),
                    _ => Err(HeaderError(
                        format!("bad header entry for '{}'", key).into(),
                    )),
                };
            }
            None => Err(HeaderError(
                format!("could not find '{}' header entry", key).into(),
            )),
        }
    }

    /// Returns a [String] value from the [Frame]'s [Header]. The [String] is empty if the key
    /// does not exist.
    fn get_header_float_as_string_or_empty(&self, key: &str, exp: bool) -> String {
        match self.header().get(key) {
            Some(entry) => match entry {
                HeaderEntry::Float(value) => {
                    if exp {
                        format!("{:e}", value)
                    } else {
                        format!("{}", value)
                    }
                }
                _ => String::new(),
            },
            None => String::new(),
        }
    }

    fn get_header_int_as_string_or_empty(&self, key: &str) -> String {
        match self.header().get(key) {
            Some(entry) => match entry {
                HeaderEntry::Number(value) => format!("{}", value),
                _ => String::new(),
            },
            None => String::new(),
        }
    }

    fn get_header_str_as_i64(&self, key: &str) -> FrameResult<i64> {
        let value = self.get_header_str(key)?;
        match value.parse::<i64>() {
            Ok(v) => Ok(v),
            Err(e) => Err(HeaderError(
                format!("could not parse key '{}' in the header: {}", key, e).into(),
            )),
        }
    }

    /// Inserts a [HeaderEntry] value into the [Frame]'s [Header] with a [String] as key.
    fn insert(&mut self, key: String, value: HeaderEntry) {
        self.header_mut().insert(key, value);
    }

    /// Swaps dims of the [Frame]'s [Array].
    fn swap_dims(&mut self) {
        self.array_mut().swap_dims();
    }
}

impl fmt::Display for HeaderEntry {
    fn fmt(&self, f: &mut fmt::Formatter) -> Result<(), fmt::Error> {
        match self {
            HeaderEntry::Number(v) => write!(f, "{}", v),
            HeaderEntry::Float(v) => write!(f, "{}", v),
            HeaderEntry::String(v) => write!(f, "{}", v),
            HeaderEntry::Empty => write!(f, "empty"),
            HeaderEntry::Pixels(v) => write!(f, "[{}, {}]", v[0], v[1]),
        }
    }
}

/// A universal function to open a file as a [Frame].
///
/// Example:
/// ```rust
/// use std::{io, fmt};
/// use cryiorust::frame::{Frame, self};
/// use std::path::Path;
///
/// fn test_frame<P: AsRef<Path> + fmt::Debug>(path: P) -> io::Result<Box<dyn Frame>> {
///     let testfile = path;
///     let frame: Box<dyn Frame> = frame::open(testfile)?;
///     Ok(frame)
/// }
/// ```
pub fn open<P: AsRef<Path> + fmt::Debug>(path: P) -> FrameResult<Box<dyn Frame>> {
    if let Some(ext) = path.as_ref().to_path_buf().extension() {
        if ext == "cbf" {
            return Ok(Box::new(Cbf::read_file(path)?));
        } else if ext == "edf" {
            return Ok(Box::new(Edf::read_file(path)?));
        } else if ext == "gfrm" {
            return Ok(Box::new(Bruker::read_file(path)?));
        } else if ext == "mar2300" || ext == "mar3450" || ext == "mar2560" {
            return Ok(Box::new(Mar::read_file(path)?));
        } else if ext == "tif" || ext == "tiff" {
            return Ok(Box::new(Tif::read_file(path)?));
        } else if ext == "esperanto" || ext == "esp" {
            return Ok(Box::new(Esperanto::read_file(path)?));
        } else if ext == "h5" {
            return Ok(Box::new(Eiger::read_file(path)?));
        }
    }
    return Err(FrameError::UnsupportedError);
}

/// A universal checker whether current file can be open as a [Frame].
/// It checks only the extensions of the file name.
pub fn is_frame<P: AsRef<Path> + fmt::Debug>(path: P) -> bool {
    if let Some(ext) = path.as_ref().to_path_buf().extension() {
        if let Some(ext) = ext.to_str() {
            return match ext {
                "cbf" => true,
                "edf" => true,
                "gfrm" => true,
                "mar2300" | "mar3450" | "mar2560" => true,
                "tif" | "tiff" => true,
                "esperanto" | "esp" => true,
                "h5" => true,
                _ => false,
            };
        }
    }
    false
}

impl ops::Index<usize> for dyn Frame {
    type Output = f64;

    fn index(&self, index: usize) -> &Self::Output {
        self.array().index(index)
    }
}

impl ops::IndexMut<usize> for dyn Frame {
    fn index_mut(&mut self, index: usize) -> &mut Self::Output {
        self.array_mut().index_mut(index)
    }
}

/// [FrameError] error descriptor struct.
#[derive(Debug)]
pub struct ErrorDescription {
    pub description: String,
}

impl ErrorDescription {
    fn new<S: Into<String>>(description: S) -> ErrorDescription {
        ErrorDescription {
            description: description.into(),
        }
    }
}

/// Enum with possible errors for [Frame].
#[derive(Debug)]
pub enum FrameError {
    UnsupportedError,
    IoError(io::Error),
    HeaderError(ErrorDescription),
    FormatError(ErrorDescription),
    NoMoreFrames,
}

impl fmt::Display for FrameError {
    fn fmt(&self, f: &mut Formatter<'_>) -> Result<(), fmt::Error> {
        match self {
            FrameError::UnsupportedError => write!(f, "not a frame file"),
            FrameError::IoError(e) => std::fmt::Display::fmt(&e, f),
            FrameError::HeaderError(e) => write!(f, "{}", e.description),
            FrameError::FormatError(e) => write!(f, "{}", e.description),
            FrameError::NoMoreFrames => write!(f, "no more frames available"),
        }
    }
}

/// [Result] enum for [Frame].
pub type FrameResult<T> = Result<T, FrameError>;

impl From<io::Error> for FrameError {
    fn from(err: io::Error) -> Self {
        FrameError::IoError(err)
    }
}

impl From<FrameError> for io::Error {
    fn from(err: FrameError) -> Self {
        match err {
            FrameError::UnsupportedError => {
                io::Error::new(io::ErrorKind::InvalidData, "unsupported frame")
            }
            FrameError::IoError(err) => err,
            HeaderError(err) => io::Error::new(io::ErrorKind::InvalidData, err.description),
            FrameError::FormatError(err) => {
                io::Error::new(io::ErrorKind::InvalidData, err.description)
            }
            FrameError::NoMoreFrames => {
                io::Error::new(io::ErrorKind::NotFound, "no more frames available")
            }
        }
    }
}

impl From<std::str::Utf8Error> for FrameError {
    fn from(err: Utf8Error) -> Self {
        FrameError::HeaderError(ErrorDescription::new(format!(
            "header contains non-UTF8 characters: {}",
            err
        )))
    }
}

impl Into<ErrorDescription> for String {
    fn into(self) -> ErrorDescription {
        ErrorDescription::new(self)
    }
}

impl Into<ErrorDescription> for &str {
    fn into(self) -> ErrorDescription {
        ErrorDescription::new(self.to_owned())
    }
}

impl From<std::num::ParseIntError> for FrameError {
    fn from(err: ParseIntError) -> Self {
        FrameError::HeaderError(format!("not a number in header: {}", err).into())
    }
}

/// This function gives a hint whether a filename can be a multi-frame container.
/// Only .h5 [Eiger] files are now considered as multi-frame containers.
pub fn is_possible_multi<T: AsRef<Path>>(name: T) -> bool {
    match name.as_ref().extension() {
        None => false,
        Some(ext) => match ext.to_str() {
            None => false,
            Some(ext) => match ext {
                "h5" => true,
                _ => false,
            },
        },
    }
}
