//! Dectris Eiger frame reader.
//!
//! It supports only HDF5 with bitshuffle plugin filter.
//!
//! Example usage:
//! ```rust
//! use cryiorust::eiger::Eiger;
//! use cryiorust::frame::Frame;
//! use std::path::Path;
//!
//! fn test_eiger<P: AsRef<Path>>(path: P) {
//!     let testfile = path;
//!     let mut frame = Eiger::read_file(testfile).unwrap();
//!     assert_eq!(frame.sum(), 41808654083., "frame sum != 41808654083");
//!     frame.next_frame().unwrap();
//!     assert_eq!(frame.sum(), 41808571170., "frame sum != 41808571170");
//! }
//! ```
use crate::frame::{Array, Frame, FrameError, FrameResult, Header};
use byteorder::LittleEndian;
use hdf5::types::{FloatSize, IntSize, TypeDescriptor};
use hdf5::{Dataset, Error};
use hdf5_bitshuffle::register_bitshuffle_plugin;
use ndarray::s;
use reinterpret::reinterpret_slice;
use std::path::Path;

/// Main exported struct.
pub struct Eiger {
    array: Array,
    header: Header,
    n_frames: usize,
    current_frame: usize,
    dataset: Dataset,
    datatype: TypeDescriptor,
    dims: Vec<usize>,
}

macro_rules! copy_vec1 {
    ($type:ty, $self:ident, $frame:ident, $fn:ident) => {{
        let vec = $self
            .dataset
            .read_slice_2d(s![$frame, .., ..])?
            .into_raw_vec();
        let slice = unsafe { reinterpret_slice::<$type, u8>(&vec) };
        Array::$fn($self.dims[1], $self.dims[2], slice)?
    }};
}

macro_rules! copy_vec2 {
    ($type:ty, $self:ident, $frame:ident, $fn:ident) => {{
        let vec = $self
            .dataset
            .read_slice_2d(s![$frame, .., ..])?
            .into_raw_vec();
        let slice = unsafe { reinterpret_slice::<$type, u8>(&vec) };
        Array::$fn::<LittleEndian>($self.dims[1], $self.dims[2], slice)?
    }};
}

macro_rules! copy_vec2_and_sanitize {
    ($type:ty, $self:ident, $frame:ident, $fn:ident) => {{
        let vec = $self
            .dataset
            .read_slice_2d(s![$frame, .., ..])?
            .into_raw_vec();
        let slice = unsafe { reinterpret_slice::<$type, u8>(&vec) };
        let mut array = Array::$fn::<LittleEndian>($self.dims[1], $self.dims[2], slice)?;
        let gap = <$type>::MAX as f64;
        let dead = (<$type>::MAX - 1) as f64;
        array.data_mut().iter_mut().for_each(|v| {
            if *v == gap {
                *v = -1.
            } else if *v == dead {
                *v = -2.
            }
        });
        array
    }};
}

impl Eiger {
    /// Reads frames from a file.
    ///
    /// It can read only one frame at a time.
    pub fn read_file<T: AsRef<Path>>(path: T) -> Result<Eiger, FrameError> {
        register_bitshuffle_plugin();
        let file = hdf5::File::open(path)?;
        let group = file.group("entry/data")?;
        let dataset = group.dataset("data")?;
        let ndim = dataset.ndim();
        if ndim != 3 {
            return Err(FrameError::FormatError(
                "not an Eiger HDF5: not enough dimensions".into(),
            ));
        }
        let chunks = match dataset.num_chunks() {
            None => {
                return Err(FrameError::FormatError(
                    "not an Eiger HDF5: chunks not found".into(),
                ));
            }
            Some(chunks) => {
                if chunks == 0 {
                    return Err(FrameError::FormatError(
                        "not an Eiger HDF5: not enough chunks".into(),
                    ));
                } else {
                    chunks
                }
            }
        };
        let space = dataset.space()?;
        let dims = space.shape();
        if dims.len() != ndim || dims[0] != chunks {
            return Err(FrameError::FormatError(
                "not an Eiger HDF5: chunks are not equal to the number of frames".into(),
            ));
        }
        let datatype = dataset.dtype()?;
        let datatype = datatype.to_descriptor()?;
        let mut eiger = Eiger {
            array: Array::new(),
            header: Default::default(),
            n_frames: chunks,
            current_frame: 0,
            dataset,
            datatype,
            dims,
        };
        eiger.read_frame_frame(0)?;
        Ok(eiger)
    }

    fn read_frame_frame(&mut self, frame: usize) -> FrameResult<()> {
        self.array = match self.datatype {
            TypeDescriptor::Integer(size) => match size {
                IntSize::U1 => copy_vec1!(i8, self, frame, from_slice_i8),
                IntSize::U2 => copy_vec2!(i16, self, frame, from_slice_i16),
                IntSize::U4 => copy_vec2!(i32, self, frame, from_slice_i32),
                IntSize::U8 => copy_vec2!(i64, self, frame, from_slice_i64),
            },
            TypeDescriptor::Unsigned(size) => match size {
                IntSize::U1 => copy_vec1!(u8, self, frame, from_slice_u8),
                IntSize::U2 => copy_vec2_and_sanitize!(u16, self, frame, from_slice_u16),
                IntSize::U4 => copy_vec2_and_sanitize!(u32, self, frame, from_slice_u32),
                IntSize::U8 => copy_vec2_and_sanitize!(u64, self, frame, from_slice_u64),
            },
            TypeDescriptor::Float(size) => match size {
                FloatSize::U4 => copy_vec2!(f32, self, frame, from_slice_f32),
                FloatSize::U8 => copy_vec2!(f64, self, frame, from_slice_f64),
            },
            _ => {
                return Err(FrameError::FormatError(
                    "eiger data type is not supported".into(),
                ))
            }
        };
        self.current_frame = frame;
        Ok(())
    }

    /// Switches frame pointer to ```frame``` number.
    pub fn set_frame(&mut self, frame: usize) -> FrameResult<usize> {
        if frame < self.n_frames {
            self.read_frame_frame(frame)?;
            Ok(self.current_frame)
        } else {
            Err(FrameError::NoMoreFrames)
        }
    }
}

impl Frame for Eiger {
    fn array(&self) -> &Array {
        &self.array
    }

    fn header(&self) -> &Header {
        &self.header
    }

    fn header_mut(&mut self) -> &mut Header {
        &mut self.header
    }

    fn array_mut(&mut self) -> &mut Array {
        &mut self.array
    }

    fn set_array(&mut self, array: Array) {
        self.array = array;
    }

    fn consume_array(self: Box<Self>) -> Array {
        self.array
    }

    fn take_array(&mut self) -> Array {
        std::mem::take(&mut self.array)
    }

    fn next_frame(&mut self) -> FrameResult<usize> {
        self.set_frame(self.current_frame() + 1)
    }

    fn total_frames(&self) -> usize {
        self.n_frames
    }

    fn current_frame(&self) -> usize {
        self.current_frame
    }

    fn is_multi(&self) -> bool {
        true
    }
}

impl From<hdf5::Error> for FrameError {
    fn from(err: Error) -> Self {
        match err {
            Error::HDF5(es) => match es.expand() {
                Ok(es) => FrameError::FormatError(es.description().into()),
                Err(_) => FrameError::FormatError("HDF5 crate unknown error".into()),
            },
            Error::Internal(err) => FrameError::FormatError(err.into()),
        }
    }
}

#[cfg(test)]
mod tests {
    extern crate utilities;
    use crate::eiger::Eiger;
    use crate::frame::Frame;

    #[test]
    fn test_eiger() {
        let testfile = utilities::download_test_file(
            "200Hz_0.1dpf_1_data_000001.h5",
            "eiger.tar.bz2",
            "9614362dae905499f32c095c6201776f",
        )
        .unwrap();
        let mut frame = Eiger::read_file(testfile).unwrap();
        assert_eq!(frame.sum(), 2715395., "frame sum != 2715395");
        frame.next_frame().unwrap();
        assert_eq!(frame.sum(), 2698018., "frame sum != 2698018");
        frame.set_frame(49).unwrap();
        assert_eq!(frame.sum(), 2666414., "frame sum != 2666414");
    }
}
