//! EDF (ESRF Data format) reader and writer.
//!
//! Only simple EDF frames are supported, no gzip, no multiframe.
//!
//! Usage example:
//! * Read and write to a file:
//! ```rust
//! use cryiorust::edf::Edf;
//! use std::path::Path;
//!
//! fn edf_reader_writer<P: AsRef<Path>>(path: P) {
//!     let mut frame = Edf::read_file(path).unwrap();
//!     let path = tempfile::NamedTempFile::new().unwrap();
//!     frame.write_file(&path).unwrap();
//! }
//! ```
//! * Read from a buffer:
//! ```rust
//! use std::fs::File;
//! use std::io::{Read, self};
//! use cryiorust::edf::Edf;
//! use std::path::Path;
//!
//! fn edf_buffer_reader<P: AsRef<Path>>(path: P) {
//!     let mut file = File::open(path).unwrap();
//!     let mut data = vec![];
//!     file.read_to_end(&mut data).unwrap();
//!     let data = io::Cursor::new(data);
//!     let _frame = Edf::read_buffer(data).unwrap();
//! }
//! ```
use crate::frame::{Array, Frame, FrameResult, Header, HeaderEntry};
use byteorder::{BigEndian, ByteOrder, LittleEndian, NativeEndian, ReadBytesExt, WriteBytesExt};
use std::fs::File;
use std::io::{BufRead, BufWriter, Seek, SeekFrom, Write};
use std::{io, mem};
extern crate askama;
use crate::frame::FrameError::FormatError;
use askama::Template;
use cryiorust_derive::Frame;
use std::path::Path;

/// Main exported struct.
#[derive(Frame)]
pub struct Edf {
    header: Header,
    array: Array,
    bin_size: usize,
    data_start: u64,
    dim1: usize,
    dim2: usize,
    size: usize,
}

impl Edf {
    /// Creates a new empty EDF struct.
    pub fn new() -> Edf {
        Edf {
            header: Default::default(),
            array: Array::new(),
            bin_size: 0,
            data_start: 0,
            dim1: 0,
            dim2: 0,
            size: 0,
        }
    }

    /// Reads an EDF frame from a buffer.
    pub fn read_buffer<R: BufRead + Seek>(mut reader: R) -> FrameResult<Edf> {
        let mut frame = Edf::new();
        frame.read_header(&mut reader)?;
        frame.read_data(&mut reader)?;
        Ok(frame)
    }

    /// Writes EDF into a file.
    pub fn write_file<P: AsRef<Path>>(&mut self, path: P) -> FrameResult<()> {
        self.save_file(path, DataType::F64)
    }

    /// Reads EDF from a file
    pub fn read_file<P: AsRef<Path>>(path: P) -> FrameResult<Edf> {
        let mut frame = Edf::new();
        let mut reader = io::BufReader::new(File::open(path)?);
        frame.read_header(&mut reader)?;
        frame.read_data(&mut reader)?;
        Ok(frame)
    }

    fn read_header<R: BufRead + Seek>(&mut self, reader: &mut R) -> FrameResult<()> {
        loop {
            if reader.read_u8()? as char == HEADER_START {
                break;
            }
        }
        let mut header: Vec<u8> = vec![];
        loop {
            let chr = reader.read_u8()?;
            self.data_start += 1;
            if chr as char == HEADER_END {
                loop {
                    self.data_start += 1;
                    if reader.read_u8()? as char == '\n' {
                        self.data_start += 1;
                        break;
                    }
                }
                break;
            }
            header.push(chr);
        }
        self.split_header(&header);
        Ok(())
    }

    fn read_data<R: BufRead + Seek>(&mut self, reader: &mut R) -> FrameResult<()> {
        self.set_array_parameters()?;
        if self.dim1 == 0 || self.dim2 == 0 || self.bin_size == 0 {
            return Err(FormatError("bad header values".into()));
        }
        let data_type = self.parse_data_type()?;
        reader.seek(SeekFrom::Start(self.data_start))?;
        let mut data: Vec<u8> = Vec::with_capacity(self.bin_size);
        unsafe { data.set_len(self.bin_size) };
        reader.read_exact(&mut data)?;
        match self.get_header_str(KEY_BYTEORDER)? {
            BO_LE => self.unpack::<LittleEndian>(data_type, &data),
            BO_BE => self.unpack::<BigEndian>(data_type, &data),
            _ => return Err(FormatError("bad endiannes value".into())),
        }
    }

    fn parse_data_type(&self) -> FrameResult<DataType> {
        match self.get_header_str(KEY_DATATYPE)? {
            "SignedByte" | "Signed8" => Ok(DataType::I8),
            "UnsignedByte" | "Unsigned8" => Ok(DataType::U8),
            "SignedShort" | "Signed16" => Ok(DataType::I16),
            "UnsignedShort" | "Unsigned16" | "UnsignedShortInteger" => Ok(DataType::U16),
            "SignedInteger" | "Signed32" | "SignedLong" => Ok(DataType::I32),
            "UnsignedInteger" | "Unsigned32" | "UnsignedLong" => Ok(DataType::U32),
            "Signed64" => Ok(DataType::I64),
            "Unsigned64" => Ok(DataType::U64),
            "FloatValue" | "FLOATVALUE" | "FLOAT" | "Float" | "FloatIEEE32" | "Float32" => {
                Ok(DataType::F32)
            }
            "Double" | "DoubleValue" | "FloatIEEE64" | "DoubleIEEE64" => Ok(DataType::F64),
            _ => Err(FormatError("data type not found".into())),
        }
    }

    fn set_array_parameters(&mut self) -> FrameResult<()> {
        self.dim1 = self.get_header_str_as_i64(KEY_DIM1)? as usize;
        self.dim2 = self.get_header_str_as_i64(KEY_DIM2)? as usize;
        self.size = self.dim1 * self.dim2;
        self.bin_size = self.get_header_str_as_i64(KEY_SIZE)? as usize;
        Ok(())
    }

    fn unpack<F: ByteOrder>(&mut self, data_type: DataType, data: &[u8]) -> FrameResult<()> {
        self.array = match data_type {
            DataType::I8 => Array::from_slice_i8(self.dim1, self.dim2, &data),
            DataType::U8 => Array::from_slice_u8(self.dim1, self.dim2, &data),
            DataType::I16 => Array::from_slice_i16::<F>(self.dim1, self.dim2, &data),
            DataType::U16 => Array::from_slice_u16::<F>(self.dim1, self.dim2, &data),
            DataType::I32 => Array::from_slice_i32::<F>(self.dim1, self.dim2, &data),
            DataType::U32 => Array::from_slice_u32::<F>(self.dim1, self.dim2, &data),
            DataType::I64 => Array::from_slice_i64::<F>(self.dim1, self.dim2, &data),
            DataType::U64 => Array::from_slice_u64::<F>(self.dim1, self.dim2, &data),
            DataType::F32 => Array::from_slice_f32::<F>(self.dim1, self.dim2, &data),
            DataType::F64 => Array::from_slice_f64::<F>(self.dim1, self.dim2, &data),
        }?;
        Ok(())
    }

    fn split_header(&mut self, header: &Vec<u8>) {
        let x: &[_] = &[' ', ';'];
        let header = unsafe { std::str::from_utf8_unchecked(header) };
        for line in header.lines() {
            let line = line.trim();
            if line.is_empty() {
                continue;
            }
            let parts: Vec<&str> = line.split('=').map(|z| z.trim_matches(x).trim()).collect();
            if parts.len() != 2 {
                continue;
            }
            self.header.insert(
                parts[0].to_string(),
                HeaderEntry::String(parts[1].to_string()),
            );
        }
    }

    /// Saves an arbitrary array into a Writer
    pub fn save_array<P: Write>(
        array: &Array,
        mut header: &mut Header,
        writer: &mut P,
        cast: DataType,
    ) -> FrameResult<()> {
        Edf::save(writer, &mut header, &array, cast)
    }

    fn save<P: Write>(
        writer: &mut P,
        header: &mut Header,
        array: &Array,
        cast: DataType,
    ) -> FrameResult<()> {
        header.insert(
            KEY_DIM1.to_string(),
            HeaderEntry::Number(array.dim1() as i64),
        );
        header.insert(
            KEY_DIM2.to_string(),
            HeaderEntry::Number(array.dim2() as i64),
        );
        let bo = if cfg!(target_endian = "little") {
            BO_LE
        } else {
            BO_BE
        };
        header.insert(
            KEY_BYTEORDER.to_string(),
            HeaderEntry::String(bo.to_string()),
        );
        match cast {
            DataType::I8 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("SignedByte".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<i8>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_i8(*v as i8)?;
                }
            }
            DataType::U8 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("UnsignedByte".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<u8>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_u8(*v as u8)?;
                }
            }
            DataType::I16 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("SignedShort".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<i16>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_i16::<NativeEndian>(*v as i16)?;
                }
            }
            DataType::U16 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("UnsignedShort".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<u16>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_u16::<NativeEndian>(*v as u16)?;
                }
            }
            DataType::I32 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("SignedInteger".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<i32>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_i32::<NativeEndian>(*v as i32)?;
                }
            }
            DataType::U32 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("UnsignedInteger".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<u32>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_u32::<NativeEndian>(*v as u32)?;
                }
            }
            DataType::I64 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("Signed64".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<i64>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_i64::<NativeEndian>(*v as i64)?;
                }
            }
            DataType::U64 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("Unsigned64".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<u64>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_u64::<NativeEndian>(*v as u64)?;
                }
            }
            DataType::F32 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("Float".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<f32>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_f32::<NativeEndian>(*v as f32)?;
                }
            }
            DataType::F64 => {
                header.insert(
                    KEY_DATATYPE.to_string(),
                    HeaderEntry::String("Double".to_string()),
                );
                header.insert(
                    KEY_SIZE.to_string(),
                    HeaderEntry::Number((array.len() * mem::size_of::<f64>()) as i64),
                );
                writer.write_all(
                    EdfHeaderTemplate { header: &header }
                        .render()
                        .unwrap()
                        .as_bytes(),
                )?;
                for v in array.data() {
                    writer.write_f64::<NativeEndian>(*v as f64)?;
                }
            }
        }
        Ok(())
    }

    /// Saves EDF into a buffer.
    pub fn save_buffer<P: Write>(&mut self, writer: &mut P, cast: DataType) -> FrameResult<()> {
        Edf::save(writer, &mut self.header, &self.array, cast)
    }

    /// Saves EDF into a file.
    pub fn save_file<P: AsRef<Path>>(&mut self, path: P, cast: DataType) -> FrameResult<()> {
        let mut writer = BufWriter::new(File::create(path)?);
        self.save_buffer(&mut writer, cast)
    }
}

const HEADER_START: char = '{';
const HEADER_END: char = '}';
const KEY_DIM1: &'static str = "Dim_2";
const KEY_DIM2: &'static str = "Dim_1";
const KEY_SIZE: &'static str = "Size";
const KEY_DATATYPE: &'static str = "DataType";
const KEY_BYTEORDER: &'static str = "ByteOrder";
const BO_LE: &'static str = "LowByteFirst";
const BO_BE: &'static str = "HighByteFirst";

/// EDF data types.
pub enum DataType {
    I8,
    U8,
    I16,
    U16,
    I32,
    U32,
    I64,
    U64,
    F32,
    F64,
}
#[derive(Template)]
#[template(path = "edf_header.jinja2", escape = "none")]
struct EdfHeaderTemplate<'a> {
    header: &'a Header,
}

#[cfg(test)]
mod tests {
    use crate::edf::Edf;
    use crate::frame::Frame;
    use std::fs::File;
    use std::io;
    use std::io::Read;
    use std::path::Path;

    fn compare(frame: Edf) {
        if frame.dim1 != 64 || frame.dim2 != 64 || frame.size != 64 * 64 {
            panic!(
                "Bad frame dimensions: expected 64x64={}, found {}x{}={}",
                64 * 64,
                frame.dim1,
                frame.dim2,
                frame.size
            );
        }
        let sum = frame.sum();
        if sum != 2073437085. {
            panic!("Expected sum is 2073437085 but found {}", sum);
        }
        let min = frame.min();
        if min != -904. {
            panic!("Expected min is -904 but found {}", min);
        }
        let max = frame.max();
        if max != 999395. {
            panic!("Expected max is 999395 but found {}", max);
        }
    }

    fn read_test_file<P: AsRef<Path>>(path: P) {
        let frame = Edf::read_file(path).unwrap();
        compare(frame);
    }

    #[test]
    fn test_edf_read() {
        read_test_file("testdata/test.edf");
    }

    #[test]
    fn test_edf_writer() {
        let mut frame = Edf::read_file("testdata/test.edf").unwrap();
        let path = tempfile::NamedTempFile::new().unwrap();
        frame.write_file(&path).unwrap();
        read_test_file(&path);
    }

    #[test]
    fn test_edf_buffer_reader() {
        let mut file = File::open("testdata/test.edf").unwrap();
        let mut data = vec![];
        file.read_to_end(&mut data).unwrap();
        let data = io::Cursor::new(data);
        let frame = Edf::read_buffer(data).unwrap();
        compare(frame);
    }

    #[test]
    fn test_edf_tab_first() {
        let testfile = utilities::download_test_file(
            "PBS_100C_SAXS_0199.edf",
            "PBS_100C_SAXS_0199.edf.tar.bz2",
            "955f0a88a12d8df2a02bc6339f50f15a",
        )
        .unwrap();
        let frame = Edf::read_file(testfile).unwrap();
        assert_eq!(frame.sum(), 24911666., "frame sum != 24911666");
    }
}
