//! Pure Rust implementation of the Bruker detector format reader.
//!
//! Usage example:
//! ```rust
//! use cryiorust::bruker::Bruker;
//! use std::path::Path;
//! use cryiorust::frame::Frame;
//!
//! fn read_test_file<P: AsRef<Path>>(path: P) {
//!     let frame = Bruker::read_file(path).unwrap();
//!     if frame.dim1() != 1024 || frame.dim2() != 1024 {
//!         panic!(
//!             "Bad frame dimensions: expected 1024x1024, found {}x{}",
//!             frame.dim1(),
//!             frame.dim2(),
//!         );
//!     }
//!     let sum = frame.sum();
//!     if sum != 21268323. {
//!         panic!("Expected sum is 21268323 but found {}", sum);
//!     }
//!     let min = frame.min();
//!     if min != 0. {
//!         panic!("Expected min is 0 but found {}", min);
//!     }
//!     let max = frame.max();
//!     if max != 6886. {
//!         panic!("Expected max is 6886. but found {}", max);
//!     }
//! }
//! ```
//!
//! Only LittleEndian files are supported.
//!
use crate::frame::FrameError::{FormatError, IoError};
use crate::frame::{Array, Frame, FrameResult, Header, HeaderEntry};
use byteorder::{LittleEndian, ReadBytesExt};
use cryiorust_derive::Frame;
use std::collections::HashMap;
use std::fs::File;
use std::io::Read;
use std::num::ParseFloatError;
use std::path::Path;
use std::{io, mem};

/// Main exported structure.
#[derive(Frame)]
pub struct Bruker {
    header: Header,
    array: Array,
    data_start: usize,
    overflow_start: usize,
    dim1: usize,
    dim2: usize,
}

impl Bruker {
    /// Creates a new empty Bruker structure.
    pub fn new() -> Bruker {
        Bruker {
            header: HashMap::new(),
            array: Array::new(),
            data_start: 0,
            overflow_start: 0,
            dim1: 0,
            dim2: 0,
        }
    }

    /// Reads file as a Bruker frame
    pub fn read_file<P: AsRef<Path>>(path: P) -> FrameResult<Bruker> {
        let mut file = File::open(path)?;
        let size = file.metadata()?.len() as usize;
        let mut data = Vec::with_capacity(size);
        unsafe { data.set_len(size) };
        file.read_exact(&mut data)?;
        let mut frame = Bruker::new();
        frame.read(&data)?;
        Ok(frame)
    }

    fn read(&mut self, data: &[u8]) -> FrameResult<()> {
        self.read_header(data)?;
        self.read_array(data)?;
        self.read_overflow(data)?;
        self.check_linearity();
        Ok(())
    }

    fn read_header_chunks(&mut self, start: usize, end: usize, data: &[u8]) -> FrameResult<()> {
        for i in (start..end).step_by(HEADER_CHUNK_SIZE) {
            let j = i + HEADER_CHUNK_SIZE;
            if j >= data.len() {
                return Err(IoError(io::Error::new(
                    io::ErrorKind::UnexpectedEof,
                    "not enough data in buffer",
                )));
            }
            self.fill_header(&data[i..j])
        }
        Ok(())
    }

    fn read_header(&mut self, data: &[u8]) -> FrameResult<()> {
        self.read_header_chunks(0, N_BLOCKS, data)?;
        if let Ok(v) = self.get_header_str_as_i64(KEY_HEADER_BLOCKS) {
            self.data_start = v as usize;
        } else {
            self.data_start = N_HEADER_BLOCKS
        };
        self.data_start *= BLOCK_SIZE;
        self.read_header_chunks(N_BLOCKS, self.data_start, data)?;
        if let Ok(dim1) = self.get_header_str_as_i64(KEY_N_ROWS) {
            if let Ok(dim2) = self.get_header_str_as_i64(KEY_N_COLS) {
                if dim1 == 0 || dim2 == 0 {
                    return Err(FormatError("zero dimensions in the header".into()));
                }
                self.dim1 = dim1 as usize;
                self.dim2 = dim2 as usize;
            } else {
                return Err(FormatError("could not find dim2 in the header".into()));
            }
        } else {
            return Err(FormatError("could not find dim1 in the header".into()));
        };
        if let Ok(v) = self.get_header_str_as_i64(KEY_VERSION) {
            if v != VERSION {
                return Err(FormatError(
                    format!("version {} is not yet supported", v).into(),
                ));
            }
        }
        Ok(())
    }

    fn fill_header(&mut self, chunk: &[u8]) {
        if chunk.is_empty() {
            return;
        }
        let chunk = unsafe { std::str::from_utf8_unchecked(chunk) };
        let parts: Vec<&str> = chunk.split(':').map(|x| x.trim()).collect();
        if parts.len() < 2 {
            return;
        }
        self.header
            .entry(parts[0].to_string())
            .or_insert(HeaderEntry::String(parts[1].to_string()));
    }

    fn read_array(&mut self, data: &[u8]) -> FrameResult<()> {
        let npixelb;
        if let Ok(v) = self.get_header_str_as_i64(KEY_N_PIXELB) {
            npixelb = v as usize;
        } else {
            return Err(FormatError(
                format!("could not find {} in the header", KEY_N_PIXELB).into(),
            ));
        }
        if self.data_start >= data.len() {
            return Err(FormatError("not enough binary data".into()));
        }
        let array = &data[self.data_start..];
        self.array = match npixelb {
            SIZE_U8 => Array::from_slice_u8(self.dim1, self.dim2, array)?,
            SIZE_U16 => Array::from_slice_u16::<LittleEndian>(self.dim1, self.dim2, array)?,
            SIZE_U32 => Array::from_slice_u32::<LittleEndian>(self.dim1, self.dim2, array)?,
            _ => {
                return Err(FormatError(
                    format!("bad value for {} = {} in the header", KEY_N_PIXELB, npixelb).into(),
                ))
            }
        };
        self.overflow_start = self.array.len() * npixelb + self.data_start;
        Ok(())
    }

    fn check_linearity(&mut self) {
        if let Ok(s) = self.get_header_str(KEY_LINEAR) {
            let parts: Vec<Result<f64, ParseFloatError>> = s
                .split_ascii_whitespace()
                .map(|x| x.trim().parse::<f64>())
                .collect();
            if parts.len() < 2 {
                return;
            }
            if let Ok(slope) = parts[0] {
                if let Ok(offset) = parts[1] {
                    if slope != 1.0 || offset != 0.0 {
                        for value in self.array.data_mut() {
                            *value = *value * slope + offset;
                        }
                    }
                }
            }
        }
    }

    fn read_overflow(&mut self, data: &[u8]) -> FrameResult<()> {
        let mut n;
        if let Ok(v) = self.get_header_str_as_i64(KEY_N_OVERFLOW) {
            n = v as usize;
        } else {
            return Err(FormatError(
                format!("bad value for {} in the header", KEY_N_OVERFLOW).into(),
            ));
        }
        let mut reader = io::Cursor::new(&data[self.overflow_start..]);
        while n > 0 {
            let intensity = reader.read_u64::<LittleEndian>()? as f64;
            reader.read_u8()?;
            let position = reader.read_u32::<LittleEndian>()? as usize;
            if position >= self.array.len() {
                return Err(FormatError(
                    format!("bad overflow position: {}", position).into(),
                ));
            }
            reader.read_u24::<LittleEndian>()?;
            self.array[position] = intensity;
            n -= 1;
        }
        Ok(())
    }
}

const BLOCK_SIZE: usize = 512;
const N_HEADER_BLOCKS: usize = 5;
const VERSION: i64 = 8;
const HEADER_CHUNK_SIZE: usize = 80;
const N_BLOCKS: usize = BLOCK_SIZE * N_HEADER_BLOCKS;
const KEY_HEADER_BLOCKS: &'static str = "HDRBLKS";
const KEY_N_ROWS: &'static str = "NROWS";
const KEY_N_COLS: &'static str = "NCOLS";
const KEY_N_PIXELB: &'static str = "NPIXELB";
const KEY_N_OVERFLOW: &'static str = "NOVERFL";
const KEY_LINEAR: &'static str = "LINEAR";
const KEY_VERSION: &'static str = "VERSION";
const SIZE_U8: usize = mem::size_of::<u8>();
const SIZE_U16: usize = mem::size_of::<u16>();
const SIZE_U32: usize = mem::size_of::<u32>();

#[cfg(test)]
mod tests {
    use crate::bruker::Bruker;
    use crate::frame::Frame;
    use std::path::Path;

    fn read_test_file<P: AsRef<Path>>(path: P) {
        let frame = Bruker::read_file(path).unwrap();
        if frame.dim1() != 1024 || frame.dim2() != 1024 {
            panic!(
                "Bad frame dimensions: expected 1024x1024, found {}x{}",
                frame.dim1(),
                frame.dim2(),
            );
        }
        let sum = frame.sum();
        if sum != 21268323. {
            panic!("Expected sum is 21268323 but found {}", sum);
        }
        let min = frame.min();
        if min != 0. {
            panic!("Expected min is 0 but found {}", min);
        }
        let max = frame.max();
        if max != 6886. {
            panic!("Expected max is 6886. but found {}", max);
        }
    }

    #[test]
    fn test_bruker_read() {
        read_test_file("testdata/test.gfrm");
    }
}
