#[macro_use]
extern crate criterion;
use criterion::{black_box, Benchmark, Criterion};
use cryiorust::bruker::Bruker;
use cryiorust::cbf::Cbf;
use cryiorust::edf::Edf;
use cryiorust::eiger::Eiger;
use cryiorust::esperanto::Esperanto;
use cryiorust::frame::Frame;
use cryiorust::mar::Mar;
use cryiorust::tif::Tif;

extern crate utilities;

fn criterion_benchmark(c: &mut Criterion) {
    c.bench_function("CBF reader", |b| {
        b.iter(|| Cbf::read_file(black_box("testdata/lab6.cbf")))
    });
    let mut cbf = Cbf::read_file("testdata/lab6.cbf").unwrap();
    c.bench_function("CBF writer", move |b| {
        b.iter(|| cbf.write_file(black_box(tempfile::NamedTempFile::new().unwrap())))
    });
    c.bench_function("EDF reader", |b| {
        b.iter(|| Edf::read_file(black_box("testdata/test.edf")))
    });
    let mut edf = Edf::read_file("testdata/test.edf").unwrap();
    c.bench_function("EDF writer", move |b| {
        b.iter(|| edf.write_file(black_box(tempfile::NamedTempFile::new().unwrap())))
    });
    c.bench_function("Bruker reader", |b| {
        b.iter(|| Bruker::read_file(black_box("testdata/test.gfrm")))
    });
    c.bench_function("Mar reader", |b| {
        b.iter(|| Mar::read_file(black_box("testdata/test.mar3450")))
    });
    c.bench(
        "Mar real reader",
        Benchmark::new("Mar real reader", move |b| {
            let testfile = utilities::download_test_file(
                "LR104_000.mar3450",
                "LR104_000.mar3450.tar.bz2",
                "85ce193ed808f2a2ff95f992691f33cc",
            )
            .unwrap();
            b.iter(|| Mar::read_file(black_box(&testfile)))
        })
        .sample_size(10),
    );
    c.bench(
        "Tif",
        Benchmark::new("Reader", move |b| {
            let testfile = utilities::download_test_file(
                "lab6.tif",
                "lab6.tif.tar.bz2",
                "49c454bf346439642324406c2a347a9b",
            )
            .unwrap();
            b.iter(|| Tif::read_file(black_box(&testfile)))
        })
        .sample_size(10),
    );
    c.bench(
        "Tif i32",
        Benchmark::new("Reader", move |b| {
            let testfile = utilities::download_test_file(
                "tiff_i32.tif",
                "tiff_i32.tar.bz2",
                "e0d7c21f026332d459bbb04aad2847c2",
            )
            .unwrap();
            b.iter(|| Tif::read_file(black_box(&testfile)))
        })
        .sample_size(10),
    );
    c.bench_function("Esperanto reader", |b| {
        b.iter(|| Esperanto::read_file(black_box("testdata/sucros.esperanto")))
    });
    c.bench_function("Esperanto writer", |b| {
        let esp = Esperanto::read_file("testdata/sucros.esperanto").unwrap();
        b.iter(|| esp.write_file(black_box(tempfile::NamedTempFile::new().unwrap())))
    });
    c.bench(
        "Eiger",
        Benchmark::new("Reader", move |b| {
            let testfile = utilities::download_test_file(
                "200Hz_0.1dpf_1_data_000001.h5",
                "eiger.tar.bz2",
                "9614362dae905499f32c095c6201776f",
            )
            .unwrap();
            b.iter(|| Eiger::read_file(black_box(&testfile)))
        })
        .sample_size(10),
    );
    c.bench(
        "Eiger frame switcher",
        Benchmark::new("Switch frame", move |b| {
            let testfile = utilities::download_test_file(
                "200Hz_0.1dpf_1_data_000001.h5",
                "eiger.tar.bz2",
                "9614362dae905499f32c095c6201776f",
            )
            .unwrap();
            let mut frame = Eiger::read_file(black_box(&testfile)).unwrap();
            b.iter(|| frame.set_frame(5).unwrap())
        })
        .sample_size(10),
    );
}

criterion_group!(benches, criterion_benchmark);
criterion_main!(benches);
