#include <cry/ecp.h>
#include <cry/config.h>

#if CRY_MPI_DIGIT_MAX == 255UL

# define BYTES_TO_DIGITS_2(a, b) \
    a, b

# define BYTES_TO_DIGITS_4(a, b, c, d) \
    BYTES_TO_DIGITS_2(a, b), \
    BYTES_TO_DIGITS_2(c, d)

# define BYTES_TO_DIGITS_8(a, b, c, d, e, f, g, h) \
    BYTES_TO_DIGITS_4(a, b, c, d), \
    BYTES_TO_DIGITS_4(e, f, g, h)

#elif CRY_MPI_DIGIT_MAX == 65535UL

# define BYTES_TO_DIGITS_W(a, b) \
    ((cry_mpi_digit)a << 0) | ((cry_mpi_digit)b << 8)

# define BYTES_TO_DIGITS_2(a, b) \
    BYTES_TO_DIGITS_W(a, b)

# define BYTES_TO_DIGITS_4(a, b, c, d) \
    BYTES_TO_DIGITS_2(a, b), \
    BYTES_TO_DIGITS_2(c, d)

# define BYTES_TO_DIGITS_8(a, b, c, d, e, f, g, h) \
    BYTES_TO_DIGITS_4(a, b, c, d), \
    BYTES_TO_DIGITS_4(e, f, g, h)

#elif CRY_MPI_DIGIT_MAX == 4294967295UL

# define BYTES_TO_DIGITS_W(a, b, c, d) \
    ((cry_mpi_digit)a <<  0) | ((cry_mpi_digit)b <<  8) | \
    ((cry_mpi_digit)c << 16) | ((cry_mpi_digit)d << 24)

# define BYTES_TO_DIGITS_2(a, b) \
    BYTES_TO_DIGITS_W(a, b, 0, 0)

# define BYTES_TO_DIGITS_4(a, b, c, d) \
    BYTES_TO_DIGITS_W(a, b, c, d)

# define BYTES_TO_DIGITS_8(a, b, c, d, e, f, g, h) \
    BYTES_TO_DIGITS_4(a, b, c, d), \
    BYTES_TO_DIGITS_4(e, f, g, h)

#elif CRY_MPI_DIGIT_MAX == 18446744073709551615UL

# define BYTES_TO_DIGITS_W(a, b, c, d, e, f, g, h) \
    ((cry_mpi_digit)a <<  0) | ((cry_mpi_digit)b <<  8) | \
    ((cry_mpi_digit)c << 16) | ((cry_mpi_digit)d << 24) | \
    ((cry_mpi_digit)e << 32) | ((cry_mpi_digit)f << 40) | \
    ((cry_mpi_digit)g << 48) | ((cry_mpi_digit)h << 56)

# define BYTES_TO_DIGITS_2(a, b) \
    BYTES_TO_DIGITS_W(a, b, 0, 0, 0, 0, 0, 0)

# define BYTES_TO_DIGITS_4(a, b, c, d) \
    BYTES_TO_DIGITS_W(a, b, c, d, 0, 0, 0, 0)

# define BYTES_TO_DIGITS_8(a, b, c, d, e, f, g, h) \
    BYTES_TO_DIGITS_W(a, b, c, d, e, f, g, h)

#else

# error "Invalid DIGIT_MAX value"

#endif


/*
 * Domain parameters for secp192r1
 */

#ifdef CRY_ECP_GRP_SECP192R1_ENABLE

static const cry_mpi_digit secp192r1_p[] = {
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
};

#define secp192r1_a     NULL

static const cry_mpi_digit secp192r1_b[] = {
    BYTES_TO_DIGITS_8(0xB1, 0xB9, 0x46, 0xC1, 0xEC, 0xDE, 0xB8, 0xFE),
    BYTES_TO_DIGITS_8(0x49, 0x30, 0x24, 0x72, 0xAB, 0xE9, 0xA7, 0x0F),
    BYTES_TO_DIGITS_8(0xE7, 0x80, 0x9C, 0xE5, 0x19, 0x05, 0x21, 0x64),
};
static const cry_mpi_digit secp192r1_gx[] = {
    BYTES_TO_DIGITS_8(0x12, 0x10, 0xFF, 0x82, 0xFD, 0x0A, 0xFF, 0xF4),
    BYTES_TO_DIGITS_8(0x00, 0x88, 0xA1, 0x43, 0xEB, 0x20, 0xBF, 0x7C),
    BYTES_TO_DIGITS_8(0xF6, 0x90, 0x30, 0xB0, 0x0E, 0xA8, 0x8D, 0x18),
};
static const cry_mpi_digit secp192r1_gy[] = {
    BYTES_TO_DIGITS_8(0x11, 0x48, 0x79, 0x1E, 0xA1, 0x77, 0xF9, 0x73),
    BYTES_TO_DIGITS_8(0xD5, 0xCD, 0x24, 0x6B, 0xED, 0x11, 0x10, 0x63),
    BYTES_TO_DIGITS_8(0x78, 0xDA, 0xC8, 0xFF, 0x95, 0x2B, 0x19, 0x07),
};
static const cry_mpi_digit secp192r1_n[] = {
    BYTES_TO_DIGITS_8(0x31, 0x28, 0xD2, 0xB4, 0xB1, 0xC9, 0x6B, 0x14),
    BYTES_TO_DIGITS_8(0x36, 0xF8, 0xDE, 0x99, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
};

#endif /* ~CRY_ECP_GRP_SECP192R1_ENABLE */


/*
 * Domain parameters for secp224r1
 */

#ifdef CRY_ECP_GRP_SECP224R1_ENABLE

static const cry_mpi_digit secp224r1_p[] = {
    BYTES_TO_DIGITS_8(0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00),
    BYTES_TO_DIGITS_8(0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_4(0xFF, 0xFF, 0xFF, 0xFF),
};

#define secp224r1_a     NULL

static const cry_mpi_digit secp224r1_b[] = {
    BYTES_TO_DIGITS_8(0xB4, 0xFF, 0x55, 0x23, 0x43, 0x39, 0x0B, 0x27),
    BYTES_TO_DIGITS_8(0xBA, 0xD8, 0xBF, 0xD7, 0xB7, 0xB0, 0x44, 0x50),
    BYTES_TO_DIGITS_8(0x56, 0x32, 0x41, 0xF5, 0xAB, 0xB3, 0x04, 0x0C),
    BYTES_TO_DIGITS_4(0x85, 0x0A, 0x05, 0xB4),
};
static const cry_mpi_digit secp224r1_gx[] = {
    BYTES_TO_DIGITS_8(0x21, 0x1D, 0x5C, 0x11, 0xD6, 0x80, 0x32, 0x34),
    BYTES_TO_DIGITS_8(0x22, 0x11, 0xC2, 0x56, 0xD3, 0xC1, 0x03, 0x4A),
    BYTES_TO_DIGITS_8(0xB9, 0x90, 0x13, 0x32, 0x7F, 0xBF, 0xB4, 0x6B),
    BYTES_TO_DIGITS_4(0xBD, 0x0C, 0x0E, 0xB7),
};
static const cry_mpi_digit secp224r1_gy[] = {
    BYTES_TO_DIGITS_8(0x34, 0x7E, 0x00, 0x85, 0x99, 0x81, 0xD5, 0x44),
    BYTES_TO_DIGITS_8(0x64, 0x47, 0x07, 0x5A, 0xA0, 0x75, 0x43, 0xCD),
    BYTES_TO_DIGITS_8(0xE6, 0xDF, 0x22, 0x4C, 0xFB, 0x23, 0xF7, 0xB5),
    BYTES_TO_DIGITS_4(0x88, 0x63, 0x37, 0xBD),
};
static const cry_mpi_digit secp224r1_n[] = {
    BYTES_TO_DIGITS_8(0x3D, 0x2A, 0x5C, 0x5C, 0x45, 0x29, 0xDD, 0x13),
    BYTES_TO_DIGITS_8(0x3E, 0xF0, 0xB8, 0xE0, 0xA2, 0x16, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_4(0xFF, 0xFF, 0xFF, 0xFF),
};

#endif /* ~CRY_ECP_GRP_SECP224R1_ENABLE */


/*
 * Domain parameters for secp256r1
 */

#ifdef CRY_ECP_GRP_SECP256R1_ENABLE

static const cry_mpi_digit secp256r1_p[] = {
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00),
    BYTES_TO_DIGITS_8(0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00),
    BYTES_TO_DIGITS_8(0x01, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF),
};

#define secp256r1_a     NULL

static const cry_mpi_digit secp256r1_b[] = {
    BYTES_TO_DIGITS_8(0x4B, 0x60, 0xD2, 0x27, 0x3E, 0x3C, 0xCE, 0x3B),
    BYTES_TO_DIGITS_8(0xF6, 0xB0, 0x53, 0xCC, 0xB0, 0x06, 0x1D, 0x65),
    BYTES_TO_DIGITS_8(0xBC, 0x86, 0x98, 0x76, 0x55, 0xBD, 0xEB, 0xB3),
    BYTES_TO_DIGITS_8(0xE7, 0x93, 0x3A, 0xAA, 0xD8, 0x35, 0xC6, 0x5A),
};
static const cry_mpi_digit secp256r1_gx[] = {
    BYTES_TO_DIGITS_8(0x96, 0xC2, 0x98, 0xD8, 0x45, 0x39, 0xA1, 0xF4),
    BYTES_TO_DIGITS_8(0xA0, 0x33, 0xEB, 0x2D, 0x81, 0x7D, 0x03, 0x77),
    BYTES_TO_DIGITS_8(0xF2, 0x40, 0xA4, 0x63, 0xE5, 0xE6, 0xBC, 0xF8),
    BYTES_TO_DIGITS_8(0x47, 0x42, 0x2C, 0xE1, 0xF2, 0xD1, 0x17, 0x6B),
};
static const cry_mpi_digit secp256r1_gy[] = {
    BYTES_TO_DIGITS_8(0xF5, 0x51, 0xBF, 0x37, 0x68, 0x40, 0xB6, 0xCB),
    BYTES_TO_DIGITS_8(0xCE, 0x5E, 0x31, 0x6B, 0x57, 0x33, 0xCE, 0x2B),
    BYTES_TO_DIGITS_8(0x16, 0x9E, 0x0F, 0x7C, 0x4A, 0xEB, 0xE7, 0x8E),
    BYTES_TO_DIGITS_8(0x9B, 0x7F, 0x1A, 0xFE, 0xE2, 0x42, 0xE3, 0x4F),
};
static const cry_mpi_digit secp256r1_n[] = {
    BYTES_TO_DIGITS_8(0x51, 0x25, 0x63, 0xFC, 0xC2, 0xCA, 0xB9, 0xF3),
    BYTES_TO_DIGITS_8(0x84, 0x9E, 0x17, 0xA7, 0xAD, 0xFA, 0xE6, 0xBC),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF),
};

#endif /* ~CRY_ECP_GRP_SECP256R1_ENABLE */


/*
 * Domain parameters for secp384r1
 */

#ifdef CRY_ECP_GRP_SECP384R1_ENABLE

static const cry_mpi_digit secp384r1_p[] = {
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00),
    BYTES_TO_DIGITS_8(0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
};

#define secp384r1_a     NULL

static const cry_mpi_digit secp384r1_b[] = {
    BYTES_TO_DIGITS_8(0xEF, 0x2A, 0xEC, 0xD3, 0xED, 0xC8, 0x85, 0x2A),
    BYTES_TO_DIGITS_8(0x9D, 0xD1, 0x2E, 0x8A, 0x8D, 0x39, 0x56, 0xC6),
    BYTES_TO_DIGITS_8(0x5A, 0x87, 0x13, 0x50, 0x8F, 0x08, 0x14, 0x03),
    BYTES_TO_DIGITS_8(0x12, 0x41, 0x81, 0xFE, 0x6E, 0x9C, 0x1D, 0x18),
    BYTES_TO_DIGITS_8(0x19, 0x2D, 0xF8, 0xE3, 0x6B, 0x05, 0x8E, 0x98),
    BYTES_TO_DIGITS_8(0xE4, 0xE7, 0x3E, 0xE2, 0xA7, 0x2F, 0x31, 0xB3),
};
static const cry_mpi_digit secp384r1_gx[] = {
    BYTES_TO_DIGITS_8(0xB7, 0x0A, 0x76, 0x72, 0x38, 0x5E, 0x54, 0x3A),
    BYTES_TO_DIGITS_8(0x6C, 0x29, 0x55, 0xBF, 0x5D, 0xF2, 0x02, 0x55),
    BYTES_TO_DIGITS_8(0x38, 0x2A, 0x54, 0x82, 0xE0, 0x41, 0xF7, 0x59),
    BYTES_TO_DIGITS_8(0x98, 0x9B, 0xA7, 0x8B, 0x62, 0x3B, 0x1D, 0x6E),
    BYTES_TO_DIGITS_8(0x74, 0xAD, 0x20, 0xF3, 0x1E, 0xC7, 0xB1, 0x8E),
    BYTES_TO_DIGITS_8(0x37, 0x05, 0x8B, 0xBE, 0x22, 0xCA, 0x87, 0xAA),
};
static const cry_mpi_digit secp384r1_gy[] = {
    BYTES_TO_DIGITS_8(0x5F, 0x0E, 0xEA, 0x90, 0x7C, 0x1D, 0x43, 0x7A),
    BYTES_TO_DIGITS_8(0x9D, 0x81, 0x7E, 0x1D, 0xCE, 0xB1, 0x60, 0x0A),
    BYTES_TO_DIGITS_8(0xC0, 0xB8, 0xF0, 0xB5, 0x13, 0x31, 0xDA, 0xE9),
    BYTES_TO_DIGITS_8(0x7C, 0x14, 0x9A, 0x28, 0xBD, 0x1D, 0xF4, 0xF8),
    BYTES_TO_DIGITS_8(0x29, 0xDC, 0x92, 0x92, 0xBF, 0x98, 0x9E, 0x5D),
    BYTES_TO_DIGITS_8(0x6F, 0x2C, 0x26, 0x96, 0x4A, 0xDE, 0x17, 0x36),
};
static const cry_mpi_digit secp384r1_n[] = {
    BYTES_TO_DIGITS_8(0x73, 0x29, 0xC5, 0xCC, 0x6A, 0x19, 0xEC, 0xEC),
    BYTES_TO_DIGITS_8(0x7A, 0xA7, 0xB0, 0x48, 0xB2, 0x0D, 0x1A, 0x58),
    BYTES_TO_DIGITS_8(0xDF, 0x2D, 0x37, 0xF4, 0x81, 0x4D, 0x63, 0xC7),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
};

#endif /* ~CRY_ECP_GRP_SECP384R1_ENABLE */


/*
 * Domain parameters for secp521r1
 */

#ifdef CRY_ECP_GRP_SECP521R1_ENABLE

static const cry_mpi_digit secp521r1_p[] = {
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_2(0xFF, 0x01),
};

#define secp521r1_a     NULL

static const cry_mpi_digit secp521r1_b[] = {
    BYTES_TO_DIGITS_8(0x00, 0x3F, 0x50, 0x6B, 0xD4, 0x1F, 0x45, 0xEF),
    BYTES_TO_DIGITS_8(0xF1, 0x34, 0x2C, 0x3D, 0x88, 0xDF, 0x73, 0x35),
    BYTES_TO_DIGITS_8(0x07, 0xBF, 0xB1, 0x3B, 0xBD, 0xC0, 0x52, 0x16),
    BYTES_TO_DIGITS_8(0x7B, 0x93, 0x7E, 0xEC, 0x51, 0x39, 0x19, 0x56),
    BYTES_TO_DIGITS_8(0xE1, 0x09, 0xF1, 0x8E, 0x91, 0x89, 0xB4, 0xB8),
    BYTES_TO_DIGITS_8(0xF3, 0x15, 0xB3, 0x99, 0x5B, 0x72, 0xDA, 0xA2),
    BYTES_TO_DIGITS_8(0xEE, 0x40, 0x85, 0xB6, 0xA0, 0x21, 0x9A, 0x92),
    BYTES_TO_DIGITS_8(0x1F, 0x9A, 0x1C, 0x8E, 0x61, 0xB9, 0x3E, 0x95),
    BYTES_TO_DIGITS_2(0x51, 0x00),
};
static const cry_mpi_digit secp521r1_gx[] = {
    BYTES_TO_DIGITS_8(0x66, 0xBD, 0xE5, 0xC2, 0x31, 0x7E, 0x7E, 0xF9),
    BYTES_TO_DIGITS_8(0x9B, 0x42, 0x6A, 0x85, 0xC1, 0xB3, 0x48, 0x33),
    BYTES_TO_DIGITS_8(0xDE, 0xA8, 0xFF, 0xA2, 0x27, 0xC1, 0x1D, 0xFE),
    BYTES_TO_DIGITS_8(0x28, 0x59, 0xE7, 0xEF, 0x77, 0x5E, 0x4B, 0xA1),
    BYTES_TO_DIGITS_8(0xBA, 0x3D, 0x4D, 0x6B, 0x60, 0xAF, 0x28, 0xF8),
    BYTES_TO_DIGITS_8(0x21, 0xB5, 0x3F, 0x05, 0x39, 0x81, 0x64, 0x9C),
    BYTES_TO_DIGITS_8(0x42, 0xB4, 0x95, 0x23, 0x66, 0xCB, 0x3E, 0x9E),
    BYTES_TO_DIGITS_8(0xCD, 0xE9, 0x04, 0x04, 0xB7, 0x06, 0x8E, 0x85),
    BYTES_TO_DIGITS_2(0xC6, 0x00),
};
static const cry_mpi_digit secp521r1_gy[] = {
    BYTES_TO_DIGITS_8(0x50, 0x66, 0xD1, 0x9F, 0x76, 0x94, 0xBE, 0x88),
    BYTES_TO_DIGITS_8(0x40, 0xC2, 0x72, 0xA2, 0x86, 0x70, 0x3C, 0x35),
    BYTES_TO_DIGITS_8(0x61, 0x07, 0xAD, 0x3F, 0x01, 0xB9, 0x50, 0xC5),
    BYTES_TO_DIGITS_8(0x40, 0x26, 0xF4, 0x5E, 0x99, 0x72, 0xEE, 0x97),
    BYTES_TO_DIGITS_8(0x2C, 0x66, 0x3E, 0x27, 0x17, 0xBD, 0xAF, 0x17),
    BYTES_TO_DIGITS_8(0x68, 0x44, 0x9B, 0x57, 0x49, 0x44, 0xF5, 0x98),
    BYTES_TO_DIGITS_8(0xD9, 0x1B, 0x7D, 0x2C, 0xB4, 0x5F, 0x8A, 0x5C),
    BYTES_TO_DIGITS_8(0x04, 0xC0, 0x3B, 0x9A, 0x78, 0x6A, 0x29, 0x39),
    BYTES_TO_DIGITS_2(0x18, 0x01),
};
static const cry_mpi_digit secp521r1_n[] = {
    BYTES_TO_DIGITS_8(0x09, 0x64, 0x38, 0x91, 0x1E, 0xB7, 0x6F, 0xBB),
    BYTES_TO_DIGITS_8(0xAE, 0x47, 0x9C, 0x89, 0xB8, 0xC9, 0xB5, 0x3B),
    BYTES_TO_DIGITS_8(0xD0, 0xA5, 0x09, 0xF7, 0x48, 0x01, 0xCC, 0x7F),
    BYTES_TO_DIGITS_8(0x6B, 0x96, 0x2F, 0xBF, 0x83, 0x87, 0x86, 0x51),
    BYTES_TO_DIGITS_8(0xFA, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_2(0xFF, 0x01),
};

#endif /* ~CRY_ECP_GRP_SECP521R1_ENABLE */


/*
 * Domain parameters for secp192k1
 */

#ifdef CRY_ECP_GRP_SECP192K1_ENABLE

static const cry_mpi_digit secp192k1_p[] = {
    BYTES_TO_DIGITS_8(0x37, 0xEE, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
};
static const cry_mpi_digit secp192k1_a[] = {
    BYTES_TO_DIGITS_2(0x00, 0x00),
};
static const cry_mpi_digit secp192k1_b[] = {
    BYTES_TO_DIGITS_2(0x03, 0x00),
};
static const cry_mpi_digit secp192k1_gx[] = {
    BYTES_TO_DIGITS_8(0x7D, 0x6C, 0xE0, 0xEA, 0xB1, 0xD1, 0xA5, 0x1D),
    BYTES_TO_DIGITS_8(0x34, 0xF4, 0xB7, 0x80, 0x02, 0x7D, 0xB0, 0x26),
    BYTES_TO_DIGITS_8(0xAE, 0xE9, 0x57, 0xC0, 0x0E, 0xF1, 0x4F, 0xDB),
};
static const cry_mpi_digit secp192k1_gy[] = {
    BYTES_TO_DIGITS_8(0x9D, 0x2F, 0x5E, 0xD9, 0x88, 0xAA, 0x82, 0x40),
    BYTES_TO_DIGITS_8(0x34, 0x86, 0xBE, 0x15, 0xD0, 0x63, 0x41, 0x84),
    BYTES_TO_DIGITS_8(0xA7, 0x28, 0x56, 0x9C, 0x6D, 0x2F, 0x2F, 0x9B),
};
static const cry_mpi_digit secp192k1_n[] = {
    BYTES_TO_DIGITS_8(0x8D, 0xFD, 0xDE, 0x74, 0x6A, 0x46, 0x69, 0x0F),
    BYTES_TO_DIGITS_8(0x17, 0xFC, 0xF2, 0x26, 0xFE, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
};

#endif /* ~CRY_ECP_GRP_SECP192K1_ENABLE */


/*
 * Domain parameters for secp224k1
 */

#ifdef CRY_ECP_GRP_SECP224K1_ENABLE

static const cry_mpi_digit secp224k1_p[] = {
    BYTES_TO_DIGITS_8(0x6D, 0xE5, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_4(0xFF, 0xFF, 0xFF, 0xFF),
};
static const cry_mpi_digit secp224k1_a[] = {
    BYTES_TO_DIGITS_2(0x00, 0x00),
};
static const cry_mpi_digit secp224k1_b[] = {
    BYTES_TO_DIGITS_2(0x05, 0x00),
};
static const cry_mpi_digit secp224k1_gx[] = {
    BYTES_TO_DIGITS_8(0x5C, 0xA4, 0xB7, 0xB6, 0x0E, 0x65, 0x7E, 0x0F),
    BYTES_TO_DIGITS_8(0xA9, 0x75, 0x70, 0xE4, 0xE9, 0x67, 0xA4, 0x69),
    BYTES_TO_DIGITS_8(0xA1, 0x28, 0xFC, 0x30, 0xDF, 0x99, 0xF0, 0x4D),
    BYTES_TO_DIGITS_4(0x33, 0x5B, 0x45, 0xA1),
};
static const cry_mpi_digit secp224k1_gy[] = {
    BYTES_TO_DIGITS_8(0xA5, 0x61, 0x6D, 0x55, 0xDB, 0x4B, 0xCA, 0xE2),
    BYTES_TO_DIGITS_8(0x59, 0xBD, 0xB0, 0xC0, 0xF7, 0x19, 0xE3, 0xF7),
    BYTES_TO_DIGITS_8(0xD6, 0xFB, 0xCA, 0x82, 0x42, 0x34, 0xBA, 0x7F),
    BYTES_TO_DIGITS_4(0xED, 0x9F, 0x08, 0x7E),
};
static const cry_mpi_digit secp224k1_n[] = {
    BYTES_TO_DIGITS_8(0xF7, 0xB1, 0x9F, 0x76, 0x71, 0xA9, 0xF0, 0xCA),
    BYTES_TO_DIGITS_8(0x84, 0x61, 0xEC, 0xD2, 0xE8, 0xDC, 0x01, 0x00),
    BYTES_TO_DIGITS_8(0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00),
    BYTES_TO_DIGITS_8(0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00),
};

#endif /* ~CRY_ECP_GRP_SECP224K1_ENABLE */


/*
 * Domain parameters for secp256k1
 */

#ifdef CRY_ECP_GRP_SECP256K1_ENABLE

static const cry_mpi_digit secp256k1_p[] = {
    BYTES_TO_DIGITS_8(0x2F, 0xFC, 0xFF, 0xFF, 0xFE, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
};
static const cry_mpi_digit secp256k1_a[] = {
    BYTES_TO_DIGITS_2(0x00, 0x00),
};
static const cry_mpi_digit secp256k1_b[] = {
    BYTES_TO_DIGITS_2(0x07, 0x00),
};
static const cry_mpi_digit secp256k1_gx[] = {
    BYTES_TO_DIGITS_8(0x98, 0x17, 0xF8, 0x16, 0x5B, 0x81, 0xF2, 0x59),
    BYTES_TO_DIGITS_8(0xD9, 0x28, 0xCE, 0x2D, 0xDB, 0xFC, 0x9B, 0x02),
    BYTES_TO_DIGITS_8(0x07, 0x0B, 0x87, 0xCE, 0x95, 0x62, 0xA0, 0x55),
    BYTES_TO_DIGITS_8(0xAC, 0xBB, 0xDC, 0xF9, 0x7E, 0x66, 0xBE, 0x79),
};
static const cry_mpi_digit secp256k1_gy[] = {
    BYTES_TO_DIGITS_8(0xB8, 0xD4, 0x10, 0xFB, 0x8F, 0xD0, 0x47, 0x9C),
    BYTES_TO_DIGITS_8(0x19, 0x54, 0x85, 0xA6, 0x48, 0xB4, 0x17, 0xFD),
    BYTES_TO_DIGITS_8(0xA8, 0x08, 0x11, 0x0E, 0xFC, 0xFB, 0xA4, 0x5D),
    BYTES_TO_DIGITS_8(0x65, 0xC4, 0xA3, 0x26, 0x77, 0xDA, 0x3A, 0x48),
};
static const cry_mpi_digit secp256k1_n[] = {
    BYTES_TO_DIGITS_8(0x41, 0x41, 0x36, 0xD0, 0x8C, 0x5E, 0xD2, 0xBF),
    BYTES_TO_DIGITS_8(0x3B, 0xA0, 0x48, 0xAF, 0xE6, 0xDC, 0xAE, 0xBA),
    BYTES_TO_DIGITS_8(0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
    BYTES_TO_DIGITS_8(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF),
};

#endif /* ~CRY_ECP_GRP_SECP256K1_ENABLE */


/*
 * Domain parameters for brainpoolP256r1 (RFC 5639 3.4)
 */

#ifdef CRY_ECP_GRP_BP256R1_ENABLE

static const cry_mpi_digit brainpoolP256r1_p[] = {
    BYTES_TO_DIGITS_8(0x77, 0x53, 0x6E, 0x1F, 0x1D, 0x48, 0x13, 0x20),
    BYTES_TO_DIGITS_8(0x28, 0x20, 0x26, 0xD5, 0x23, 0xF6, 0x3B, 0x6E),
    BYTES_TO_DIGITS_8(0x72, 0x8D, 0x83, 0x9D, 0x90, 0x0A, 0x66, 0x3E),
    BYTES_TO_DIGITS_8(0xBC, 0xA9, 0xEE, 0xA1, 0xDB, 0x57, 0xFB, 0xA9),
};
static const cry_mpi_digit brainpoolP256r1_a[] = {
    BYTES_TO_DIGITS_8(0xD9, 0xB5, 0x30, 0xF3, 0x44, 0x4B, 0x4A, 0xE9),
    BYTES_TO_DIGITS_8(0x6C, 0x5C, 0xDC, 0x26, 0xC1, 0x55, 0x80, 0xFB),
    BYTES_TO_DIGITS_8(0xE7, 0xFF, 0x7A, 0x41, 0x30, 0x75, 0xF6, 0xEE),
    BYTES_TO_DIGITS_8(0x57, 0x30, 0x2C, 0xFC, 0x75, 0x09, 0x5A, 0x7D),
};
static const cry_mpi_digit brainpoolP256r1_b[] = {
    BYTES_TO_DIGITS_8(0xB6, 0x07, 0x8C, 0xFF, 0x18, 0xDC, 0xCC, 0x6B),
    BYTES_TO_DIGITS_8(0xCE, 0xE1, 0xF7, 0x5C, 0x29, 0x16, 0x84, 0x95),
    BYTES_TO_DIGITS_8(0xBF, 0x7C, 0xD7, 0xBB, 0xD9, 0xB5, 0x30, 0xF3),
    BYTES_TO_DIGITS_8(0x44, 0x4B, 0x4A, 0xE9, 0x6C, 0x5C, 0xDC, 0x26),
};
static const cry_mpi_digit brainpoolP256r1_gx[] = {
    BYTES_TO_DIGITS_8(0x62, 0x32, 0xCE, 0x9A, 0xBD, 0x53, 0x44, 0x3A),
    BYTES_TO_DIGITS_8(0xC2, 0x23, 0xBD, 0xE3, 0xE1, 0x27, 0xDE, 0xB9),
    BYTES_TO_DIGITS_8(0xAF, 0xB7, 0x81, 0xFC, 0x2F, 0x48, 0x4B, 0x2C),
    BYTES_TO_DIGITS_8(0xCB, 0x57, 0x7E, 0xCB, 0xB9, 0xAE, 0xD2, 0x8B),
};
static const cry_mpi_digit brainpoolP256r1_gy[] = {
    BYTES_TO_DIGITS_8(0x97, 0x69, 0x04, 0x2F, 0xC7, 0x54, 0x1D, 0x5C),
    BYTES_TO_DIGITS_8(0x54, 0x8E, 0xED, 0x2D, 0x13, 0x45, 0x77, 0xC2),
    BYTES_TO_DIGITS_8(0xC9, 0x1D, 0x61, 0x14, 0x1A, 0x46, 0xF8, 0x97),
    BYTES_TO_DIGITS_8(0xFD, 0xC4, 0xDA, 0xC3, 0x35, 0xF8, 0x7E, 0x54),
};
static const cry_mpi_digit brainpoolP256r1_n[] = {
    BYTES_TO_DIGITS_8(0xA7, 0x56, 0x48, 0x97, 0x82, 0x0E, 0x1E, 0x90),
    BYTES_TO_DIGITS_8(0xF7, 0xA6, 0x61, 0xB5, 0xA3, 0x7A, 0x39, 0x8C),
    BYTES_TO_DIGITS_8(0x71, 0x8D, 0x83, 0x9D, 0x90, 0x0A, 0x66, 0x3E),
    BYTES_TO_DIGITS_8(0xBC, 0xA9, 0xEE, 0xA1, 0xDB, 0x57, 0xFB, 0xA9),
};

#endif /* ~CRY_ECP_GRP_BP256R1_ENABLE */


/*
 * Domain parameters for brainpoolP384r1 (RFC 5639 3.6)
 */

#ifdef CRY_ECP_GRP_BP384R1_ENABLE

static const cry_mpi_digit brainpoolP384r1_p[] = {
    BYTES_TO_DIGITS_8(0x53, 0xEC, 0x07, 0x31, 0x13, 0x00, 0x47, 0x87),
    BYTES_TO_DIGITS_8(0x71, 0x1A, 0x1D, 0x90, 0x29, 0xA7, 0xD3, 0xAC),
    BYTES_TO_DIGITS_8(0x23, 0x11, 0xB7, 0x7F, 0x19, 0xDA, 0xB1, 0x12),
    BYTES_TO_DIGITS_8(0xB4, 0x56, 0x54, 0xED, 0x09, 0x71, 0x2F, 0x15),
    BYTES_TO_DIGITS_8(0xDF, 0x41, 0xE6, 0x50, 0x7E, 0x6F, 0x5D, 0x0F),
    BYTES_TO_DIGITS_8(0x28, 0x6D, 0x38, 0xA3, 0x82, 0x1E, 0xB9, 0x8C),
};
static const cry_mpi_digit brainpoolP384r1_a[] = {
    BYTES_TO_DIGITS_8(0x26, 0x28, 0xCE, 0x22, 0xDD, 0xC7, 0xA8, 0x04),
    BYTES_TO_DIGITS_8(0xEB, 0xD4, 0x3A, 0x50, 0x4A, 0x81, 0xA5, 0x8A),
    BYTES_TO_DIGITS_8(0x0F, 0xF9, 0x91, 0xBA, 0xEF, 0x65, 0x91, 0x13),
    BYTES_TO_DIGITS_8(0x87, 0x27, 0xB2, 0x4F, 0x8E, 0xA2, 0xBE, 0xC2),
    BYTES_TO_DIGITS_8(0xA0, 0xAF, 0x05, 0xCE, 0x0A, 0x08, 0x72, 0x3C),
    BYTES_TO_DIGITS_8(0x0C, 0x15, 0x8C, 0x3D, 0xC6, 0x82, 0xC3, 0x7B),
};
static const cry_mpi_digit brainpoolP384r1_b[] = {
    BYTES_TO_DIGITS_8(0x11, 0x4C, 0x50, 0xFA, 0x96, 0x86, 0xB7, 0x3A),
    BYTES_TO_DIGITS_8(0x94, 0xC9, 0xDB, 0x95, 0x02, 0x39, 0xB4, 0x7C),
    BYTES_TO_DIGITS_8(0xD5, 0x62, 0xEB, 0x3E, 0xA5, 0x0E, 0x88, 0x2E),
    BYTES_TO_DIGITS_8(0xA6, 0xD2, 0xDC, 0x07, 0xE1, 0x7D, 0xB7, 0x2F),
    BYTES_TO_DIGITS_8(0x7C, 0x44, 0xF0, 0x16, 0x54, 0xB5, 0x39, 0x8B),
    BYTES_TO_DIGITS_8(0x26, 0x28, 0xCE, 0x22, 0xDD, 0xC7, 0xA8, 0x04),
};
static const cry_mpi_digit brainpoolP384r1_gx[] = {
    BYTES_TO_DIGITS_8(0x1E, 0xAF, 0xD4, 0x47, 0xE2, 0xB2, 0x87, 0xEF),
    BYTES_TO_DIGITS_8(0xAA, 0x46, 0xD6, 0x36, 0x34, 0xE0, 0x26, 0xE8),
    BYTES_TO_DIGITS_8(0xE8, 0x10, 0xBD, 0x0C, 0xFE, 0xCA, 0x7F, 0xDB),
    BYTES_TO_DIGITS_8(0xE3, 0x4F, 0xF1, 0x7E, 0xE7, 0xA3, 0x47, 0x88),
    BYTES_TO_DIGITS_8(0x6B, 0x3F, 0xC1, 0xB7, 0x81, 0x3A, 0xA6, 0xA2),
    BYTES_TO_DIGITS_8(0xFF, 0x45, 0xCF, 0x68, 0xF0, 0x64, 0x1C, 0x1D),
};
static const cry_mpi_digit brainpoolP384r1_gy[] = {
    BYTES_TO_DIGITS_8(0x15, 0x53, 0x3C, 0x26, 0x41, 0x03, 0x82, 0x42),
    BYTES_TO_DIGITS_8(0x11, 0x81, 0x91, 0x77, 0x21, 0x46, 0x46, 0x0E),
    BYTES_TO_DIGITS_8(0x28, 0x29, 0x91, 0xF9, 0x4F, 0x05, 0x9C, 0xE1),
    BYTES_TO_DIGITS_8(0x64, 0x58, 0xEC, 0xFE, 0x29, 0x0B, 0xB7, 0x62),
    BYTES_TO_DIGITS_8(0x52, 0xD5, 0xCF, 0x95, 0x8E, 0xEB, 0xB1, 0x5C),
    BYTES_TO_DIGITS_8(0xA4, 0xC2, 0xF9, 0x20, 0x75, 0x1D, 0xBE, 0x8A),
};
static const cry_mpi_digit brainpoolP384r1_n[] = {
    BYTES_TO_DIGITS_8(0x65, 0x65, 0x04, 0xE9, 0x02, 0x32, 0x88, 0x3B),
    BYTES_TO_DIGITS_8(0x10, 0xC3, 0x7F, 0x6B, 0xAF, 0xB6, 0x3A, 0xCF),
    BYTES_TO_DIGITS_8(0xA7, 0x25, 0x04, 0xAC, 0x6C, 0x6E, 0x16, 0x1F),
    BYTES_TO_DIGITS_8(0xB3, 0x56, 0x54, 0xED, 0x09, 0x71, 0x2F, 0x15),
    BYTES_TO_DIGITS_8(0xDF, 0x41, 0xE6, 0x50, 0x7E, 0x6F, 0x5D, 0x0F),
    BYTES_TO_DIGITS_8(0x28, 0x6D, 0x38, 0xA3, 0x82, 0x1E, 0xB9, 0x8C),
};

#endif /* ~CRY_ECP_GRP_BP384R1_ENABLE */


/*
 * Domain parameters for brainpoolP512r1 (RFC 5639 3.7)
 */

#ifdef CRY_ECP_GRP_BP512R1_ENABLE

static const cry_mpi_digit brainpoolP512r1_p[] = {
    BYTES_TO_DIGITS_8(0xF3, 0x48, 0x3A, 0x58, 0x56, 0x60, 0xAA, 0x28),
    BYTES_TO_DIGITS_8(0x85, 0xC6, 0x82, 0x2D, 0x2F, 0xFF, 0x81, 0x28),
    BYTES_TO_DIGITS_8(0xE6, 0x80, 0xA3, 0xE6, 0x2A, 0xA1, 0xCD, 0xAE),
    BYTES_TO_DIGITS_8(0x42, 0x68, 0xC6, 0x9B, 0x00, 0x9B, 0x4D, 0x7D),
    BYTES_TO_DIGITS_8(0x71, 0x08, 0x33, 0x70, 0xCA, 0x9C, 0x63, 0xD6),
    BYTES_TO_DIGITS_8(0x0E, 0xD2, 0xC9, 0xB3, 0xB3, 0x8D, 0x30, 0xCB),
    BYTES_TO_DIGITS_8(0x07, 0xFC, 0xC9, 0x33, 0xAE, 0xE6, 0xD4, 0x3F),
    BYTES_TO_DIGITS_8(0x8B, 0xC4, 0xE9, 0xDB, 0xB8, 0x9D, 0xDD, 0xAA),
};
static const cry_mpi_digit brainpoolP512r1_a[] = {
    BYTES_TO_DIGITS_8(0xCA, 0x94, 0xFC, 0x77, 0x4D, 0xAC, 0xC1, 0xE7),
    BYTES_TO_DIGITS_8(0xB9, 0xC7, 0xF2, 0x2B, 0xA7, 0x17, 0x11, 0x7F),
    BYTES_TO_DIGITS_8(0xB5, 0xC8, 0x9A, 0x8B, 0xC9, 0xF1, 0x2E, 0x0A),
    BYTES_TO_DIGITS_8(0xA1, 0x3A, 0x25, 0xA8, 0x5A, 0x5D, 0xED, 0x2D),
    BYTES_TO_DIGITS_8(0xBC, 0x63, 0x98, 0xEA, 0xCA, 0x41, 0x34, 0xA8),
    BYTES_TO_DIGITS_8(0x10, 0x16, 0xF9, 0x3D, 0x8D, 0xDD, 0xCB, 0x94),
    BYTES_TO_DIGITS_8(0xC5, 0x4C, 0x23, 0xAC, 0x45, 0x71, 0x32, 0xE2),
    BYTES_TO_DIGITS_8(0x89, 0x3B, 0x60, 0x8B, 0x31, 0xA3, 0x30, 0x78),
};
static const cry_mpi_digit brainpoolP512r1_b[] = {
    BYTES_TO_DIGITS_8(0x23, 0xF7, 0x16, 0x80, 0x63, 0xBD, 0x09, 0x28),
    BYTES_TO_DIGITS_8(0xDD, 0xE5, 0xBA, 0x5E, 0xB7, 0x50, 0x40, 0x98),
    BYTES_TO_DIGITS_8(0x67, 0x3E, 0x08, 0xDC, 0xCA, 0x94, 0xFC, 0x77),
    BYTES_TO_DIGITS_8(0x4D, 0xAC, 0xC1, 0xE7, 0xB9, 0xC7, 0xF2, 0x2B),
    BYTES_TO_DIGITS_8(0xA7, 0x17, 0x11, 0x7F, 0xB5, 0xC8, 0x9A, 0x8B),
    BYTES_TO_DIGITS_8(0xC9, 0xF1, 0x2E, 0x0A, 0xA1, 0x3A, 0x25, 0xA8),
    BYTES_TO_DIGITS_8(0x5A, 0x5D, 0xED, 0x2D, 0xBC, 0x63, 0x98, 0xEA),
    BYTES_TO_DIGITS_8(0xCA, 0x41, 0x34, 0xA8, 0x10, 0x16, 0xF9, 0x3D),
};
static const cry_mpi_digit brainpoolP512r1_gx[] = {
    BYTES_TO_DIGITS_8(0x22, 0xF8, 0xB9, 0xBC, 0x09, 0x22, 0x35, 0x8B),
    BYTES_TO_DIGITS_8(0x68, 0x5E, 0x6A, 0x40, 0x47, 0x50, 0x6D, 0x7C),
    BYTES_TO_DIGITS_8(0x5F, 0x7D, 0xB9, 0x93, 0x7B, 0x68, 0xD1, 0x50),
    BYTES_TO_DIGITS_8(0x8D, 0xD4, 0xD0, 0xE2, 0x78, 0x1F, 0x3B, 0xFF),
    BYTES_TO_DIGITS_8(0x8E, 0x09, 0xD0, 0xF4, 0xEE, 0x62, 0x3B, 0xB4),
    BYTES_TO_DIGITS_8(0xC1, 0x16, 0xD9, 0xB5, 0x70, 0x9F, 0xED, 0x85),
    BYTES_TO_DIGITS_8(0x93, 0x6A, 0x4C, 0x9C, 0x2E, 0x32, 0x21, 0x5A),
    BYTES_TO_DIGITS_8(0x64, 0xD9, 0x2E, 0xD8, 0xBD, 0xE4, 0xAE, 0x81),
};
static const cry_mpi_digit brainpoolP512r1_gy[] = {
    BYTES_TO_DIGITS_8(0x92, 0x08, 0xD8, 0x3A, 0x0F, 0x1E, 0xCD, 0x78),
    BYTES_TO_DIGITS_8(0x06, 0x54, 0xF0, 0xA8, 0x2F, 0x2B, 0xCA, 0xD1),
    BYTES_TO_DIGITS_8(0xAE, 0x63, 0x27, 0x8A, 0xD8, 0x4B, 0xCA, 0x5B),
    BYTES_TO_DIGITS_8(0x5E, 0x48, 0x5F, 0x4A, 0x49, 0xDE, 0xDC, 0xB2),
    BYTES_TO_DIGITS_8(0x11, 0x81, 0x1F, 0x88, 0x5B, 0xC5, 0x00, 0xA0),
    BYTES_TO_DIGITS_8(0x1A, 0x7B, 0xA5, 0x24, 0x00, 0xF7, 0x09, 0xF2),
    BYTES_TO_DIGITS_8(0xFD, 0x22, 0x78, 0xCF, 0xA9, 0xBF, 0xEA, 0xC0),
    BYTES_TO_DIGITS_8(0xEC, 0x32, 0x63, 0x56, 0x5D, 0x38, 0xDE, 0x7D),
};
static const cry_mpi_digit brainpoolP512r1_n[] = {
    BYTES_TO_DIGITS_8(0x69, 0x00, 0xA9, 0x9C, 0x82, 0x96, 0x87, 0xB5),
    BYTES_TO_DIGITS_8(0xDD, 0xDA, 0x5D, 0x08, 0x81, 0xD3, 0xB1, 0x1D),
    BYTES_TO_DIGITS_8(0x47, 0x10, 0xAC, 0x7F, 0x19, 0x61, 0x86, 0x41),
    BYTES_TO_DIGITS_8(0x19, 0x26, 0xA9, 0x4C, 0x41, 0x5C, 0x3E, 0x55),
    BYTES_TO_DIGITS_8(0x70, 0x08, 0x33, 0x70, 0xCA, 0x9C, 0x63, 0xD6),
    BYTES_TO_DIGITS_8(0x0E, 0xD2, 0xC9, 0xB3, 0xB3, 0x8D, 0x30, 0xCB),
    BYTES_TO_DIGITS_8(0x07, 0xFC, 0xC9, 0x33, 0xAE, 0xE6, 0xD4, 0x3F),
    BYTES_TO_DIGITS_8(0x8B, 0xC4, 0xE9, 0xDB, 0xB8, 0x9D, 0xDD, 0xAA),
};

#endif /* ~CRY_ECP_GRP_BP512R1_ENABLE */


static void ecp_mpi_load(cry_mpi *n, const cry_mpi_digit *d, size_t dlen)
{
    n->data  = (cry_mpi_digit *)d;
    n->alloc = 0;
    n->used  = dlen / sizeof(cry_mpi_digit);
    n->sign  = 0;
}

static void ecp_grp_load(cry_ecp_grp *grp,
                         const cry_mpi_digit *p,  size_t plen,
                         const cry_mpi_digit *a,  size_t alen,
                         const cry_mpi_digit *b,  size_t blen,
                         const cry_mpi_digit *gx, size_t gxlen,
                         const cry_mpi_digit *gy, size_t gylen,
                         const cry_mpi_digit *n,  size_t nlen)
{
    static const cry_mpi_digit one = 0x01;
    static const cry_mpi_digit three = 0x03;

    ecp_mpi_load(&grp->p, p, plen);
    if (a != NULL) {
        ecp_mpi_load(&grp->a, a, alen);
    } else {
        ecp_mpi_load(&grp->a, &three, sizeof(three));
        grp->a.sign = 1;
    }
    ecp_mpi_load(&grp->b, b, blen);
    ecp_mpi_load(&grp->g.x, gx, gxlen);
    ecp_mpi_load(&grp->g.y, gy, gylen);
    ecp_mpi_load(&grp->g.z, &one, sizeof(one));
    ecp_mpi_load(&grp->n, n, nlen);
}

#define GRP_LOAD(name) ecp_grp_load(grp, \
                           name ## _p,  sizeof(name ## _p),  \
                           name ## _a,  sizeof(name ## _a),  \
                           name ## _b,  sizeof(name ## _b),  \
                           name ## _gx, sizeof(name ## _gx), \
                           name ## _gy, sizeof(name ## _gy), \
                           name ## _n,  sizeof(name ## _n))

int cry_ecp_grp_load(cry_ecp_grp *grp, int grp_id)
{
    int res = 0;

    switch (grp_id) {
#ifdef CRY_ECP_GRP_SECP192R1_ENABLE
    case CRY_ECP_GRP_SECP192R1:
        GRP_LOAD(secp192r1);
        break;
#endif
#ifdef CRY_ECP_GRP_SECP224R1_ENABLE
    case CRY_ECP_GRP_SECP224R1:
        GRP_LOAD(secp224r1);
        break;
#endif
#ifdef CRY_ECP_GRP_SECP256R1_ENABLE
    case CRY_ECP_GRP_SECP256R1:
        GRP_LOAD(secp256r1);
        break;
#endif
#ifdef CRY_ECP_GRP_SECP384R1_ENABLE
    case CRY_ECP_GRP_SECP384R1:
        GRP_LOAD(secp384r1);
        break;
#endif
#ifdef CRY_ECP_GRP_SECP521R1_ENABLE
    case CRY_ECP_GRP_SECP521R1:
        GRP_LOAD(secp521r1);
        break;
#endif
#ifdef CRY_ECP_GRP_SECP192K1_ENABLE
    case CRY_ECP_GRP_SECP192K1:
        GRP_LOAD(secp192k1);
        break;
#endif
#ifdef CRY_ECP_GRP_SECP224K1_ENABLE
    case CRY_ECP_GRP_SECP224K1:
        GRP_LOAD(secp224k1);
        break;
#endif
#ifdef CRY_ECP_GRP_SECP256K1_ENABLE
    case CRY_ECP_GRP_SECP256K1:
        GRP_LOAD(secp256k1);
        break;
#endif
#ifdef CRY_ECP_GRP_BP256R1_ENABLE
    case CRY_ECP_GRP_BP256R1:
        GRP_LOAD(brainpoolP256r1);
        break;
#endif
#ifdef CRY_ECP_GRP_BP384R1_ENABLE
    case CRY_ECP_GRP_BP384R1:
        GRP_LOAD(brainpoolP384r1);
        break;
#endif
#ifdef CRY_ECP_GRP_BP512R1_ENABLE
    case CRY_ECP_GRP_BP512R1:
        GRP_LOAD(brainpoolP512r1);
        break;
#endif
    default:
        res = -1;
        break;
    }
    return res;
}
