use std::fs::File;
use std::io::{BufRead, BufReader, Seek, SeekFrom};

use crate::auto_decompress::auto_decompress;

use cres::event::Event;

use hepmc2::reader::{LineParseError, Reader};
use jetty::{anti_kt_f, cambridge_aachen_f, cluster_if, kt_f, PseudoJet};
use log::info;
use noisy_float::prelude::*;

use crate::opt::{JetAlgorithm, JetDefinition};

fn is_parton(particle: &hepmc2::event::Particle) -> bool {
    let id = particle.id;
    id.abs() <= 5 || id == 21
}

const OUTGOING_STATUS: i32 = 1;
const PID_JET: i32 = 81;

fn cluster(partons: Vec<PseudoJet>, jet_def: &JetDefinition) -> Vec<PseudoJet> {
    let minpt2 = jet_def.jetpt * jet_def.jetpt;
    let cut = |jet: PseudoJet| jet.pt2() > minpt2;
    let r = jet_def.jetradius;
    match jet_def.jetalgorithm {
        JetAlgorithm::AntiKt => cluster_if(partons, &anti_kt_f(r), cut),
        JetAlgorithm::Kt => cluster_if(partons, &kt_f(r), cut),
        JetAlgorithm::CambridgeAachen => {
            cluster_if(partons, &cambridge_aachen_f(r), cut)
        }
    }
}

pub(crate) fn into_event(
    event: hepmc2::event::Event,
    jet_def: &JetDefinition,
) -> Event {
    let mut res = Event::new();
    let mut partons = Vec::new();
    res.weight = n64(*event.weights.first().unwrap());
    for vx in event.vertices {
        let outgoing = vx
            .particles_out
            .into_iter()
            .filter(|p| p.status == OUTGOING_STATUS);
        for out in outgoing {
            if is_parton(&out) {
                partons.push(out.p.0.into());
            } else {
                let p = [
                    n64(out.p[0]),
                    n64(out.p[1]),
                    n64(out.p[2]),
                    n64(out.p[3]),
                ];
                res.add_outgoing(out.id, p.into())
            }
        }
    }
    let jets = cluster(partons, jet_def);
    for jet in jets {
        let p = [jet.e(), jet.px(), jet.py(), jet.pz()];
        res.add_outgoing(PID_JET, p.into());
    }
    res
}

pub struct CombinedReader {
    next_files: Vec<File>,
    previous_files: Vec<File>,
    reader: Reader<Box<dyn BufRead>>,
}

fn empty_reader() -> Reader<Box<dyn BufRead>> {
    Reader::new(Box::new(BufReader::new(std::io::empty())))
}

impl CombinedReader {
    pub fn new(files: Vec<File>) -> Self {
        CombinedReader {
            next_files: files,
            previous_files: Vec::new(),
            reader: empty_reader(),
        }
    }

    pub fn rewind(&mut self) -> Result<(), std::io::Error> {
        self.previous_files.reverse();
        self.next_files.append(&mut self.previous_files);
        for file in &mut self.next_files {
            file.seek(SeekFrom::Start(0))?;
        }
        self.reader = empty_reader();
        Ok(())
    }
}

impl Iterator for CombinedReader {
    type Item = Result<hepmc2::event::Event, LineParseError>;

    fn next(&mut self) -> Option<Self::Item> {
        let next = self.reader.next();
        if next.is_none() {
            if let Some(next_file) = self.next_files.pop() {
                self.previous_files.push(next_file.try_clone().unwrap());
                info!(
                    "Reading from file {}/{}",
                    self.previous_files.len(),
                    self.previous_files.len() + self.next_files.len()
                );

                let decoder = auto_decompress(BufReader::new(next_file));
                self.reader = Reader::from(decoder);
                self.next()
            } else {
                None
            }
        } else {
            next
        }
    }
}
