use crate::utils::fs;
use crate::utils::logger::add_file_msg;
use crate::plugins::InstallConfig;
use crate::plugins::Plugin;
use anyhow::Result;
use rust_embed::RustEmbed;
use crate::BackendFramework;

pub struct Dev {}

#[derive(RustEmbed)]
#[folder = "template-plugin-dev"]
struct Asset;

impl Plugin for Dev {
    fn name(&self) -> &'static str {
        "Dev"
    }

    fn install(&self, install_config: InstallConfig) -> Result<()> {
        for filename in Asset::iter() {
            if filename.starts_with("README.md") || filename.contains(".cargo/admin") && !filename.contains(".cargo/admin/dist") {
                continue;
            }

            let file_contents = Asset::get(filename.as_ref()).unwrap();
            let mut file_path = std::path::PathBuf::from(&install_config.project_dir);
            file_path.push(filename.as_ref());
            let mut directory_path = std::path::PathBuf::from(&file_path);
            directory_path.pop();

            add_file_msg(filename.as_ref());
            std::fs::create_dir_all(directory_path)?;
            std::fs::write(file_path, file_contents)?;
        }

        // TODO: Fix these appends/prepends by prepending the filepath with project_dir
        // currently, this works because we assume the current working directory is the project's root

        // TODO: don't use concurrently as the anchor for new frontend dependencies
        fs::replace(
            "frontend/package.json",
            r#""concurrently": "^7.1.0""#,
            r#""concurrently": "^7.1.0",
    "react-query": "^3.21.0""#,
        )?;

        fs::replace(
            "frontend/src/App.tsx",
            "const App = () => {",
            r#"if (process.env.NODE_ENV === 'development') import('./setupDevelopment')
    
    const App = () => {"#,
        )?;

        match install_config.backend_framework {
            BackendFramework::ActixWeb => {
                fs::replace("backend/main.rs",
                            r#"/* Development-only routes */"#,
                            r#"/* Development-only routes */
            // Mount development-only API routes
            api_scope = api_scope.service(create_rust_app::dev::endpoints(web::scope("/development")));
            // Mount the admin dashboard on /admin
            app = app.service(web::scope("/admin").service(Files::new("/", ".cargo/admin/dist/").index_file("admin.html")));"#)?;

            },
            BackendFramework::Poem => {
                fs::replace(
                    "backend/main.rs",
                    "let mut app = Route::new().nest(\"/api\", api);",
                    r#"#[cfg(debug_assertions)]
        {
            api = api.nest("/development", create_rust_app::dev::api());
        }

        let mut app = Route::new().nest("/api", api);

        #[cfg(debug_assertions)]
        {
            app = app.at(
                "*",
                StaticFilesEndpoint::new(".cargo/admin/dist").index_file("admin.html"),
            );
        }"#
                )?;
            }
        }

        Ok(())
    }
}
