use rand::Rng;
use std::collections::HashMap;
use std::collections::HashSet;

mod distribution_map;

static WEIGHTS: &'static [u32] = &[
    0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
];

static GRAYTBL: &'static [i8] = &[0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0];
const START_POL: u64 = 0x200001;
const END_POL: u64 = 0x3fffff;
const POLAR_CODE_LENGTH: u64 = 8192;

#[derive(Default)]
struct Pol {
    pol: u64,
    ord: u64,
}

#[allow(dead_code)]
impl Pol {
    fn reciprocal(&self) -> Pol {
        let mut new_pol = self.pol;
        let mut res = 0u64;
        while new_pol > 0 {
            res *= 2;
            if new_pol % 2 == 1 {
                res += 1;
            }
            new_pol /= 2;
        }
        let result = Pol {
            pol: res,
            ord: self.ord,
        };
        result
    }

    fn get_weight(&self) -> u8 {
        let mut new_pol = self.pol;
        let mut res = 0u8;
        while new_pol > 0 {
            if new_pol % 2 == 1 {
                res += 1;
            }
            new_pol /= 2;
        }
        res
    }

    fn get_order(&self) -> u64 {
        //make it to return error as well
        let mut co = 1u64;
        if self.ord == 0 {
            let mut pp = 1u64;
            let mut p = 0u64;

            while pp * 2 <= self.pol {
                pp *= 2;
                p += 1;
            }
            let short_pol = self.pol ^ pp;

            let mask = 1u64 << (p - 1);
            let mut cur_pol = 2u64;

            while cur_pol != 1 {
                if cur_pol >= mask {
                    cur_pol = cur_pol ^ mask;
                    cur_pol = cur_pol << 1;
                    cur_pol = cur_pol ^ short_pol;
                } else {
                    cur_pol = cur_pol << 1;
                }
                co += 1;
            } //while
        }
        co
    }
    //returns a(x) mod g(x), deg g(x)<32
    pub fn get_reminder(&self, i: &Vec<u32>, cardinality: u32) -> u32 {
        assert!(cardinality < 32);
        let mut curr = 0u64;
        for (idx, el) in i.iter().enumerate().rev() {
            curr = curr ^ (*el as u64);
            if idx > 0 {
                curr = curr << 32;
                while curr != 0 && curr.leading_zeros() < 32 {
                    curr = curr ^ (self.pol << (63 - curr.leading_zeros() - cardinality));
                }
            } else {
                while curr != 0 && 63 - curr.leading_zeros() >= cardinality {
                    curr = curr ^ (self.pol << (63 - curr.leading_zeros() - cardinality));
                }
            }
        }
        curr as u32
    }
    pub fn add_noise(&self, number_of_errors: u16, origin: &mut Vec<u32>) -> () {
        assert!(number_of_errors > 0);
        let mut used: HashSet<usize> = vec![].into_iter().collect();
        let pol_length = 32 * (origin.len() as u32) - origin[origin.len() - 1].leading_zeros();
        //sprintln!("Pol Len: {}", pol_length);
        for _ in 0..number_of_errors {
            let mut updated_value = rand::thread_rng().gen_range(0, pol_length) as usize;
            while used.contains(&updated_value) {
                updated_value = rand::thread_rng().gen_range(0, pol_length) as usize;
            }
            used.insert(updated_value);
            let updated_index = updated_value / 32 as usize;
            let updated_bit = updated_value - 32 * updated_index;
            //            println!("Updating : {} {} {}", updated_value, updated_index, updated_bit);
            // FIXME: check whether top one is updated
            origin[updated_index] = origin[updated_index] ^ (1u32 << updated_bit);
        }
    }

    pub fn irrpol(&self, large_n: u64, cardinality: u32) -> Vec<u32> {
        // deli se na (2 deg {p})^irr
        // int i, br, br1, range;
        // unsigned int n=32767; //counter
        let mask = 1u64 << cardinality;
        // cout<<"mask:"<<mask<<"\n";
        let mut n = large_n - cardinality as u64; // counter
        let mut m = mask; // memory $FFFFFFFF
                          //
                          /*
                           * if (1+N/32 > COLS) return POL_OUTOFRANGE;
                           */
        //		let range = 1 + large_n / 32;
        let mut codewords: Vec<u32> = Vec::new();

        //		for (i = 0; i < range; i++) {
        //			codewords[i] = 0;
        //		}
        let mut br = 0u16;
        let mut br1 = 0u16;

        let mut codeword = 0u32;

        while n > 0 {
            m = m ^ self.pol;
            m = m << 1;

            codeword = 1 + (codeword << 1);
            br1 = br1 + 1;
            n = n - 1;
            if br1 == 32 {
                codewords.push(codeword);
                br = br + 1;
                br1 = 0;
                codeword = 0;
            }
            while (m < mask) && (n > 0) {
                n = n - 1;
                m = m << 1;
                codeword = codeword << 1;
                br1 = br1 + 1;
                if br1 == 32 {
                    br = br + 1;
                    br1 = 0;
                    codewords.push(codeword);
                    codeword = 0;
                }
            }
        }
        codeword = 1 + (codeword << 1);
        br1 = br1 + 1;
        while br1 < 32 {
            br1 = br1 + 1;
            codeword = codeword << 1;
        }

        codewords.push(codeword);
        codewords
    }

    pub fn calculate_dual_dist_distr(
        &self,
        shorten_position: u64,
        force: bool,
        cardinality: u32,
    ) -> u8 {
        let mut res = 0u8;
        const MAX_N: usize = (POLAR_CODE_LENGTH / 32 + 1) as usize;
        let mut c: [u32; MAX_N] = [0u32; MAX_N];
        let mut dualdistance_distribution_calculated = false; //FIXME: make this part of the class

        if dualdistance_distribution_calculated && !force {
            0u8
        } else {
            let numcodel = 1 << cardinality;
            let d2k = numcodel;
            let ordp = self.get_order();
            let mut position = ordp;
            if shorten_position != 0 {
                position = shorten_position;
            }
            let codewords = self.irrpol(self.get_order(), cardinality); //FIXME: use already calculated order

            let mut n_16 = (position / 32) as u32;
            let k = cardinality;

            if force && (n_16 > (POLAR_CODE_LENGTH / 32) as u32) {
                n_16 = (POLAR_CODE_LENGTH / 32) as u32;
            }

            let jj = position % 32;

            let r = if jj == 0 { n_16 } else { n_16 + 1 };

            if (ordp == position) || force {
                //				let mut G = [[0u32, ..cardinality as u32], ..(n_16 + 1) as u32];

                // Base 1d array
                let mut grid_raw = vec![0u32; (cardinality * (n_16 + 1)) as usize];

                let mut grid_base: Vec<_> = grid_raw
                    .as_mut_slice()
                    .chunks_mut((n_16 + 1) as usize)
                    .collect();

                // Final 2d array `&mut [&mut [_]]`
                let G = grid_base.as_mut_slice();

                for j in 0..=n_16 {
                    G[0][j as usize] = codewords[j as usize];
                }

                for i in 1..k {
                    //<k
                    G[i as usize][0] = G[(i - 1) as usize][0] >> 1;
                    for j in 1..r {
                        G[i as usize][j as usize] =
                            (G[(i - 1) as usize][(j - 1) as usize] & 0x00000001) << 31
                                | G[(i - 1) as usize][j as usize] >> 1;
                    }
                }
                let mut dual_distance = distribution_map::DistributionMap {
                    frequency: HashMap::new(),
                    weights: HashMap::new(),
                };
                dual_distance.init();
                // Gray code method
                for i in 1..numcodel {
                    let mut j = 0;
                    let mut tmp = i / 16;
                    let mut tmp1 = i % 16;
                    while tmp1 == 0 {
                        j += 1;
                        tmp1 = tmp % 16;
                        tmp = tmp / 16;
                    }
                    let chel = GRAYTBL[tmp1 - 1] + 4 * j;
                    for m in 0..r {
                        c[m as usize] = c[m as usize] ^ G[chel as usize][m as usize];
                    }

                    let mut wt = 0u32;
                    for j in 0..r {
                        // count words here
                        let mut m = c[j as usize];

                        //						if m < 0 {
                        //							wt += 1;
                        //							m = m & 0x7FFFFFFF;
                        //						}
                        // System.out.println("c["+j+"]=0x"+Integer.toHexString(c[(int)j]));
                        while m > 0 {
                            wt += WEIGHTS[(m % 256) as usize];
                            m = m >> 8;
                        }
                    }

                    //					if (wt == 33)
                    //						for j in 0..r {
                    //							// 40000000 is wrongly assumed 7FFFFFF
                    //							m = c[j];
                    //						}
                    dual_distance.add(wt);
                } // for

                dualdistance_distribution_calculated = true;
                dual_distance.print_all();
            }
            res
        } // if
    } // calculateDualDistDistr
}

// static tools

pub fn add_noise_vars(number_of_errors: u16, origin: u32, max_size: u32) -> u32 {
    assert!(number_of_errors > 0);
    let mut used: HashSet<u32>;
    let mut res = origin;
    for _ in 0..number_of_errors {
        let updated_bit = rand::thread_rng().gen_range(0, max_size) as usize;
        res = res ^ (1u32 << updated_bit);
    }
    res
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_order_4() {
        let pol = Pol { pol: 19, ord: 0 };
        assert_eq!(pol.get_order(), 15);
    }

    #[test]
    fn test_order_13() {
        let pol = Pol {
            pol: 0x299d,
            ord: 0,
        };
        assert_eq!(pol.get_order(), 3556);
    }

    #[test]
    fn test_order_16() {
        let pol = Pol {
            pol: 0x1a2eb,
            ord: 0,
        };
        assert_eq!(pol.get_order(), 32767);
    }

    #[test]
    fn test_order_16_2() {
        let pol = Pol {
            pol: 0x158ff,
            ord: 0,
        };
        assert_eq!(pol.get_order(), 7161);
    }

    #[test]
    fn test_weight() {
        let pol = Pol {
            pol: 0x158ff,
            ord: 0,
        };
        assert_eq!(pol.get_weight(), 12);
    }
    #[test]
    fn test_reminder_1() {
        let pol = Pol {
            pol: 0b11001,
            ord: 0,
        };
        let mut i: Vec<u32> = vec![0x10001];
        assert_eq!(pol.get_reminder(&i, 4), 0b11);
    }

    #[test]
    fn test_reminder_2() {
        let pol = Pol {
            pol: 0b11001,
            ord: 0,
        };
        let mut i: Vec<u32> = vec![0x10001, 0x1];
        assert_eq!(pol.get_reminder(&i, 4), 0b111);
    }

    #[test]
    fn test_reminder_3() {
        let pol = Pol { pol: 0b11, ord: 0 };
        let mut i: Vec<u32> = vec![0x10001, 0x1];
        assert_eq!(pol.get_reminder(&i, 1), 0b1);
    }

    #[test]
    fn test_reminder_4() {
        let pol = Pol {
            pol: 0x1004103,
            ord: 0,
        };
        let i: Vec<u32> = vec![0x10001, 0x1];
        assert_eq!(pol.get_reminder(&i, 24), 0x400301);
    }
    #[test]
    fn test_reminder_5() {
        let pol = Pol { pol: 0b101, ord: 0 };
        let i: Vec<u32> = vec![0x10001, 0x1];
        assert_eq!(pol.get_reminder(&i, 2), 1);
    }

    #[test]
    fn test_add_noise_var_1() {
        let res: u32 = add_noise_vars(1, 0xCAFEBABE, 1);
        assert_eq!(res, 0xCAFEBABF);
    }
    
        #[test]
    fn test_add_more_noise_1() {
        let pol_a = Pol { pol: 0xA9, ord: 0 };
        let i: Vec<u32> = vec![
            0xCAFEBABE, 0x155
        ];
        for _ in 0..100 {
            let mut j = i.clone();
            pol_a.add_noise(2, &mut j);
            assert_eq!(j[0] == 0xCAFEBABE && j[1] == 0x155, false);
        }
    }

}

fn vec_compare(va: &[u32], vb: &[u32]) -> bool {
    (va.len() == vb.len()) &&
     va.iter()
       .zip(vb)
       .all(|(a,b)| a == b)
}

fn main() {
    let pol = Pol { pol: 19, ord: 0 };
    println!(
        "Hello weight {} world with order {} and irrpol:",
        pol.get_weight(),
        pol.get_order()
    );
    let polvec = pol.irrpol(pol.get_order(), 4);
    for v in polvec {
        println!("Hello {:?}", v);
    }

    let mut dd = distribution_map::DistributionMap {
        frequency: HashMap::new(),
        weights: HashMap::new(),
    };
    dd.init();
    dd.add(7);
    dd.add(6);
    dd.add(7);
    dd.init_elems(5);
    dd.recalculate_elems(5, 3);
    println!("Hello {} .. {}!", dd.get(0), dd.get(42));
    dd.print_all();
    dd.print_all();
    println!("Calc bi = {} .. {}!", dd.calc_bi(), dd.calc_bi());
    let pol2 = Pol {
        pol: 0x1004103,
        ord: 0,
    };
    let mut i: Vec<u32> = vec![0x10001, 0x1];
    println!("Reminder {:x?}", pol2.get_reminder(&i, 24));
    pol2.add_noise(3, &mut i);
    pol2.add_noise(2, &mut i);

    let pol_0 = Pol { pol: 3, ord:0};
    let pol_a = Pol { pol: 0x67, ord: 0 };//A9/(x+1)=1100111
    let pol_b = Pol { pol: 0x12D, ord: 0 };
    let pol_c = Pol { pol: 0x28B, ord: 0 };
    //max length 5355
    let mut success = 0;
    let mut failure = 0;
    for w in 1..200 {
        println!("Word length {}",w*32); 
        
        for _ in 0..500000 {
            //FIXME: Generate random info on random length
//            let mut i: Vec<u32> = vec![
//                0x1010BEBE, 0xCAFEBABE, 0x1450421, 0x123ACAFE, 0x1010BEBE, 0xCAFEBABE, 0x1450421,
//                0x123ACAFE, 0x10107777, 0xCAFEBABE, 0x1450421, 0x1033CAFE, 0x1010BEBE, 0xCAFEBABE,
//                0x6307021, 0x123ACAFE, 0x155,
//            ];
            let mut rng = rand::thread_rng();
            let mut i: Vec<u32> = (0..w).map(|_| {
                rng.gen_range(0u32,0xFFFFFFFFu32)
            }).collect();

            let mut j = i.clone();
            let x0 = pol_0.get_reminder(&i, 1);
            let x1 = pol_a.get_reminder(&i, 6);
            let x2 = pol_b.get_reminder(&i, 8);
            let x3 = pol_c.get_reminder(&i, 9);
            let u1 = (x0 & 1) as u32 | ((x1<<1) & 0x7E) as u32 | ((x2<<7) & 0x7F80) as u32 | ((x3<<15) & 0xFF8000) as u32;
            i.extend([u1].iter().copied()); //FIXME: make this compact, i.e. one variables
            pol_a.add_noise(3, &mut i); // FIXME: make this method static
            let v1 = i.pop().unwrap_or(0);
            let p3 = (v1 & 0xFF8000) >> 15;
            let p2 = (v1 & 0x7F80) >> 7;
            let p1 = (v1 & 0x7E) >> 1;
            let p0 = v1 & 1;
            let y0 = pol_0.get_reminder(&i, 1);
            let y1 = pol_a.get_reminder(&i, 6);
            let y2 = pol_b.get_reminder(&i, 8);
            let y3 = pol_c.get_reminder(&i, 9);
            let mut good_check = 0;
            if p0 == y0 {
                good_check += 1
            }
            if p1 == y1 {
                good_check += 1
            } else if (p1 ^ y1).count_ones() > 1 {
                good_check += -10
            }
    
            if p2 == y2 {
                good_check += 1
            } else if (p2 ^ y2).count_ones() > 1 {
                good_check += -10
            }
            if p3==y3 {
                good_check += 1
            } else if (p3 ^ y3).count_ones() > 1 {
                good_check += -10
            }
            if !vec_compare(&i, &j){
                if good_check >= 3 {
                    success += 1;
                    println!("x0 = {:x?}, p0 = {:x?}, y0 = {:x?}, ", x0, p0, y0);
                    println!("x1 = {:x?}, p1 = {:x?}, y1 = {:x?}, x2 = {:x?}, p2 = {:x?}, y2 = {:x?}, x3={:x?}, p3={:x?}, y3={:x?}, {}!", x1, p1, y1, x2, p2, y2, x3, p3, y3, vec_compare(&i, &j));
                    println!("original {:x?}", j);
                    println!("noise    {:x?}", i);
                } else {
                    failure += 1;
                }
            }
            //println!("Success = {}, Failures = {}!", success, failure);
        }
        println!("Success = {}, Failures = {}!", success, failure);
    }    
//Word length 608
//x0 = 1, p0 = 1, y0 = 1, 
//x1 = 2e, p1 = 2e, y1 = 2f, x2 = df, p2 = 5f, y2 = 5f, x3=145, p3=145, y3=145, false!
//original [b2afd660, 5ac80476, be77a903, c57405cc, 55b947e2, 57ab3a09, 1a7c1708, 1a02ba52, 2c7ea7fd, 1c70806b, 1d1eae51, a903261b, e75400a9, bb796dbe, 847a7af2, 5f5aa26e, 230140eb, 52539637, c9b20565]
//noise    [b2afd660, 5ac80476, be77a903, c57405cc, 55b947e2, 57ab3a09, 1a7c1f08, 1a02ba52, 2c7ea7fd, 1c70806b, 1d1eae51, a903265b, e75400a9, bb796dbe, 847a7af2, 5f5aa26e, 230140eb, 52539637, c9b20565]
//Success = 1, Failures = 9450433!
//Word length 640
//Success = 1, Failures = 9950421!
//Word length 672
//Success = 1, Failures = 10450402!
//Word length 704
//Success = 1, Failures = 10950393!
//Word length 736
//x0 = 0, p0 = 0, y0 = 0, 
//x1 = 7, p1 = 6, y1 = 6, x2 = b9, p2 = b9, y2 = 39, x3=13b, p3=13b, y3=13b, false!
//original [cbd65def, e3de0511, 6d61b570, c9dd87e4, 99d9d15b, e7dcd7ee, ab87c507, 58f8d9b, 54fc51c, 69ed909b, 1a4fc014, 804f0b28, b6bba792, 8b28b2a3, a98ffb51, b17bcbcd, fa63b364, e4d69516, fa71d293, 4352849, 42e409d4, 466bf2de, ec9095be]
//noise    [cbd65def, e3de0511, 6d61b570, c9dd87e4, 99d9d15b, e7dcd7ee, ab87cd07, 58f8d9b, 54fc51c, 69ed909b, 1a4fc014, 804f0b68, b6bba792, 8b28b2a3, a98ffb51, b17bcbcd, fa63b364, e4d69516, fa71d293, 4352849, 42e409d4, 466bf2de, ec9095be]
//Success = 2, Failures = 11450382!
//Word length 768
//Success = 2, Failures = 11950371!
//Word length 800
//Success = 2, Failures = 12450360!
//Word length 832
//Success = 2, Failures = 12950349!
//Word length 864
//Success = 2, Failures = 13450344!
//Word length 896
//Success = 2, Failures = 13950342!
//Word length 928
//Success = 2, Failures = 14450333!
//Word length 960
//Success = 2, Failures = 14950327!
//Word length 992
//x0 = 0, p0 = 0, y0 = 0, 
//x1 = 18, p1 = 18, y1 = 1a, x2 = 8f, p2 = 9f, y2 = 9f, x3=192, p3=192, y3=192, false!
//original [26e7e2d7, a9d56adc, 4c3cced2, 9023cedb, a9e0598, 1a2346a5, 45595d2f, b4ced6ce, c31d56a8, bc2523cd, e3fb0618, 3952a67c, 492dcc7e, 4511d513, e9801570, e5cf844f, 78838fe4, 524df966, 1e087764, 853cd3ea, a35a5709, ba77fa11, 386868e2, eee15d2a, cfb67c21, d42d1c0e, 886979bb, 2214b5c9, 7db11ff9, 2765a8cf, 4b1c191e]
//noise    [26e7e2d7, a9d56adc, 4c3cded2, 9023cedb, a9e0598, 1a2346a5, 45595d2f, b4ced6ce, c31d56a8, bc2523cd, e3fb0618, 3952a67c, 492dcc7e, 4511d513, e9801570, e5cf844f, 78838fe4, 524df966, 1e087764, 853cd3ea, a35a5709, ba77fa11, 386868e2, eee15d2a, cfb67c21, d42d1c0e, 886179bb, 2214b5c9, 7db11ff9, 2765a8cf, 4b1c191e]
//Success = 3, Failures = 15450323!
//Word length 1024
//Success = 3, Failures = 15950321!
//Word length 1056
//Success = 3, Failures = 16450319!
//Word length 1088
//Success = 3, Failures = 16950316!
//Word length 1120
//Success = 3, Failures = 17450314!
//Word length 1152
//Success = 3, Failures = 17950310!
//Word length 1184
//Success = 3, Failures = 18450307!
//Word length 1216
//x0 = 1, p0 = 1, y0 = 1, 
//x1 = 1a, p1 = a, y1 = a, x2 = 93, p2 = 93, y2 = 13, x3=1ba, p3=1ba, y3=1ba, false!
//original [cf0251db, 418e7dfe, ae5b8bc3, ea6817c2, 12d768be, 8932e98e, ee7c7f36, d0b8e3b0, a407d834, 3fe770dd, a8fd0692, d52b9a9b, 9cbeff3e, cad0dd1c, 797ba645, 6a536d11, fe88d9ea, b0095ee6, f29ce008, c731f815, f84b2a64, 16d4902c, 581966de, d379cd7c, eea43ce1, bc3d1932, 3adb9fd8, ffe32961, 9db612ed, abbb679c, 725eca62, d9f2beb7, bd70f05, 23e216a9, 96a7eef9, bfe28d2c, ad23afc6, e59a59aa]
//noise    [cf0251db, 418e7dfe, ae5b0bc3, ea6817c2, 12d768be, 8932e98e, ee7c7f36, d0b8e3b0, a407d834, 3fe770dd, a8fd0692, d52b9a9b, 9cbeff3e, cad0dd1c, 797ba645, 6a536d11, fe88d9ea, b0095ee6, f29ce008, c731f815, f84b2a64, 16d4902c, 581966de, d379cd7c, eea43ce1, bc3d1932, 3a9b9fd8, ffe32961, 9db612ed, abbb679c, 725eca62, d9f2beb7, bd70f05, 23e216a9, 96a7eef9, bfe28d2c, ad23afc6, e59a59aa]
//Success = 4, Failures = 18950303!
//Word length 1248
}
