use crcany::crc::Computer;
use crcany::model::{BitwiseModel, BytewiseModel, WordwiseModel};
use crcany::spec::Spec;

const ALL_CRCS: &str = include_str!(concat!(env!("CARGO_MANIFEST_DIR"), "/allcrcs.txt"));

fn main() {
    let all_specs: Vec<Spec> = ALL_CRCS
        .lines()
        .map(|line| line.parse())
        .collect::<Result<Vec<_>, _>>()
        .unwrap();

    let mut good_bits = 0;
    let mut good_bytes = 0;
    let mut good_words = 0;
    let total = all_specs.len();

    for spec in all_specs {
        let bitly = BitwiseModel::from_spec(spec.clone());

        let crc = Computer::crc(&bitly, b"123456789");

        if spec.check != crc {
            println!(
                "Error on CRC {} (bitwise), expected {:#06x}, got {:#06x}",
                spec.name, spec.check, crc
            );
        } else {
            good_bits += 1;
        }

        let bytely = BytewiseModel::from_spec(spec.clone());

        let crc = Computer::crc(&bytely, b"123456789");

        if spec.check != crc {
            println!(
                "Error on CRC {} (bytewise), expected {:#06x}, got {:#06x}",
                spec.name, spec.check, crc
            );
        } else {
            good_bytes += 1;
        }

        let wordly = WordwiseModel::<byteorder::NativeEndian, 64>::from_spec(spec.clone());

        let crc = Computer::crc(&wordly, b"123456789");

        if spec.check != crc {
            println!(
                "Error on CRC {} (wordwise), expected {:#06x}, got {:#06x}",
                spec.name, spec.check, crc
            );
        } else {
            good_words += 1;
        }
    }

    println!("Passed bitwise: {}/{}", good_bits, total);
    println!("Passed bytewise: {}/{}", good_bytes, total);
    println!("Passed wordwise: {}/{}", good_words, total);
}
