extern crate proc_macro;
use proc_macro::TokenStream;

use std::collections::HashMap;

use once_cell::sync::OnceCell;
use quote::quote;
use syn::parse::{Parse, ParseStream};
use syn::parse_macro_input;

use crcany_core::spec::Spec;

// TODO: explicit params?
struct CrcInput {
    name: syn::LitStr,
}

impl Parse for CrcInput {
    fn parse(input: ParseStream) -> syn::Result<Self> {
        let name = input.parse()?;
        Ok(CrcInput { name })
    }
}

const ALL_CRCS_STR: &str = include_str!(concat!(env!("CARGO_MANIFEST_DIR"), "/allcrcs-abbrev.txt"));

fn all_crcs() -> &'static HashMap<String, Spec> {
    static ALL_CRCS: OnceCell<HashMap<String, Spec>> = OnceCell::new();
    ALL_CRCS.get_or_init(|| {
        let all_specs = ALL_CRCS_STR
            .lines()
            .map(|line| line.parse())
            .collect::<Result<Vec<Spec>, _>>()
            .unwrap_or_default();

        all_specs
            .into_iter()
            .map(|spec| (spec.name.clone(), spec))
            .collect::<HashMap<String, Spec>>()
    })
}

// TODO: search maybe?
fn get_crc(name: &str) -> Option<&'static Spec> {
    let map = all_crcs();
    map.get(name)
}

/// Implement a module to calculate a CRC.
///
/// The string parameter must match exactly the name of one of
/// the sets of CRC parameters.
///
/// ```expectfail
/// use crcany::crc::v2::Crc;
/// use crcany::impl_crc;
///
/// impl_crc!("CRC-3/GSM");
///
/// let mut crc = crc3gsm::bitwise::Crc3Gsm::new();
/// crc.add_bytes(b"123456789");
/// assert_eq!(4, crc.to_inner());
/// ```
#[proc_macro]
pub fn impl_crc(input: TokenStream) -> TokenStream {
    let input = parse_macro_input!(input as CrcInput);
    let name = input.name.value();

    let tokens = if let Some(spec) = get_crc(&name) {
        crcany_core::gen::gen_mod(spec)
    } else {
        panic!("Unable to find CRC named {}!", name);
    };

    tokens.into()
}

#[proc_macro]
pub fn impl_all(_input: TokenStream) -> TokenStream {
    let mut tokens = quote!();

    for name in all_crcs().keys() {
        tokens = quote!(#tokens ::crcany::impl_crc!(#name););
    }

    tokens.into()
}
