use crate::spec::Spec;
use crate::util::{ones, parse_int};

#[derive(Debug)]
pub enum SpecParseError {
    InvalidInput,
    MissingParams,
}

pub fn read_spec(s: &str) -> Result<Spec, SpecParseError> {
    let mut width = None;
    let mut poly = None;
    let mut init = 0;
    let mut refin = None;
    let mut refout = None;
    let mut xorout = 0;
    let mut check = None;
    let mut residue = 0;
    let mut name = None;

    //println!("parsing: {}", s);

    for (param, value) in parse_spec(s).or(Err(SpecParseError::InvalidInput))?.1 {
        //println!("checking param: {} = {}", param, value);

        match param {
            "width" | "w" => {
                width = Some(parse_int(value).or(Err(SpecParseError::InvalidInput))? as _)
            }
            "poly" | "p" => poly = Some(parse_int(value).or(Err(SpecParseError::InvalidInput))?),
            "init" | "i" => init = parse_int(value).or(Err(SpecParseError::InvalidInput))?,
            "refin" | "r" => refin = Some(value == "true" || value == "t"),
            "refout" | "refo" => refout = Some(value == "true" || value == "t"),
            "xorout" | "x" => xorout = parse_int(value).or(Err(SpecParseError::InvalidInput))?,
            "check" | "c" => check = Some(parse_int(value).or(Err(SpecParseError::InvalidInput))?),
            "residue" | "res" => {
                residue = parse_int(value).or(Err(SpecParseError::InvalidInput))?
            }
            "name" | "n" => name = Some(value.to_string()),
            _ => panic!(),
        }
    }

    let (refin, refout) = match (refin, refout) {
        (Some(a), Some(b)) => (a, b),
        (Some(a), None) => (a, a),
        (None, Some(b)) => (b, b),
        (None, None) => return Err(SpecParseError::MissingParams),
    };

    let (width, poly, check, name) = match (width, poly, check, name) {
        (Some(w), Some(p), Some(c), Some(n)) => (w, p, c, n),
        _ => return Err(SpecParseError::MissingParams),
    };

    xorout &= ones(width);

    Ok(Spec {
        width,
        poly,
        init,
        refin,
        refout,
        xorout,
        check,
        residue,
        name,
    })
}

fn parse_var(i: &str) -> nom::IResult<&str, (&str, &str)> {
    use nom::{
        branch::alt,
        character::complete::{char, none_of},
        combinator::recognize,
        multi::{many0_count, many1_count},
        sequence::{delimited, separated_pair},
    };

    separated_pair(
        recognize(many1_count(none_of(" ="))),
        char('='),
        alt((
            delimited(char('"'), recognize(many0_count(none_of("\""))), char('"')),
            recognize(many1_count(none_of(" ="))),
        )),
    )(i)
}

fn parse_spec(i: &str) -> nom::IResult<&str, Vec<(&str, &str)>> {
    use nom::{character::complete::space1, multi::separated_list1};

    separated_list1(space1, parse_var)(i)
}
