use crate::crc::{Crc, FromSpec};
use crate::model::bitwise::BitwiseModel;
use crate::spec::Spec;
use crate::util::{ones, reverse};

pub struct BytewiseModel {
    model: BitwiseModel,
    // table for byte-wise calculation
    pub table: [u128; 256],
}

impl BytewiseModel {
    pub fn reflect(&self) -> bool {
        self.model.reflect()
    }

    pub fn reverse(&self) -> bool {
        self.model.reverse()
    }

    pub fn width(&self) -> u16 {
        self.model.width()
    }

    pub fn xorout(&self) -> u128 {
        self.model.xorout()
    }

    pub fn init(&self) -> u128 {
        self.model.init()
    }

    fn fill_table(&mut self) {
        for k in 0..256_usize {
            let byte = k as _;
            let mut crc = self.model.crc(0, core::iter::once(byte)) as u128;
            if self.model.reverse() {
                crc = reverse(crc, self.model.width());
            }
            if self.model.width() < 8 && !self.model.reflect() {
                crc <<= 8 - self.model.width();
            }
            self.table[k] = crc;
        }
    }
}

impl FromSpec for BytewiseModel {
    fn from_spec(spec: Spec) -> BytewiseModel {
        let mut me = BytewiseModel {
            model: BitwiseModel::from_spec(spec),
            table: [0; 256],
        };
        me.fill_table();
        me
    }
}

impl Crc for BytewiseModel {
    fn crc<I: Iterator<Item = u8>>(&self, crc: u128, dat: I) -> u128 {
        let mut crc = crc as u128;
        let mut peek = dat.peekable();

        if peek.peek().is_none() {
            return self.model.init();
        }

        if self.model.reverse() {
            crc = reverse(crc, self.model.width());
        }

        if self.model.reflect() {
            crc &= ones(self.model.width());
            for byte in peek {
                crc = (crc >> 8) ^ self.table[(crc ^ (byte as u128)) as u8 as usize];
            }
        } else if self.model.width() < 8 {
            let shift = 8 - self.model.width();
            crc <<= shift;
            for byte in peek {
                crc = self.table[(crc ^ (byte as u128)) as usize];
            }
            crc >>= shift;
        } else {
            let shift = self.model.width() - 8;
            for byte in peek {
                crc = (crc << 8) ^ self.table[(((crc >> shift) ^ (byte as u128)) as u8) as usize];
            }
            crc &= ones(self.model.width());
        }

        if self.model.reverse() {
            crc = reverse(crc, self.model.width());
        }

        crc as _
    }
}
