<!--
Copyright 2021 The Crashpad Authors. All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
-->

# iOS Crashpad Overview Design

[TOC]

## iOS Limitations

Crashpad on other platforms captures exceptions out-of-process. The iOS sandbox,
however, restricts applications from delegating work to separate processes.
This limitation means Crashpad on iOS must combine the work of the handler and
the client into the same process as the main application.

## The Crashpad In-Process Handler

In-process handling comes with a number of limitations and difficulties. It is
not possible to catch the specific Mach exception `EXC_CRASH`, so certain groups
of crashes cannot be captured. This includes some major ones, like out-of-memory
crashes. This also introduces difficulties in capturing all the relevant crash
data and writing the minidump, as the process itself is in an unsafe state.

While handling an exception, the handler may not, for example:

 - Allocate memory.
 - Use libc, or most any library call.

While handling an exception, the handler may only:

 - Use audited syscalls.
 - access memory via `vm_read`.

In conjunction with Crashpad’s existing minidump writer and structural
limitations of the minidump format, it is not possible to write a minidump
immediately from the crash handler. Instead, an intermediate dump is written
when a handler would normally write a minidump (such as during an exception or a
forced dump without crashing). The intermediate dump file will be converted to
a minidump on the next run (or when the application decides it's safe to do so).

During Crashpad initialization, the handler gathers basic system information
and opens a pending intermediate dump adjacent to the Crashpad database.

## The Crashpad IntermediateDump Format

Due to the limitations of in-process handling, an intermediate dump file is
written during exceptions. The data is streamed to a file, which will be used to
generate a final minidump when appropriate.

The file format is similar to binary JSON, supporting keyed properties, maps and
arrays.

 - `Property` [key:int, length:int, value:intarray]
 - `StartMap` [key:int], followed by repeating Properties until `EndMap`
 - `StartArray` [key:int], followed by repeating Maps until `EndArray`
 - `EndMap`, `EndArray`, `EndDocument`

Similar to JSON, maps can contain other maps, arrays and properties.

## The Crashpad In-Process Client

Other Crashpad platforms handle exceptions and upload minidumps out-of-process.
On iOS, everything must happen in-process. Once started, the client will
automatically handle exceptions and capture the crashed process state in an
intermediate dump file. Converting that intermediate dump file into a minidump
is likely not safe to do from within a crashed process, and uploading a minidump
is definitely unsafe to do at crash time. Applications are expected to process
intermediate dumps into pending minidumps and begin processing pending
minidumps, possibly for upload, at suitable times following the next application
restart.


### `ProcessIntermediateDumps`
For performance and stability reasons applications may choose the correct time
to convert intermediate dumps, as well as append metadata to the pending
intermediate dumps. This is expected to happen during application startup, when
suitable. After converting, a minidump will be written to the Crashpad database,
similar to how other platforms write a minidump on exception handling. If
uploading is enabled, this minidump will also be immediately uploaded. New
intermediate dumps generated by exceptions or by
`CRASHPAD_SIMULATE_CRASH_AND_DEFER_PROCESSING` will not be processed until
the next call to `ProcessIntermediateDumps`. Conversely,
`CRASHPAD_SIMULATE_CRASH` can be called when the client has no performance or
stability concerns. In this case, intermediate dumps are automatically
converted to minidumps and immediately eligible for uploading.

### `StartProcessingPendingReports`
For similar reasons, applications may choose the correct time to begin uploading
pending reports, such as when ideal network conditions exist. By default,
clients start with uploading disabled. Applications should call this API when
it is determined that it is appropriate to do so (such as on a few seconds after
startup, or when network connectivity is appropriate).
