//
// Copyright 2018 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

#import "Service/Sources/EDOObject.h"

NS_ASSUME_NONNULL_BEGIN

/**
 *  The data object that proxies the block between processes.
 *
 *  When a block is being transferred between processes, it will be encoded into
 *  @c EDOBlockObject, and on the other end, it will be decoded and wrapped into
 *  @c EDOProxyBlock to capture and forward the invocation. The underlying block
 *  object isn't actually being transferred, but saved into @c EDOBlockObject for
 *  the remote invocation forwarding.
 */
@interface EDOBlockObject : EDOObject

/** The signature for the underlying block object. */
@property(nonatomic) NSString *signature;

/** If the block returns data structures. */
@property(nonatomic, assign) BOOL returnsStruct;

/** Check if the given @c object is a block. */
+ (BOOL)isBlock:(id)object;

/**
 *  Get the @c EDOBlockObject from the given @c block.
 *
 *  @note If the @c block is a proxied block, which will forward the invocation to the remote
 *        block, it is generated by @c EDOBlockObject. This retrieves the underlying
 *        @c EDOBlockObject.
 *
 *  @param block The block object.
 *  @return @c EDOBlockObject if it is a proxy block generated by @c EDOBlockObject;
 *          @c nil otherwise.
 */
+ (EDOBlockObject *_Nullable)EDOBlockObjectFromBlock:(id)block;

/** Get the method signature encoding from the given @c block. */
+ (char const *)signatureFromBlock:(id)block;

@end

NS_ASSUME_NONNULL_END
