//use ruma_identifiers::{RoomId, UserId};
//use serde::{Serialize, Deserialize};
use crackle_lib::{RoomAliasId, RoomId, RoomName, UserId};
use crossbeam_channel::{unbounded, TryRecvError};
use crossterm::{
    event::{self, DisableMouseCapture, EnableMouseCapture, Event, KeyCode},
    execute,
    terminal::{disable_raw_mode, enable_raw_mode, EnterAlternateScreen, LeaveAlternateScreen},
};
use scanpw::scanpw;
use serde::{Deserialize, Serialize};
use std::collections::BTreeMap;
use std::convert::TryFrom;
//use ruma_client_api::sync;
use std::{error::Error, io};
use tui::{
    backend::{Backend, CrosstermBackend},
    layout::{Constraint, Direction, Layout},
    style::{Color, Modifier, Style},
    text::{Span, Spans, Text},
    widgets::{Block, Borders, List, ListItem, Paragraph},
    Frame, Terminal,
};
//use std::time::Duration;
//const TIMEOUT: Duration = Duration::from_secs(3);
use unicode_width::UnicodeWidthStr;
//use crossbeam_utils::atomic::AtomicCell;

//static message_box_height: AtomicCell<u16> = AtomicCell::new(3);
#[derive(Serialize, Deserialize, Debug, Clone)]
struct CrackleConfig {
    matrix_url: String,
    auth_token: String,
    username: String,
    room: String,
}

enum Focus {
    MessageInput,
    JoinRoomInput,
    //RoomList,
    //PaticipantList,
    //History,
    Watching,
    //Login,
    //Register,
    //RoomSearch,
    RoomSelect,
}

/// App holds the state of the application
struct App {
    /// Current value of the input box
    input: String,
    /// Current input mode
    focus: Focus,
    /// History of recorded messages
    messages: BTreeMap<RoomId, Vec<String>>,
    host: String,
    server_name: String,
    user_name: String,
    password: Option<String>,
    selected_room: Option<RoomId>,
    room: String,
    rooms: BTreeMap<RoomId, Box<RoomName>>,
    current_room: Option<RoomId>,
    //using_config: bool,
    room_id_to_canonical_alias: BTreeMap<RoomId, RoomAliasId>,
    auth_token: Option<String>,
    joined_rooms: Vec<RoomId>, //message_box_height: u16,
}
/*
impl Default for App {
    fn default() -> App {
        App {
            input: String::new(),
            focus: Focus::Watching,
            messages: Vec::new(),

        }
    }
}
*/

impl App {
    pub async fn new(
        home_server_url: impl Into<String>,
        user_name: String,
        password: Option<String>,
        room: String,
        auth_token: Option<String>,
    ) -> Self {
        let mut hs_url = home_server_url.into();
        //let mut host: String;
        if !hs_url.starts_with("http://") && !hs_url.starts_with("https://") {
            hs_url = format!("https://{}", hs_url);
        }
        let hostparts = hs_url.split_once("://").unwrap();
        let server_name = hostparts
            .1
            .clone()
            .split(":")
            .map(|z| z.to_string())
            .collect::<Vec<String>>()[0]
            .clone();
        let username = match UserId::try_from(user_name.clone()) {
            Ok(user_id) => user_id.localpart().to_string(),
            Err(_) => user_name.clone(),
        };
        Self {
            input: String::new(),
            focus: Focus::Watching,
            messages: BTreeMap::new(),
            host: hs_url,
            server_name: server_name,
            user_name: username,
            password: password,
            selected_room: None,
            current_room: None,
            rooms: BTreeMap::new(),
            room_id_to_canonical_alias: BTreeMap::new(),
            joined_rooms: Vec::new(),
            auth_token: auth_token,
            room: room,
            //message_box_height: 3,
            //client: ruma_client::Client::builder().homeserver_url(home_server.into()).build::<ruma_client::http_client::Reqwest>().await.unwrap(),
        }
    }
}

#[tokio::main]
async fn main() -> Result<(), Box<dyn Error>> {
    //let app: App;
    let mut host: String;
    let mut username: String;
    let password: Option<String>;
    let auth_token: Option<String>;
    let mut room: String;
    if std::path::Path::new("./config.yaml").exists() {
        let config: CrackleConfig =
            serde_yaml::from_str(&std::fs::read_to_string("./config.yaml").unwrap()).unwrap();
        host = config.matrix_url;
        username = config.username;
        auth_token = Some(config.auth_token);
        room = config.room;
        password = None;
    } else {
        host = String::new();
        println!("Matrix URL: ");
        std::io::stdin().read_line(&mut host).unwrap();
        username = String::new();
        println!("Username: ");
        std::io::stdin().read_line(&mut username).unwrap();
        password = Some(scanpw!("Password: ").trim().to_string());
        room = String::new();
        println!("\nRoom: ");
        std::io::stdin().read_line(&mut room).unwrap();
        auth_token = None;
    }
    // setup terminal
    enable_raw_mode()?;
    let mut stdout = io::stdout();
    execute!(stdout, EnterAlternateScreen, EnableMouseCapture)?;
    let backend = CrosstermBackend::new(stdout);
    let mut terminal = Terminal::new(backend)?;

    // create app and run it
    let app = App::new(
        host.trim().to_string(),
        username.trim().to_string(),
        password,
        room.trim().to_string(),
        auth_token,
    )
    .await;
    let res = run_app(&mut terminal, app).await;

    // restore terminal
    disable_raw_mode()?;
    execute!(
        terminal.backend_mut(),
        LeaveAlternateScreen,
        DisableMouseCapture
    )?;
    terminal.show_cursor()?;

    if let Err(err) = res {
        println!("{:?}", err)
    }

    Ok(())
}
/*
async fn sync(client: ruma_client::Client::<ruma_client::http_client::Reqwest>, since: Option<String>, full_state: bool) -> sync::sync_events::v3::Response {
    let mut sync_request = sync::sync_events::v3::Request::new();
    sync_request.full_state = full_state;
    sync_request.timeout = Some(TIMEOUT);
    let mut sync_result = client.send_request(sync_request.clone()).await;
    while sync_result.is_err() {
        sync_result = client.send_request(sync_request.clone()).await;
    }
    //let sync_response: sync::sync_events::v3::Response = s
    sync_result.unwrap()
}
*/
#[derive(Debug, Clone)]
enum SyncMessage {
    Quit,
    //Response(sync::sync_events::v3::Response),
    SendMessage(RoomId, String, String),
    RoomsUpdated(
        Vec<RoomId>,
        BTreeMap<RoomId, Box<RoomName>>,
        BTreeMap<RoomId, RoomAliasId>,
    ),
    LeaveRoom(RoomId),
    JoinRoom(String),
}
#[allow(unused_assignments)]
async fn run_app<B: Backend>(terminal: &mut Terminal<B>, mut app: App) -> io::Result<()> {
    let mut client = crackle_lib::Client::new(
        app.host.clone(),
        app.user_name.clone(),
        app.server_name.clone(),
    );
    if let Some(pass) = app.password.clone() {
        client.login(pass);
    } else {
        client.set_auth_token(app.auth_token.as_ref().unwrap().clone());
    }
    let me = client.get_me();
    let (tx1, rx1) = unbounded::<SyncMessage>();
    let tx1_clone = tx1.clone();
    let (tx2, rx2) = unbounded::<SyncMessage>();
    //let rx_clone = rx.clone();
    let sm_room_id = client.room_id_from_alias(app.room.clone());
    let sm_room_id_clone = sm_room_id.clone();
    //let tx_clone = tx.clone();
    let networking = tokio::task::spawn(async move {
        let mut sync_results = client.sync().unwrap();
        //let mut known_room_list: Vec<RoomId> = Vec::new();
        client.extract_data(sync_results.clone());
        //let mut f = std::fs::File::create("sync.log").unwrap();
        //writeln!(f, "{:#?}", sync_results);
        //client.join_with_alias(app.room);
        //let mut sync_results = sync(client.clone(), None, true).await;
        //let mut since = sync_results.next_batch;
        //client.send_request(ruma_client_api::membership::join_room_by_id::v3::Request::new(&distrotube_room_id)).await.unwrap();
        client.join_room_id(sm_room_id_clone);
        loop {
            match rx2.try_recv() {
                Ok(SyncMessage::Quit) => break,
                Ok(SyncMessage::SendMessage(room_id, _user_id, message)) => {
                    client.send_message(room_id.try_into().unwrap(), message);
                }
                Ok(SyncMessage::LeaveRoom(room_id)) => {
                    client.leave_room_id(room_id);
                }
                Ok(SyncMessage::JoinRoom(room)) => {
                    client.join_with_alias(room);
                }
                Ok(_) => {}
                Err(TryRecvError::Empty) => {}
                Err(_) => break,
            }
            sync_results = client.sync().unwrap();
            let messages = client.extract_data(sync_results.clone()); //, client.clone().me);
            if client.room_list_updated() {
                //println!("{:#?}", client.data.joined_rooms);
                tx1.send(SyncMessage::RoomsUpdated(
                    client.data.joined_rooms.clone(),
                    client.data.room_to_name.clone(),
                    client.data.room_id_to_cananoical_alias.clone(),
                ))
                .unwrap();
            }
            for message in messages {
                let room_id = message.room_id;
                let user_id = message.sender.to_string();
                let msg = message.message.to_string();
                //if room_id.clone() == distrotube_room_id {
                tx1.send(SyncMessage::SendMessage(room_id, user_id, msg))
                    .unwrap();
                //}
            }
        }
    });
    loop {
        terminal.draw(|f| ui(f, &mut app))?;
        if event::poll(std::time::Duration::from_secs(0)).unwrap() {
            if let Event::Key(key) = event::read()? {
                match app.focus {
                    Focus::RoomSelect => {
                        //let num_rooms = app.rooms.len();
                        /*
                        let mut prev: RoomId;
                        let mut next: RoomId;
                        let last = app.rooms.iter().last().unwrap().0;
                        let first: RoomId;
                        for (index_num, (room_id, _room_name)) in app.rooms.iter().enumerate() {
                            if index_num == 0 {
                                first = room_id;
                            }
                        }
                        */
                        match key.code {
                            KeyCode::Up => {
                                if app.joined_rooms.len() > 1 {
                                    let last = app.joined_rooms.iter().last().unwrap();
                                    let first = app.joined_rooms.iter().nth(0).unwrap();
                                    let mut prev = last.clone();
                                    for (index_num, room_id) in app.joined_rooms.iter().enumerate()
                                    {
                                        if index_num == 0 {
                                            if first.clone() == app.selected_room.clone().unwrap() {
                                                app.selected_room = Some(last.clone());
                                                break;
                                            }
                                            prev = room_id.clone();
                                        } else {
                                            if room_id.clone() == app.selected_room.clone().unwrap()
                                            {
                                                app.selected_room = Some(prev);
                                                break;
                                            }
                                            prev = room_id.clone();
                                        } //else {
                                          //    app.selected_room = Some(last.clone());
                                          //}
                                    }
                                }
                            }
                            KeyCode::Down => {
                                if app.joined_rooms.len() > 1 {
                                    let last = app.joined_rooms.iter().last().unwrap();
                                    let first = app.joined_rooms.iter().nth(0).unwrap();
                                    let mut use_next = false;
                                    for room_id in app.joined_rooms.iter() {
                                        if use_next {
                                            app.selected_room = Some(room_id.clone());
                                            break;
                                        }
                                        /*else if index_num == 0 {
                                            //first = room_id.clone();
                                            if first.clone() == app.selected_room.clone().unwrap() {
                                                use_next = true;
                                            }
                                        }*/
                                        else if room_id != last {
                                            if room_id.clone() == app.selected_room.clone().unwrap()
                                            {
                                                use_next = true;
                                            }
                                        } else {
                                            app.selected_room = Some(first.clone());
                                        }
                                    }
                                }
                            }
                            KeyCode::Char('l') => {
                                tx2.send(SyncMessage::LeaveRoom(
                                    app.selected_room.clone().unwrap(),
                                ))
                                .unwrap();
                                //app.rooms.remove(app.selected_room.as_ref().unwrap());
                                app.selected_room = None;
                                //app.current_room = None;
                            }
                            KeyCode::Char('j') => {
                                app.focus = Focus::JoinRoomInput;
                            }
                            KeyCode::Enter => {
                                app.current_room = Some(app.selected_room.clone().unwrap());
                                app.selected_room = None;
                                app.focus = Focus::Watching;
                            }
                            KeyCode::Esc => {
                                app.selected_room = None;
                                app.focus = Focus::Watching;
                            }
                            _ => {}
                        }
                    }
                    Focus::Watching => match key.code {
                        KeyCode::Char('e') => {
                            app.focus = Focus::MessageInput;
                            //app.messages.push(String::from("test"));
                        }
                        KeyCode::Char('q') => {
                            tx2.send(SyncMessage::Quit).unwrap();
                            networking.await.unwrap();
                            return Ok(());
                        }
                        KeyCode::Char('r') => {
                            app.focus = Focus::RoomSelect;
                        }
                        _ => {
                            /*
                            if let Ok(msg) = rx1.try_recv() {
                                if let SyncMessage::SendMessage(room_id, user_id, message) = msg {
                                    app.messages.push(format!("[{}] <{}> {}", room_id, user_id, message));
                                }
                            }
                            */
                        }
                    },
                    Focus::MessageInput => match key.code {
                        KeyCode::Enter => {
                            let message: String = app.input.drain(..).collect();
                            //app.messages.push(message.clone());
                            if message != String::new() {
                                let sync_message = SyncMessage::SendMessage(
                                    app.current_room.clone().unwrap(),
                                    format!("{}", me),
                                    message,
                                );
                                tx1_clone.send(sync_message.clone()).unwrap();
                                tx2.send(sync_message.clone()).unwrap();
                            }
                        }
                        KeyCode::Char(c) => {
                            app.input.push(c);
                        }
                        KeyCode::Backspace => {
                            app.input.pop();
                        }
                        KeyCode::Esc => {
                            app.focus = Focus::Watching;
                        }
                        _ => {
                            /*
                            if let Ok(msg) = rx1.try_recv() {
                                if let SyncMessage::SendMessage(room_id, user_id, message) = msg {
                                    app.messages.push(format!("[{}] <{}> {}", room_id, user_id, message));
                                }
                            }
                            */
                        }
                    },
                    Focus::JoinRoomInput => match key.code {
                        KeyCode::Enter => {
                            let room: String = app.input.drain(..).collect();
                            //app.messages.push(message.clone());
                            let sync_message = SyncMessage::JoinRoom(room);
                            //tx1_clone.send(sync_message.clone()).unwrap();
                            tx2.send(sync_message.clone()).unwrap();
                            app.focus = Focus::Watching;
                        }
                        KeyCode::Char(c) => {
                            app.input.push(c);
                        }
                        KeyCode::Backspace => {
                            app.input.pop();
                        }
                        KeyCode::Esc => {
                            app.focus = Focus::Watching;
                        }
                        _ => {
                            /*
                            if let Ok(msg) = rx1.try_recv() {
                                if let SyncMessage::SendMessage(room_id, user_id, message) = msg {
                                    app.messages.push(format!("[{}] <{}> {}", room_id, user_id, message));
                                }
                            }
                            */
                        }
                    },
                }
            }
        } else {
            if let Ok(msg) = rx1.try_recv() {
                if let SyncMessage::SendMessage(room_id, user_id, message) = msg {
                    let mut msgs = app.messages[&room_id].clone();
                    if msgs.len() > 100 {
                        msgs.remove(0);
                    }
                    msgs.push(format!("<{}> {}", user_id, message));
                    app.messages.insert(room_id.clone(), msgs);
                } else if let SyncMessage::RoomsUpdated(
                    room_id_list,
                    room_to_name,
                    room_id_to_canonical_alias,
                ) = msg
                {
                    //app.rooms = room_id_list;
                    app.joined_rooms = room_id_list.clone();
                    app.rooms = room_to_name;
                    app.room_id_to_canonical_alias = room_id_to_canonical_alias;
                    for room_id in room_id_list {
                        if !app.messages.contains_key(&room_id) {
                            app.messages.insert(room_id.clone(), Vec::new());
                        }
                    }
                    /*
                    for (room_id, room_name) in room_id_list {
                        if app.current_room.is_none() {
                            app.current_room = Some(room_id.clone());
                        }
                        if !app.rooms.contains_key(&room_id.clone()) {
                            app.rooms.aaaaaa(room_id.clone(), room_name.clone());
                            app.messages.insert(room_id.clone(), Vec::new());
                        }
                    }
                    */
                }
            }
        }
    }
}

fn ui<B: Backend>(f: &mut Frame<B>, app: &mut App) {
    let chunks = Layout::default()
        .direction(Direction::Vertical)
        //.margin(2)
        .constraints(
            [
                Constraint::Length(1),
                Constraint::Min(1),
                Constraint::Length(3),
            ]
            .as_ref(),
        )
        .split(f.size());
    //message_box_height.store(chunks[1].height);
    let middle_chunk = Layout::default()
        .direction(Direction::Horizontal)
        .constraints([Constraint::Percentage(20), Constraint::Percentage(80)].as_ref())
        .split(chunks[1]);
    let (msg, style) = match app.focus {
        Focus::Watching => (
            vec![
                Span::raw("Press "),
                Span::styled("q", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to exit, "),
                Span::styled("e", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to start editing, or "),
                Span::styled("r", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to choose a room "),
            ],
            Style::default().add_modifier(Modifier::RAPID_BLINK),
        ),
        Focus::MessageInput => (
            vec![
                Span::raw("Press "),
                Span::styled("Esc", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to stop editing, "),
                Span::styled("Enter", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to record the message"),
            ],
            Style::default(),
        ),
        Focus::JoinRoomInput => (
            vec![
                Span::raw("Press "),
                Span::styled("Esc", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to stop editing, "),
                Span::styled("Enter", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to join the room"),
            ],
            Style::default(),
        ),
        Focus::RoomSelect => (
            vec![
                Span::raw("Press "),
                Span::styled("Esc", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to stop choosing a room with the Up and Down arrow keys, "),
                Span::styled("Enter", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to view the selected room.  You can press l to leave the room or j to join a new one"),
            ],
            Style::default(),
        ),
    };
    let mut text = Text::from(Spans::from(msg));
    text.patch_style(style);
    let help_message = Paragraph::new(text);
    f.render_widget(help_message, chunks[0]);

    let input = Paragraph::new(app.input.as_ref())
        .style(match app.focus {
            Focus::Watching => Style::default(),
            Focus::MessageInput => Style::default().fg(Color::Yellow),
            Focus::RoomSelect => Style::default(),
            Focus::JoinRoomInput => Style::default().fg(Color::Yellow),
        })
        .block(Block::default().borders(Borders::ALL).title("Input"));
    match app.focus {
        Focus::Watching =>
            // Hide the cursor. `Frame` does this by default, so we don't need to do anything here
            {}

        Focus::MessageInput => {
            // Make the cursor visible and ask tui-rs to put it at the specified coordinates after rendering
            f.set_cursor(
                // Put cursor past the end of the input text
                chunks[2].x + app.input.width() as u16 + 1,
                // Move one line down, from the border to the input line
                chunks[2].y + 1,
            )
        }
        Focus::JoinRoomInput => {
            // Make the cursor visible and ask tui-rs to put it at the specified coordinates after rendering
            f.set_cursor(
                // Put cursor past the end of the input text
                chunks[2].x + app.input.width() as u16 + 1,
                // Move one line down, from the border to the input line
                chunks[2].y + 1,
            )
        }
        Focus::RoomSelect => {}
    }
    //println!("{:#?}", app.joined_rooms);
    let room_list: Vec<ListItem> = app
        .joined_rooms
        .iter()
        //.enumerate()
        .map(|room_id| {
            let content: Vec<Spans>;
            if app.rooms.contains_key(&room_id) {
                content = vec![Spans::from(Span::raw(format!("{}", app.rooms[room_id])))];
            } else {
                content = vec![Spans::from(Span::raw(format!(
                    "{}",
                    app.room_id_to_canonical_alias[room_id]
                )))];
            }
            //let content: Vec<Span>;
            if app.current_room.is_none() {
                app.current_room = Some(room_id.clone());
            }
            let current_room = app.current_room.clone().unwrap();
            //if let Some(room) = current_room {
            /*
            if current_room == room_id.clone() {
                content = vec![Span::from(Span::styled(
                    format!("{}", room_name),
                    Style::default().add_modifier(Modifier::BOLD),
                ))];
            } else {
                content = vec![Span::from(Span::raw(format!("{}", room_name)))];
            }
            */
            //}
            let mut z = ListItem::new(content);
            let mut style = Style::default();
            if room_id.clone() == current_room {
                style = style.add_modifier(Modifier::BOLD);
                z = z.style(style);
            }
            match app.focus {
                Focus::RoomSelect => {
                    if app.selected_room.is_none() {
                        app.selected_room = Some(current_room);
                    }
                    if app.selected_room.clone().unwrap() == room_id.clone() {
                        z = z.style(style.add_modifier(Modifier::REVERSED));
                    }
                }
                _ => {}
            }
            z
        })
        .collect();
    let room_list =
        List::new(room_list).block(Block::default().borders(Borders::ALL).title("Room List"));
    let mut msg_list: Vec<ListItem> = Vec::new();
    if app.current_room.is_some() {
        let ms = app.messages[&app.current_room.clone().unwrap()].join("\n");
        //let width: i32 = middle_chunk[1].width.try_into().unwrap(); //message_box_height.load().try_into().unwrap();
        let lines: Vec<String> = textwrap::wrap(
            ms.as_str(),
            textwrap::Options::new((middle_chunk[1].width - 2).into()),
        )
        .iter()
        .map(|x| x.to_string())
        .collect();
        //.split("\n")
        //.map(|x| x.to_string())
        //.collect();
        /*)
        let messages_len: i32 = app.messages[&app.current_room.clone().unwrap()]
            .len()
            .try_into()
            .unwrap();
        */
        let messages_len: i32 = lines.len().try_into().unwrap();
        let height: i32 = middle_chunk[1].height.try_into().unwrap(); //message_box_height.load().try_into().unwrap();
        let mut cut_off = messages_len - height + 2;
        if cut_off <= 0 {
            cut_off = 0;
        }
        //msg_list = app.messages[app.current_room.as_ref().unwrap()][cut_off.try_into().unwrap()..]
        msg_list = lines[cut_off.try_into().unwrap()..]
            .iter()
            .enumerate()
            .map(|(_i, m)| {
                let content = vec![Spans::from(Span::raw(format!("{}", m)))];
                ListItem::new(content)
            })
            .collect();
    }
    let messages =
        List::new(msg_list).block(Block::default().borders(Borders::ALL).title("Messages"));
    f.render_widget(messages, middle_chunk[1]);
    f.render_widget(input, chunks[2]);
    f.render_widget(room_list, middle_chunk[0]);
    //println!("{}", chunks[1].height);
}
