/// A simple example demonstrating how to handle user input. This is
/// a bit out of the scope of the library as it does not provide any
/// input handling out of the box. However, it may helps some to get
/// started.
///
/// This is a very simple example:
///   * A input box always focused. Every character you type is registered
///   here
///   * Pressing Backspace erases a character
///   * Pressing Enter pushes the current input in the history of previous
///   messages
//use ruma_identifiers::{RoomId, UserId};
//use serde::{Serialize, Deserialize};
use scanpw::scanpw;
use serde::{Serialize, Deserialize};
use crackle_lib::{RoomId, UserId};
use crossbeam_channel::{unbounded, TryRecvError};
use crossterm::{
    event::{self, DisableMouseCapture, EnableMouseCapture, Event, KeyCode},
    execute,
    terminal::{disable_raw_mode, enable_raw_mode, EnterAlternateScreen, LeaveAlternateScreen},
};
//use std::convert::TryFrom;
//use ruma_client_api::sync;
use std::{error::Error, io};
use tui::{
    backend::{Backend, CrosstermBackend},
    layout::{Constraint, Direction, Layout},
    style::{Color, Modifier, Style},
    text::{Span, Spans, Text},
    widgets::{Block, Borders, List, ListItem, Paragraph},
    Frame, Terminal,
};
//use std::time::Duration;
//const TIMEOUT: Duration = Duration::from_secs(3);
use unicode_width::UnicodeWidthStr;
//use crossbeam_utils::atomic::AtomicCell;

//static message_box_height: AtomicCell<u16> = AtomicCell::new(3); 
#[derive(Serialize, Deserialize, Debug, Clone)]
struct CrackleConfig {
    matrix_url: String, 
    auth_token: String,
    username: String,
    room: String,
}

enum Focus {
    MessageInput,
    //RoomList,
    //PaticipantList,
    //History,
    Watching,
    //Login,
    //Register,
    //RoomSearch,
}

/// App holds the state of the application
struct App {
    /// Current value of the input box
    input: String,
    /// Current input mode
    focus: Focus,
    /// History of recorded messages
    messages: Vec<String>,
    host: String,
    server_name: String,
    user_name: String,
    password: Option<String>,
    room: String,
    //using_config: bool,
    auth_token: Option<String>,
    //message_box_height: u16,
}
/*
impl Default for App {
    fn default() -> App {
        App {
            input: String::new(),
            focus: Focus::Watching,
            messages: Vec::new(),

        }
    }
}
*/


impl App {
    pub async fn new(home_server_url: impl Into<String>, user_name: String,  password: Option<String>, room: String, auth_token: Option<String>) -> Self {
        let mut hs_url = home_server_url.into();
        //let mut host: String;
        if !hs_url.starts_with("http://") && !hs_url.starts_with("https://") {
            hs_url = format!("https://{}", hs_url);
        }
        let hostparts = hs_url.split_once("://").unwrap();
        let server_name = hostparts.1.clone().split(":").map(|z| z.to_string()).collect::<Vec<String>>()[0].clone();
        let username = match UserId::try_from(user_name.clone()) {
            Ok(user_id) => user_id.localpart().to_string(),
            Err(_) => user_name.clone(),
        };
        Self {
            input: String::new(),
            focus: Focus::Watching,
            messages: Vec::new(),
            host: hs_url,
            server_name: server_name,
            user_name: username,
            password: password,
            room: room,
            auth_token: auth_token, 
            //message_box_height: 3,
            //client: ruma_client::Client::builder().homeserver_url(home_server.into()).build::<ruma_client::http_client::Reqwest>().await.unwrap(),
        }
    }
}

#[tokio::main]
async fn main() -> Result<(), Box<dyn Error>> {
    //let app: App;
    let mut host: String;
    let mut username: String;
    let password: Option<String>;
    let auth_token: Option<String>;
    let mut room: String;
    if std::path::Path::new("./config.yaml").exists() {
        let config: CrackleConfig = serde_yaml::from_str(&std::fs::read_to_string("./config.yaml").unwrap()).unwrap();
        host = config.matrix_url;
        username = config.username;
        auth_token = Some(config.auth_token);
        room = config.room;
        password = None;
    } else {
        host = String::new();
        println!("Matrix URL: ");
        std::io::stdin().read_line(&mut host).unwrap();
        username = String::new();
        println!("Username: ");
        std::io::stdin().read_line(&mut username).unwrap();
        password = Some(scanpw!("Password: ").trim().to_string());
        room = String::new();
        println!("\nRoom: ");
        std::io::stdin().read_line(&mut room).unwrap();
        auth_token = None;
    }
    // setup terminal
    enable_raw_mode()?;
    let mut stdout = io::stdout();
    execute!(stdout, EnterAlternateScreen, EnableMouseCapture)?;
    let backend = CrosstermBackend::new(stdout);
    let mut terminal = Terminal::new(backend)?;

    // create app and run it
    let app = App::new(host.trim().to_string(), username.trim().to_string(), password, room.trim().to_string(), auth_token).await;
    let res = run_app(&mut terminal, app).await;

    // restore terminal
    disable_raw_mode()?;
    execute!(
        terminal.backend_mut(),
        LeaveAlternateScreen,
        DisableMouseCapture
    )?;
    terminal.show_cursor()?;

    if let Err(err) = res {
        println!("{:?}", err)
    }

    Ok(())
}
/*
async fn sync(client: ruma_client::Client::<ruma_client::http_client::Reqwest>, since: Option<String>, full_state: bool) -> sync::sync_events::v3::Response {
    let mut sync_request = sync::sync_events::v3::Request::new();
    sync_request.full_state = full_state;
    sync_request.timeout = Some(TIMEOUT);
    let mut sync_result = client.send_request(sync_request.clone()).await;
    while sync_result.is_err() {
        sync_result = client.send_request(sync_request.clone()).await;
    }
    //let sync_response: sync::sync_events::v3::Response = s
    sync_result.unwrap()
}
*/
#[derive(Debug, Clone)]
enum SyncMessage {
    Quit,
    //Response(sync::sync_events::v3::Response),
    SendMessage(RoomId, String, String),
}
#[allow(unused_assignments)]
async fn run_app<B: Backend>(terminal: &mut Terminal<B>, mut app: App) -> io::Result<()> {
    /*
    let mut client = ruma_client::Client::builder().homeserver_url("https://sonicrules.fun:8448/".parse().unwrap()).build::<ruma_client::http_client::Reqwest>().await.unwrap();//.access_token(Some(std::fs::read_to_string("auth_token.txt").unwrap().trim().to_string())).build();
    
    let mut login_result = client.log_in("@crackle_client:sonicrules.fun", "50nicru1e5", None, None).await;
    while login_result.is_err() {
        login_result = client.log_in("@crackle_client:sonicrules.fun", "50nicru1e5", None, None).await;
    }
    //let mut sync_request = sync::sync_events::v3::Request::new();
    //sync_request.full_state = true;
    //let mut sync_result = client.send_request(sync_request.clone()).await;
    //while sync_result.is_err() {
    //    sync_result = client.send_request(sync_request.clone()).await;
    //}
    //let sync_response: sync::sync_events::v3::Response = sync_result.unwrap();
    */
    //let distrotube_room_id = client.send_request(ruma_client_api::alias::get_alias::v3::Request::new(ruma_identifiers::room_alias_id!("#distrotube:matrix.org"))).await.unwrap().room_id;
    //println!("'{}'", app.server_name.clone());
    let mut client = crackle_lib::Client::new(app.host.clone(), app.user_name.clone(), app.server_name.clone());
    if let Some(pass) = app.password.clone() {
        client.login(pass);
    } else {
        client.set_auth_token(app.auth_token.as_ref().unwrap().clone());
    }
    
    let me = client.get_me();
    let (tx1, rx1) = unbounded::<SyncMessage>();
    let tx1_clone = tx1.clone();
    let (tx2, rx2) = unbounded::<SyncMessage>();
    //let rx_clone = rx.clone();
    let sm_room_id = client.room_id_from_alias(app.room.clone());
    let sm_room_id_clone = sm_room_id.clone();
    //let tx_clone = tx.clone();
    let networking = tokio::task::spawn(async move {
        let mut sync_results = client.sync().unwrap();
        //client.join_with_alias(app.room);
        //let mut sync_results = sync(client.clone(), None, true).await;
        //let mut since = sync_results.next_batch;
        //client.send_request(ruma_client_api::membership::join_room_by_id::v3::Request::new(&distrotube_room_id)).await.unwrap();
        client.join_room_id(sm_room_id_clone);
        loop {
            match rx2.try_recv() {
                Ok(SyncMessage::Quit) => break,
                Ok(SyncMessage::SendMessage(room_id, _user_id, message)) => {
                    client.send_message(room_id.try_into().unwrap(), message);
                }
                Err(TryRecvError::Empty) => {},
                Err(_) => break,
            }
            sync_results = client.sync().unwrap();
            let messages = client.extract_messages(sync_results.clone());//, client.clone().me);
            for message in messages {
                let room_id = message.room_id;
                let user_id = message.sender.to_string();
                let msg = message.message.to_string();
                //if room_id.clone() == distrotube_room_id {
                tx1.send(SyncMessage::SendMessage(room_id, user_id, msg)).unwrap();
                //}
            }
        }        
    });
    loop {
        terminal.draw(|f| ui(f, &app))?;
        if event::poll(std::time::Duration::from_secs(0)).unwrap() {
            if let Event::Key(key) = event::read()? {
                match app.focus {
                    Focus::Watching => match key.code {
                        KeyCode::Char('e') => {
                            app.focus = Focus::MessageInput;
                            //app.messages.push(String::from("test"));
                        }
                        KeyCode::Char('q') => {
                            tx2.send(SyncMessage::Quit).unwrap();
                            networking.await.unwrap();
                            return Ok(());
                        }
                        _ => {
                            /*
                            if let Ok(msg) = rx1.try_recv() {
                                if let SyncMessage::SendMessage(room_id, user_id, message) = msg {
                                    app.messages.push(format!("[{}] <{}> {}", room_id, user_id, message));
                                }
                            }
                            */
                        }
                    },
                    Focus::MessageInput => match key.code {
                        KeyCode::Enter => {
                            let message: String = app.input.drain(..).collect();
                            //app.messages.push(message.clone());
                            let sync_message = SyncMessage::SendMessage(sm_room_id.clone(), format!("{}", me), message);
                            tx1_clone.send(sync_message.clone()).unwrap();
                            tx2.send(sync_message.clone()).unwrap();
                            
                        }
                        KeyCode::Char(c) => {
                            app.input.push(c);
                        }
                        KeyCode::Backspace => {
                            app.input.pop();
                        }
                        KeyCode::Esc => {
                            app.focus = Focus::Watching;
                        }
                        _ => {
                            /*
                            if let Ok(msg) = rx1.try_recv() {
                                if let SyncMessage::SendMessage(room_id, user_id, message) = msg {
                                    app.messages.push(format!("[{}] <{}> {}", room_id, user_id, message));
                                }
                            }
                            */
                        }
                    },
                }
            }
        } else {
            if let Ok(msg) = rx1.try_recv() {
                if let SyncMessage::SendMessage(room_id, user_id, message) = msg {
                    if app.messages.len() > 100 {
                        app.messages.remove(0);
                    }
                    app.messages.push(format!("[{}] <{}> {}", room_id, user_id, message));
                }
            }
        }
    }
}

fn ui<B: Backend>(f: &mut Frame<B>, app: &App) {
    let chunks = Layout::default()
        .direction(Direction::Vertical)
        //.margin(2)
        .constraints(
            [
                Constraint::Length(1),
                Constraint::Min(1),
                Constraint::Length(3),
            ]
            .as_ref(),
        )
        .split(f.size());
    //message_box_height.store(chunks[1].height);
    let (msg, style) = match app.focus {
        Focus::Watching => (
            vec![
                Span::raw("Press "),
                Span::styled("q", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to exit, "),
                Span::styled("e", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to start editing."),
            ],
            Style::default().add_modifier(Modifier::RAPID_BLINK),
        ),
        Focus::MessageInput => (
            vec![
                Span::raw("Press "),
                Span::styled("Esc", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to stop editing, "),
                Span::styled("Enter", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to record the message"),
            ],
            Style::default(),
        ),
    };
    let mut text = Text::from(Spans::from(msg));
    text.patch_style(style);
    let help_message = Paragraph::new(text);
    f.render_widget(help_message, chunks[0]);

    let input = Paragraph::new(app.input.as_ref())
        .style(match app.focus {
            Focus::Watching => Style::default(),
            Focus::MessageInput => Style::default().fg(Color::Yellow),
        })
        .block(Block::default().borders(Borders::ALL).title("Input"));
    match app.focus {
        Focus::Watching =>
            // Hide the cursor. `Frame` does this by default, so we don't need to do anything here
            {}

        Focus::MessageInput => {
            // Make the cursor visible and ask tui-rs to put it at the specified coordinates after rendering
            f.set_cursor(
                // Put cursor past the end of the input text
                chunks[2].x + app.input.width() as u16 + 1,
                // Move one line down, from the border to the input line
                chunks[2].y + 1,
            )
        }
    }
    let messages_len: i32 = app.messages.len().try_into().unwrap();
    let height: i32 = chunks[1].height.try_into().unwrap(); //message_box_height.load().try_into().unwrap();
    let mut cut_off = messages_len - height + 2;
    if cut_off <= 0 {
        cut_off = 0;
    }
    let messages: Vec<ListItem> = app
        .messages[cut_off.try_into().unwrap()..]
        .iter()
        .enumerate()
        .map(|(_i, m)| {
            let content = vec![Spans::from(Span::raw(format!("{}", m)))];
            ListItem::new(content)
        })
        .collect();
    let messages =
        List::new(messages).block(Block::default().borders(Borders::ALL).title("Messages"));
    f.render_widget(messages, chunks[1]);
    f.render_widget(input, chunks[2]);
    //println!("{}", chunks[1].height);
}
/*
fn login_ui<B: Backend>(f: &mut Frame<B>, app: &App) {
    let chunks = Layout::default()
        .direction(Direction::Vertical)
        //.margin(2)
        .constraints(
            [
                Constraint::Length(1),
                Constraint::Length(3),
                Constraint::Length(3),
                Constraint::Length(3),
            ]
            .as_ref(),
        )
        .split(f.size());
    //message_box_height.store(chunks[1].height);
    let (msg, style) = match app.focus {
        Focus::Watching => (
            vec![
                Span::raw("Press "),
                Span::styled("q", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to exit, "),
                Span::styled("e", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to start editing."),
            ],
            Style::default().add_modifier(Modifier::RAPID_BLINK),
        ),
        Focus::MessageInput => (
            vec![
                Span::raw("Press "),
                Span::styled("Esc", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to stop editing, "),
                Span::styled("Enter", Style::default().add_modifier(Modifier::BOLD)),
                Span::raw(" to record the message"),
            ],
            Style::default(),
        ),
    };
    let mut text = Text::from(Spans::from(msg));
    text.patch_style(style);
    let help_message = Paragraph::new(text);
    f.render_widget(help_message, chunks[0]);

    let input = Paragraph::new(app.input.as_ref())
        .style(match app.focus {
            Focus::Watching => Style::default(),
            Focus::MessageInput => Style::default().fg(Color::Yellow),
        })
        .block(Block::default().borders(Borders::ALL).title("Input"));
    match app.focus {
        Focus::Watching =>
            // Hide the cursor. `Frame` does this by default, so we don't need to do anything here
            {}

        Focus::MessageInput => {
            // Make the cursor visible and ask tui-rs to put it at the specified coordinates after rendering
            f.set_cursor(
                // Put cursor past the end of the input text
                chunks[2].x + app.input.width() as u16 + 1,
                // Move one line down, from the border to the input line
                chunks[2].y + 1,
            )
        }
    }
    let messages_len: i32 = app.messages.len().try_into().unwrap();
    let height: i32 = chunks[1].height.try_into().unwrap(); //message_box_height.load().try_into().unwrap();
    let mut cut_off = messages_len - height + 2;
    if cut_off <= 0 {
        cut_off = 0;
    }
    let messages: Vec<ListItem> = app
        .messages[cut_off.try_into().unwrap()..]
        .iter()
        .enumerate()
        .map(|(_i, m)| {
            let content = vec![Spans::from(Span::raw(format!("{}", m)))];
            ListItem::new(content)
        })
        .collect();
    let messages =
        List::new(messages).block(Block::default().borders(Borders::ALL).title("Messages"));
    f.render_widget(messages, chunks[1]);
    f.render_widget(input, chunks[2]);
    //println!("{}", chunks[1].height);
}
*/