use ruma::events::*;
pub use ruma::{
    api::client::r0::sync::sync_events::*, identifiers::RoomName, presence::PresenceState,
    serde::Raw, DeviceKeyAlgorithm, EventId, RoomAliasId, RoomId, UInt, UserId,
};
use serde::{Deserialize, Serialize};
use serde_json::{json, Value};
use std::collections::BTreeMap;
use std::time::Duration;
use ureq::{Agent, AgentBuilder, OrAnyStatus};
use uuid::Uuid;
const BASE: &str = "/_matrix/client/v3/";
#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct StorageData {
    pub auth_token: String,
    pub room_to_aliases: BTreeMap<RoomId, Vec<RoomAliasId>>,
    pub alias_to_room: BTreeMap<RoomAliasId, RoomId>,
    pub joined_rooms: Vec<RoomId>,
    pub room_id_to_cananoical_alias: BTreeMap<RoomId, RoomAliasId>,
    pub room_to_name: BTreeMap<RoomId, Box<RoomName>>,
    pub room_to_direct_name: BTreeMap<RoomId, String>,
    pub recent_event_ids: Vec<EventId>,
    pub last_sent_messages: Vec<(RoomId, String)>,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct EventResponse {
    /// The batch token to supply in the `since` param of the next `/sync` request.
    pub next_batch: String,

    /// Updates to rooms.
    #[serde(default, skip_serializing_if = "Rooms::is_empty")]
    pub rooms: Rooms,

    /// Updates to the presence status of other users.
    #[serde(default, skip_serializing_if = "Presence::is_empty")]
    pub presence: Presence,

    /// The global private data created by this user.
    #[serde(default, skip_serializing_if = "GlobalAccountData::is_empty")]
    pub account_data: GlobalAccountData,

    /// Messages sent directly between devices.
    #[serde(default, skip_serializing_if = "ToDevice::is_empty")]
    pub to_device: ToDevice,

    /// Information on E2E device updates.
    ///
    /// Only present on an incremental sync.
    #[serde(default, skip_serializing_if = "DeviceLists::is_empty")]
    pub device_lists: DeviceLists,

    /// For each key algorithm, the number of unclaimed one-time keys
    /// currently held on the server for a device.
    #[serde(default, skip_serializing_if = "BTreeMap::is_empty")]
    pub device_one_time_keys_count: BTreeMap<DeviceKeyAlgorithm, UInt>,

    /// For each key algorithm, the number of unclaimed one-time keys
    /// currently held on the server for a device.
    ///
    /// The presence of this field indicates that the server supports
    /// fallback keys.
    #[serde(rename = "org.matrix.msc2732.device_unused_fallback_key_types")]
    pub device_unused_fallback_key_types: Option<Vec<DeviceKeyAlgorithm>>,
}
#[derive(Debug, Clone)]
pub struct Client {
    agent: Agent,
    pub me: UserId,
    since: Option<String>,
    host: String,
    pub data: StorageData,
    room_list_updated: bool,
}

impl Client {
    pub fn new(
        host: impl Into<String>,
        username: impl Into<String>,
        server_name: impl Into<String>,
    ) -> Self {
        Self {
            host: host.into().trim_end_matches("/").to_string(),
            me: UserId::try_from(format!("@{}:{}", username.into(), server_name.into()).as_str())
                .unwrap(),
            agent: AgentBuilder::new()
                .timeout_read(Duration::from_secs(3))
                .build(),
            since: None,
            data: StorageData::default(),
            room_list_updated: false,
        }
    }
    pub fn room_list_updated(&mut self) -> bool {
        if self.room_list_updated {
            self.room_list_updated = false;
            return true;
        }
        false
    }
    pub fn set_auth_token(&mut self, auth_token: String) {
        let should_req_well_known = match self.host.split(":").last().unwrap().parse::<usize>() {
            Ok(port) => {
                if port != 443 {
                    false
                } else {
                    true
                }
            }
            Err(_) => true,
        };
        if should_req_well_known {
            if let Ok(z) =
                ureq::get(&format!("{}/.well-known/matrix/client", self.host.clone())).call()
            {
                let resp: Value = z.into_json().unwrap();
                let base_url: String = resp["m.homeserver"]["base_url"]
                    .as_str()
                    .unwrap()
                    .to_string();
                self.host = base_url.trim_end_matches("/").to_string();
            }
        }
        self.data.auth_token = auth_token;
    }
    pub fn login(&mut self, password: String) -> Result<(), Value> {
        let should_req_well_known = match self.host.split(":").last().unwrap().parse::<usize>() {
            Ok(port) => {
                if port != 443 {
                    false
                } else {
                    true
                }
            }
            Err(_) => true,
        };
        if should_req_well_known {
            if let Ok(z) =
                ureq::get(&format!("{}/.well-known/matrix/client", self.host.clone())).call()
            {
                let resp: Value = z.into_json().unwrap();
                let base_url: String = resp["m.homeserver"]["base_url"]
                    .as_str()
                    .unwrap()
                    .to_string();
                self.host = base_url.trim_end_matches("/").to_string();
            }
        }
        let content = json!({
            "type": "m.login.password",
            "identifier": {
                "type": "m.id.user",
                "user": self.me.localpart()
            },
            "password": password
        });
        let val: Value = self
            .post_r0("login", content, None)
            .unwrap()
            .into_json()
            .unwrap();
        //println!("{:#}", val);
        if !val.clone().as_object().unwrap().contains_key("auth_token") {
            return Err(val);
        }
        self.data.auth_token = val["access_token"].as_str().unwrap().to_string();
        self.me = UserId::try_from(val["user_id"].as_str().unwrap()).unwrap();
        Ok(())
    }
    pub fn get_me(&self) -> UserId {
        self.me.clone()
    }
    fn post(
        &self,
        url_ext: impl Into<String>,
        content: Value,
        query_pairs: Option<Vec<(String, String)>>,
    ) -> Result<ureq::Response, ureq::Transport> {
        let mut r = self
            .agent
            .post(format!("{}{}{}", self.host, BASE, url_ext.into()).as_str())
            .set(
                "Authorization",
                format!("Bearer {}", self.data.auth_token).as_str(),
            );
        if let Some(pairs) = query_pairs {
            for pair in pairs {
                r = r.query(pair.0.as_str(), pair.1.as_str());
            }
        }
        //let x = r.send_json(content).or_any_status();
        //println!("{:#?}", x);
        //x
        let mut resp = r.clone().send_json(content.clone()).or_any_status();
        while let Err(_e) = resp {
            resp = r.clone().send_json(content.clone()).or_any_status();
            //eprintln!("{}", e);
        }
        resp
    }

    fn post_r0(
        &self,
        url_ext: impl Into<String>,
        content: Value,
        query_pairs: Option<Vec<(String, String)>>,
    ) -> Result<ureq::Response, ureq::Transport> {
        let mut r = self
            .agent
            .post(
                format!(
                    "{}{}{}",
                    self.host,
                    BASE.replace("v3", "r0"),
                    url_ext.into()
                )
                .as_str(),
            )
            .set(
                "Authorization",
                format!("Bearer {}", self.data.auth_token).as_str(),
            );
        if let Some(pairs) = query_pairs {
            for pair in pairs {
                r = r.query(pair.0.as_str(), pair.1.as_str());
            }
        }
        //let x = r.send_json(content).or_any_status();
        //println!("{:#?}", x);
        //x
        let mut resp = r.clone().send_json(content.clone()).or_any_status();
        while let Err(_e) = resp {
            resp = r.clone().send_json(content.clone()).or_any_status();
            //eprintln!("{}", e);
        }
        resp
    }
    fn put(
        &self,
        url_ext: impl Into<String>,
        content: Value,
        query_pairs: Option<Vec<(String, String)>>,
    ) -> Result<ureq::Response, ureq::Transport> {
        let mut r = self
            .agent
            .put(format!("{}{}{}", self.host, BASE, url_ext.into()).as_str())
            .set(
                "Authorization",
                format!("Bearer {}", self.data.auth_token).as_str(),
            );
        if let Some(pairs) = query_pairs {
            for pair in pairs {
                r = r.query(pair.0.as_str(), pair.1.as_str());
            }
        }

        let mut resp = r.clone().send_json(content.clone()).or_any_status();
        while let Err(_e) = resp {
            resp = r.clone().send_json(content.clone()).or_any_status();
            //eprintln!("{}", e);
        }
        resp
    }
    fn get(
        &self,
        url_ext: impl Into<String>,
        query_pairs: Option<Vec<(String, String)>>,
    ) -> Result<ureq::Response, ureq::Transport> {
        let mut r = self
            .agent
            .get(format!("{}{}{}", self.host, BASE, url_ext.into()).as_str())
            .set(
                "Authorization",
                format!("Bearer {}", self.data.auth_token).as_str(),
            );
        if let Some(pairs) = query_pairs {
            for pair in pairs {
                r = r.query(pair.0.as_str(), pair.1.as_str());
            }
        }
        let mut resp = r.clone().call().or_any_status();
        while let Err(_e) = resp {
            resp = r.clone().call().or_any_status();
            //eprintln!("{}", e);
        }
        //println!("{:#?}", resp);
        resp
    }
    pub fn join_room_id(&self, room_id: ruma::RoomId) -> Value {
        let rid = room_id.to_string();
        let room = urlencoding::encode(rid.as_str());
        let content = json!({});
        self.post(format!("rooms/{}/join", room).as_str(), content, None)
            .unwrap()
            .into_json()
            .unwrap()
    }
    pub fn sync(&mut self) -> Result<EventResponse, Value> {
        let req: ureq::Response;
        if let Some(start) = self.since.clone() {
            req = self
                .get(
                    "sync",
                    Some(vec![
                        ("since".to_string(), start),
                        ("full_state".to_string(), "false".to_string()),
                        (
                            "set_presence".to_string(),
                            format!("{}", PresenceState::Online),
                        ),
                        ("timeout".to_string(), "3000".to_string()),
                    ]),
                )
                .unwrap();
        } else {
            req = self
                .get(
                    "sync",
                    Some(vec![
                        ("full_state".to_string(), "true".to_string()),
                        (
                            "set_presence".to_string(),
                            format!("{}", PresenceState::Online),
                        ),
                        ("timeout".to_string(), "3000".to_string()),
                    ]),
                )
                .unwrap();
        }
        let val: Value = req.into_json().unwrap();
        if val.clone().as_object().unwrap().contains_key("error") {
            Err(val)
        } else {
            let x: EventResponse =
                serde_json::from_str::<Raw<EventResponse>>(val.to_string().as_str())
                    .unwrap()
                    .deserialize()
                    .unwrap();
            self.since = Some(x.clone().next_batch);
            Ok(x)
        }
    }
    pub fn join_with_alias(&self, alias: impl Into<String>) {
        let room_alias = alias.into();
        let this_room_id_val: Value = self
            .get(
                format!(
                    "directory/room/{}",
                    urlencoding::encode(room_alias.as_str())
                        .to_string()
                        .as_str()
                ),
                None,
            )
            .unwrap()
            .into_json()
            .unwrap();
        let room_id =
            ruma::RoomId::try_from(this_room_id_val["room_id"].as_str().unwrap()).unwrap();
        self.join_room_id(room_id);
    }
    pub fn leave_room_id(&mut self, room_id: RoomId) -> Value {
        let rid = room_id.to_string();
        let room = urlencoding::encode(rid.as_str());
        let content = json!({});
        let output = self
            .post(format!("rooms/{}/leave", room).as_str(), content, None)
            .unwrap()
            .into_json()
            .unwrap();
        self.data.joined_rooms.retain(|x| x != &room_id);
        self.room_list_updated = true;
        output
    }
    pub fn room_id_from_alias(&self, alias: impl Into<String>) -> RoomId {
        let room_alias = alias.into();
        let this_room_id_val: Value = self
            .get(
                format!(
                    "directory/room/{}",
                    urlencoding::encode(room_alias.as_str())
                        .to_string()
                        .as_str()
                ),
                None,
            )
            .unwrap()
            .into_json()
            .unwrap();
        ruma::RoomId::try_from(this_room_id_val["room_id"].as_str().unwrap()).unwrap()
    }
    pub fn send_message(&mut self, room_id: RoomId, message: impl Into<String>) {
        let msg = message.into();
        let txid = Uuid::new_v4()
            .to_simple()
            .encode_lower(&mut Uuid::encode_buffer())
            .to_string();
        self.put(
            format!("rooms/{}/send/m.room.message/{}", room_id.clone(), txid).as_str(),
            json!({"body": msg, "msgtype": "m.text"}),
            None,
        )
        .unwrap();
        self.data.last_sent_messages.push((room_id.clone(), msg));
    }

    pub fn extract_data(&mut self, sync_results: EventResponse) -> Vec<MessageInfo> {
        let mut messages: Vec<MessageInfo> = Vec::new();
        let x = sync_results.rooms.join;
        for (room_id, joined_room) in x.iter() {
            if !self.data.joined_rooms.contains(room_id) {
                self.data.joined_rooms.push(room_id.clone());
                self.room_list_updated = true;
            }
            let mut room_members: Vec<String> = Vec::new();
            let mut is_direct = false;
            for event in joined_room.state.events.clone() {
                if let Ok(ev) = event.deserialize() {
                    if let ruma::events::AnySyncStateEvent::RoomAliases(room_aliases_event) =
                        ev.clone()
                    {
                        let aliases = room_aliases_event.content.aliases;
                        for alias in aliases {
                            if !self.data.alias_to_room.contains_key(&alias) {
                                self.data
                                    .alias_to_room
                                    .insert(alias.clone(), room_id.clone())
                                    .unwrap();
                            }
                            if !self.data.room_to_aliases.contains_key(room_id) {
                                self.data
                                    .room_to_aliases
                                    .insert(room_id.clone(), vec![alias])
                                    .unwrap();
                            } else {
                                if !self.data.room_to_aliases[room_id].contains(&alias) {
                                    let mut current = self.data.room_to_aliases[room_id].clone();
                                    current.push(alias.clone());
                                    self.data
                                        .room_to_aliases
                                        .insert(room_id.clone(), current)
                                        .unwrap();
                                }
                            }
                        }
                    }
                    if let ruma::events::AnySyncStateEvent::RoomName(room_name_event) = ev.clone() {
                        if let Some(room_name) = room_name_event.content.name {
                            self.data.room_to_name.insert(room_id.clone(), room_name);
                        }
                    } else if let ruma::events::AnySyncStateEvent::RoomCanonicalAlias(
                        room_canonical_alias_event,
                    ) = ev.clone()
                    {
                        if let Some(alias) = room_canonical_alias_event.content.alias {
                            self.data
                                .room_id_to_cananoical_alias
                                .insert(room_id.clone(), alias);
                        }
                    } else if let ruma::events::AnySyncStateEvent::RoomMember(room_member_event) =
                        ev.clone()
                    {
                        if let Some(display_name) = room_member_event.content.displayname {
                            room_members.push(display_name);
                        }
                        if let Some(direct) = room_member_event.content.is_direct {
                            if !is_direct && direct == true {
                                room_members.pop();
                                is_direct = true;
                            }
                        }
                    }
                }
            }
            if is_direct {
                self.data
                    .room_to_direct_name
                    .insert(room_id.clone(), room_members[0].clone());
            }
            for message_info in joined_room.timeline.events.iter().filter_map(|m| {
                if let Ok(ruma::events::AnySyncRoomEvent::Message(message_event)) =
                    m.deserialize_as()
                {
                    let msg_event_id = message_event.event_id();
                    if self.data.recent_event_ids.contains(msg_event_id) {
                        return None;
                    }
                    if !self.data.recent_event_ids.contains(msg_event_id) {
                        self.data.recent_event_ids.push(msg_event_id.clone());
                    }
                    if self.data.recent_event_ids.len() > 500 {
                        self.data.recent_event_ids.remove(0);
                    }
                    if let ruma::events::AnySyncMessageEvent::RoomMessage(room_message) =
                        message_event
                    {
                        let sender = room_message.sender;
                        let message: String;
                        let is_emote: bool;
                        if let room::message::MessageType::Text(text_message_event_content) =
                            room_message.content.msgtype
                        {
                            message = text_message_event_content.body;
                            is_emote = false;
                        } else if let room::message::MessageType::Emote(
                            emote_message_event_content,
                        ) = room_message.content.msgtype
                        {
                            message = emote_message_event_content.body;
                            is_emote = true;
                        } else {
                            return None;
                        }
                        if sender == self.me {
                            //return None;
                            let look_for = &(room_id.clone(), message.clone());
                            if self.data.last_sent_messages.contains(&look_for) {
                                self.data.last_sent_messages.retain(|q| q != look_for);
                                return None;
                            }
                        }
                        return Some(MessageInfo {
                            message: message.clone(),
                            sender: sender,
                            room_id: room_id.clone(),
                            is_emote: is_emote,
                        });
                    }
                }
                None
            }) {
                messages.push(message_info);
            }
        }
        messages
    }
}

pub struct MessageInfo {
    pub room_id: RoomId,
    pub sender: UserId,
    pub message: String,
    pub is_emote: bool,
}
