use uuid::Uuid;
pub use ruma::{DeviceKeyAlgorithm, RoomId, UserId, UInt, RoomAliasId,  presence::PresenceState, serde::Raw, api::client::r0::sync::sync_events::*};
use std::collections::BTreeMap;
use ureq::{Agent, AgentBuilder, OrAnyStatus};
use serde::{Serialize, Deserialize};
use serde_json::{json, Value};
use std::time::{Duration};
use ruma::events::*;
const BASE: &str = "/_matrix/client/v3/";
//const TIME_TO_IDLE: Duration = Duration::from_secs(3 * 60);
#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct StorageData {
    pub auth_token: String,
    pub room_to_aliases: BTreeMap<RoomId, Vec<RoomAliasId>>,
}

#[derive(Clone, Debug, Serialize, Deserialize, Default)]
pub struct EventResponse {
    /// The batch token to supply in the `since` param of the next `/sync` request.
    pub next_batch: String,

    /// Updates to rooms.
    #[serde(default, skip_serializing_if = "Rooms::is_empty")]
    pub rooms: Rooms,

    /// Updates to the presence status of other users.
    #[serde(default, skip_serializing_if = "Presence::is_empty")]
    pub presence: Presence,

    /// The global private data created by this user.
    #[serde(default, skip_serializing_if = "GlobalAccountData::is_empty")]
    pub account_data: GlobalAccountData,

    /// Messages sent directly between devices.
    #[serde(default, skip_serializing_if = "ToDevice::is_empty")]
    pub to_device: ToDevice,

    /// Information on E2E device updates.
    ///
    /// Only present on an incremental sync.
    #[serde(default, skip_serializing_if = "DeviceLists::is_empty")]
    pub device_lists: DeviceLists,

    /// For each key algorithm, the number of unclaimed one-time keys
    /// currently held on the server for a device.
    #[serde(default, skip_serializing_if = "BTreeMap::is_empty")]
    pub device_one_time_keys_count: BTreeMap<DeviceKeyAlgorithm, UInt>,

    /// For each key algorithm, the number of unclaimed one-time keys
    /// currently held on the server for a device.
    ///
    /// The presence of this field indicates that the server supports
    /// fallback keys.
    #[serde(rename = "org.matrix.msc2732.device_unused_fallback_key_types")]
    pub device_unused_fallback_key_types: Option<Vec<DeviceKeyAlgorithm>>,
}
#[derive(Debug, Clone)]
pub struct Client {
    agent: Agent,
    pub me: UserId,
    //server: String,
    since: Option<String>,
    host: String,
    //auth_token: String,
    //pub joined_rooms: Vec<RoomId>,
    data: StorageData,
    //username: String,
}

impl Client {
    pub fn new(host: impl Into<String>, username: impl Into<String>, server_name: impl Into<String>) -> Self {
        Self {
            //auth_token: String::new(),
            host: host.into().trim_end_matches("/").to_string(),
            me: UserId::try_from(format!("@{}:{}",  username.into(), server_name.into()).as_str()).unwrap(),
            agent: AgentBuilder::new().timeout_read(Duration::from_secs(3)).build(),
            //joined_rooms: Vec::new(),
            since: None,
            data: StorageData::default(),
            //username: username.into(),
            //server: server_name.into(),
            //last_event: EventResponse::default(),
        }
    }
    pub fn login(&mut self, password: String) {
        let should_req_well_known = match self.host.split(":").last().unwrap().parse::<usize>() {
            Ok(port) => {
                if port != 443 {
                    false
                } else {
                    true
                }
            },
            Err(_) => {
                true
            }
        };
        if should_req_well_known {
            if let Ok(z) = ureq::get(&self.host.clone()).call() {
                let resp: Value = z.into_json().unwrap();
                let base_url: String = resp["m.homeserver"]["base_url"].as_str().unwrap().to_string();
                self.host = base_url.trim_end_matches("/").to_string();
                //self.me = UserId::try_from(format!("@{}:{}", self.me.localpart(), self.host.clone().split_once("://").unwrap().1).as_str()).unwrap();
            }
        }
        
        let content = json!({
            "type": "m.login.password",
            "identifier": {
                "type": "m.id.user",
                "user": self.me.localpart()
            },
            "password": password
        });
        let val: Value = self.post_r0("login", content, None).unwrap().into_json().unwrap();
        //println!("{:#}", val);
        self.data.auth_token = val["access_token"].as_str().unwrap().to_string();
        self.me = UserId::try_from(val["user_id"].as_str().unwrap()).unwrap();
    }
    pub fn get_me(&self) -> UserId {
        self.me.clone()
    }
    fn post(&self, url_ext: impl Into<String>, content: Value, query_pairs: Option<Vec<(String, String)>>) -> Result<ureq::Response, ureq::Transport> {
        let mut r = self.agent.post(format!("{}{}{}", self.host, BASE, url_ext.into()).as_str()).set("Authorization", format!("Bearer {}", self.data.auth_token).as_str());
        if let Some(pairs) = query_pairs {
            for pair in pairs {
                r = r.query(pair.0.as_str(), pair.1.as_str());
            }
        }
        //let x = r.send_json(content).or_any_status();
        //println!("{:#?}", x);
        //x
        
        let mut resp = r.clone().send_json(content.clone()).or_any_status();
        while let Err(_e) = resp {
            resp = r.clone().send_json(content.clone()).or_any_status();
            //eprintln!("{}", e);
        }
        resp
    }

    fn post_r0(&self, url_ext: impl Into<String>, content: Value, query_pairs: Option<Vec<(String, String)>>) -> Result<ureq::Response, ureq::Transport> {
        let mut r = self.agent.post(format!("{}{}{}", self.host, BASE.replace("v3", "r0"), url_ext.into()).as_str()).set("Authorization", format!("Bearer {}", self.data.auth_token).as_str());
        if let Some(pairs) = query_pairs {
            for pair in pairs {
                r = r.query(pair.0.as_str(), pair.1.as_str());
            }
        }
        //let x = r.send_json(content).or_any_status();
        //println!("{:#?}", x);
        //x
        
        let mut resp = r.clone().send_json(content.clone()).or_any_status();
        while let Err(_e) = resp {
            resp = r.clone().send_json(content.clone()).or_any_status();
            //eprintln!("{}", e);
        }
        resp
    }
    fn put(&self, url_ext: impl Into<String>, content: Value, query_pairs: Option<Vec<(String, String)>>) -> Result<ureq::Response, ureq::Transport> {
        let mut r = self.agent.put(format!("{}{}{}", self.host, BASE, url_ext.into()).as_str()).set("Authorization", format!("Bearer {}", self.data.auth_token).as_str());
        if let Some(pairs) = query_pairs {
            for pair in pairs {
                r = r.query(pair.0.as_str(), pair.1.as_str());
            }
        }

        let mut resp = r.clone().send_json(content.clone()).or_any_status();
        while let Err(_e) = resp {
            resp = r.clone().send_json(content.clone()).or_any_status();
            //eprintln!("{}", e);
        }
        resp
    }
    fn get(&self, url_ext: impl Into<String>, query_pairs: Option<Vec<(String, String)>>) -> Result<ureq::Response, ureq::Transport> {
        let mut r = self.agent.get(format!("{}{}{}", self.host, BASE, url_ext.into()).as_str()).set("Authorization", format!("Bearer {}", self.data.auth_token).as_str());
        if let Some(pairs) = query_pairs {
            for pair in pairs {
                r = r.query(pair.0.as_str(), pair.1.as_str());
            }
        }
        let mut resp = r.clone().call().or_any_status();
        
        while let Err(_e) = resp {
            resp = r.clone().call().or_any_status();
            //eprintln!("{}", e);
        }
        
        //println!("{:#?}", resp);
        resp
    }
    pub fn join_room_id(&self, room_id: ruma::RoomId) -> Value {
        //let room_string = room_id_or_alias.into();
        //let serv_name = server_name.into();
        let rid = room_id.to_string();
        let room = urlencoding::encode(rid.as_str());
        let content = json!({});
        self.post(format!("rooms/{}/join", room).as_str(), content, None).unwrap().into_json().unwrap()
    }
    pub fn sync(&mut self) -> Result<EventResponse, Value> {
        //let mut r = self.get("sync")
        let req: ureq::Response;
        if let Some(start) = self.since.clone() {
            req = self.get("sync", Some(vec![("since".to_string(), start), ("full_state".to_string(), "false".to_string()), ("set_presence".to_string(), format!("{}", PresenceState::Online)),  ("timeout".to_string(), "3000".to_string())])).unwrap();
        } else {
            req = self.get("sync", Some(vec![("full_state".to_string(), "true".to_string()), ("set_presence".to_string(), format!("{}", PresenceState::Online)), ("timeout".to_string(), "3000".to_string())])).unwrap();
        }
        let val: Value = req.into_json().unwrap();
        //self.get_tx().send(format!("{:#?}", val.to_string())).unwrap();
        //sm_println!(self, "{:#?}", val);
        //println!("{:#?}", val);
        if val.clone().as_object().unwrap().contains_key("error") {
            Err(val)
        } else {
            let x: EventResponse = serde_json::from_str::<Raw<EventResponse>>(val.to_string().as_str()).unwrap().deserialize().unwrap();
            self.since = Some(x.clone().next_batch);
            Ok(x)
        }
    }
    pub fn join_with_alias(&mut self, alias: impl Into<String>) {
        //self.post(format!("join/{}", urlencoding::encode(alias.into())), content, None).unwrap().into_json().unwrap()
        let room_alias = alias.into();//format!("{}", alias.into());
        let this_room_id_val: Value = self.get(format!("directory/room/{}", urlencoding::encode(room_alias.as_str()).to_string().as_str()), None).unwrap().into_json().unwrap();
        let room_id = ruma::RoomId::try_from(this_room_id_val["room_id"].as_str().unwrap()).unwrap();
        //self.data.room_to_aliases.insert(room_id.clone(), vec![room_alias.try_into().unwrap()]);
        self.join_room_id(room_id);
    }
    pub fn room_id_from_alias(&self, alias: impl Into<String>) -> RoomId {
        let room_alias = alias.into();//format!("{}", alias.into());
        let this_room_id_val: Value = self.get(format!("directory/room/{}", urlencoding::encode(room_alias.as_str()).to_string().as_str()), None).unwrap().into_json().unwrap();
        ruma::RoomId::try_from(this_room_id_val["room_id"].as_str().unwrap()).unwrap()
    }
    /*
    #[cfg(not(target_os = "android"))]
    pub fn start(&mut self, password: impl Into<String>, room_ids: Vec<impl Into<String>>) {
        let pass = password.into();
        for this_room in room_ids {
            let this_room_string = this_room.into();
            let this_room_id_val: Value = self.get(format!("directory/room/{}", urlencoding::encode(this_room_string.as_str()).to_string().as_str()), None).unwrap().into_json().unwrap();
            let room_id = ruma::RoomId::try_from(this_room_id_val["room_id"].as_str().unwrap()).unwrap();
        }
    }
    */
    pub fn send_message(&self, room_id: RoomId, message: impl Into<String>) {
        let msg = message.into();
        let txid = Uuid::new_v4().to_simple().encode_lower(&mut Uuid::encode_buffer()).to_string();
        self.put(format!("rooms/{}/send/m.room.message/{}", room_id.clone(), txid).as_str(), json!({"body": msg, "msgtype": "m.text"}), None).unwrap();
    }


    pub fn extract_messages(&self, sync_results: EventResponse) -> Vec<MessageInfo> {
        let mut messages: Vec<MessageInfo> = Vec::new();
        //if let RoomsmTypeData::Joined(ref x) = event_args.room_data {
        let x = sync_results.rooms.join;
        for (room_id, joined_room) in x.iter() {
            //if event_args.ctrlc_handler.should_continue() {
            /*
            instructions.push(Instructions::AddRoom(room_id.clone()));
            for event in joined_room.state.events.clone() {
                if let ruma::events::AnySyncStateEvent::RoomAliases(room_aliases_event) = event.deserialize().unwrap() {
                    let aliases = room_aliases_event.content.aliases;
                    for alias in aliases {
                        instructions.push(Instructions::SaveRoomAlias(room_id.clone(), alias));
                    }
                } //else if let ruma::events::AnySyncStateEvent::RoomCanonicalAlias
            }
            */
            //if !event_args.starting {
            for message_info in joined_room.timeline.events.iter().filter_map(|m| {
                if let ruma::events::AnySyncRoomEvent::Message(message_event) = m.deserialize_as().unwrap() {
                    if let ruma::events::AnySyncMessageEvent::RoomMessage(room_message) = message_event {
                        let sender = room_message.sender;
                        
                        if sender == self.me {
                            return None;
                        }
                        
                        if let room::message::MessageType::Text(text_message_event_content) = room_message.content.msgtype {
                            let message = text_message_event_content.body;
                            //event_args.tx.send(format!("<{}> {}", sender, message)).unwrap();
                            //if message.starts_with(&event_args.prefix) {
                                /*
                                let words: Vec<String> = message.split_whitespace().map(|w| w.to_string()).collect();
                                let args = words[1..].to_vec();
                                let mut room_aliases: Vec<RoomAliasId> = Vec::new();
                                if event_args.room_to_aliases.contains_key(&room_id) {
                                    room_aliases.append(&mut event_args.room_to_aliases[&room_id].clone());
                                }
                                */
                            return Some(MessageInfo{
                                message: message.clone(),
                                //words: words,
                                //args: args,
                                sender: sender,
                                room_id: room_id.clone(),
                                //room_aliases: room_aliases,
                            });
                            //}
                        }
                    }
                }
                None
            }) {
                messages.push(message_info);
            }
            //}
            //} //else {
                //instructions.push(Instructions::Quit(event_args.cleanup_on_ctrlc));
                //return instructions;
            //}
        }
        //}
        messages
    }


}

pub struct MessageInfo {
    pub room_id: RoomId,
    pub sender: UserId,
    pub message: String,
}

