use std::io::Write;

use colored::*;
use crab_env as env;
use env::Environment;
use pest::iterators::Pair;
use reedline::*;

extern crate pest;
extern crate pest_derive;
use pest::Parser;
// TODO: Fer un map amb la key el nom del crate i els values la descripcio del crate
// TODO: Implementar sistema de crates (Posar tots els arguments dins del vector, executar recursivament fins q retorni algo i llavors fer push al vector)
// TODO: Moure tots els builtins (excepte exit) i fer-los en forma de Crate.

pub const ORANGE: colored::Color = colored::Color::TrueColor {
    r: 255,
    g: 127,
    b: 80,
};

#[derive(pest_derive::Parser)]
#[grammar = "crab.pest"]
pub struct CrabParser;

pub trait Display {
    fn display(self) -> Self;
}

impl Display for pest::iterators::Pairs<'_, Rule> {
    fn display(self) -> Self {
        let lines: Vec<_> = self
            .clone()
            .map(|pair| format_pair(pair, 0, true))
            .collect();
        let lines = lines.join("\n");
        println!("{}", lines);

        fn format_pair(
            pair: pest::iterators::Pair<Rule>,
            indent_level: usize,
            is_newline: bool,
        ) -> String {
            let indent = if is_newline {
                "  ".repeat(indent_level)
            } else {
                "".to_string()
            };

            let children: Vec<_> = pair.clone().into_inner().collect();
            let len = children.len();
            let children: Vec<_> = children
                .into_iter()
                .map(|pair| {
                    format_pair(
                        pair,
                        if len > 1 {
                            indent_level + 1
                        } else {
                            indent_level
                        },
                        len > 1,
                    )
                })
                .collect();

            let dash = if is_newline { "- " } else { "" };

            match len {
                0 => format!(
                    "{}{}{:?}: {:?}",
                    indent,
                    dash,
                    pair.as_rule(),
                    pair.as_span().as_str()
                ),
                1 => format!("{}{}{:?} > {}", indent, dash, pair.as_rule(), children[0]),
                _ => format!(
                    "{}{}{:?}\n{}",
                    indent,
                    dash,
                    pair.as_rule(),
                    children.join("\n")
                ),
            }
        }

        std::mem::drop(lines);

        self
    }
}

fn run_command(inp: String, env: &mut Environment) -> bool {
    let error: pest::iterators::Pairs<'_, Rule> =
        CrabParser::parse(Rule::Error, "##error!##").unwrap();
    // Get builtin or dynamic
    let parse = CrabParser::parse(Rule::Main, &inp)
        .unwrap_or(error)
        .display()
        .next()
        .unwrap();

    match parse.as_rule() {
        Rule::Builtin => run_builtin(parse.into_inner().next().unwrap()),
        Rule::Dyn => run_dyn(parse.into_inner().next().unwrap(), env),
        _ => false,
    }
}

fn run_builtin(parse: Pair<'_, Rule>) -> bool {
    match parse.as_rule() {
        Rule::Let => {
            todo!()
        }
        Rule::Variable => {
            todo!()
        }
        Rule::Exit => true,
        _ => {
            println!("Unknown command {}", parse.as_str());
            false
        }
    }
}

fn run_dyn(parse: Pair<'_, Rule>, env: &mut Environment) -> bool {
    match parse.as_rule() {
        Rule::Crate => {
            run_crate(parse, env);
        }
        Rule::Extern => {
            run_extern(parse);
        }
        _ => {},
    }
    
    false
}

fn run_crate(parse: Pair<'_, Rule>, env: &mut Environment) -> String {
    match parse.as_rule() {
        Rule::Crate => {
            // Get Crate name (doesn't panic)
            let mut parse = parse.into_inner();
            let name = parse.next().unwrap().as_str();
            let mut args: Vec<String> = Vec::new();

            // If the Crate has args get them
            if let Some(parse) = parse.next() {
                for arg in parse.into_inner() {
                    match arg.as_rule() {
                        // If it's a normal arg just push it
                        Rule::Arg => args.push(arg.as_str().into()),
                        // If it's the return value of a crate, run the crate
                        Rule::Crate => args.push(run_crate(arg, env)),
                        _ => {}
                    }
                }
            }

            // Save current environment before running Crate
            env.return_val = String::new();
            env::save(env);

            // Run Crate with corresponding args
            std::process::Command::new("cargo")
            .arg("run")
            .arg("-q")
            .arg("--release")
            .arg(format!(
                "--manifest-path={}/{}/Cargo.toml",
                env::CRATES_PATH.to_string_lossy(),
                name
            ))
            .args(args.get(0..).unwrap_or(&[]))
            .status()
            .expect(&format!("Crate {} failed to start", name));

            // Load possible changes the Crate made
            env::load_into(env);
        }
        _ => {},
    }

    env.return_val.clone()
}

fn run_extern(parse: Pair<'_, Rule>) {
    let parse: Vec<&str> = parse.into_inner().as_str().split_whitespace().collect();
    let comm = std::process::Command::new(parse[0]).args(&parse[1..]).status();
    if let Err(e) = comm {
        println!("{}", e);
    }
}

/*
fn run_command(command: Rule, args: &mut Vec<String>) -> bool {
    //println!("command: {:#?}, args: {:#?}", command, args);
    /*

    if command == "extern" || command.starts_with(':') {
        if args.len() == 0 {
            println!("ERROR: No arguments.\nSyntax: extern(command, arg1, arg2, ...)");
            return false;
        }

        let mut args = args.into_iter();
        let command = args.nth(0).unwrap();
        std::process::Command::new(command.strip_prefix(':').unwrap_or(command))
            .args(args)
            .status().expect(&format!("Command {} failed to run", command));
    }
    match command as &str {
        "\n" => {},
        "exit" | "error" =>  {},
        "new" => {
            if args.len() > 1 {
                println!("ERROR: Too many arguments.\nSyntax: new(crate_name)");
                return false;
            }
        },
        "compile" => {
            for cr in CRATES.iter() {
                println!("Compiling {}", cr);
                std::process::Command::new("cargo")
                    .arg("build")
                    .arg("--release")
                    .arg(format!("--manifest-path={}/{}/Cargo.toml", CRATES_PATH.to_string_lossy(), cr))
                    .status().expect(&format!("Crate {} failed to build", cr));
            }
        }
        "help" => {
            println!("{}", "Built-In:".bold());
            for cr in BUILT_IN.iter() { println!("{}", cr)}
            println!("{}","\nCrates:".bold());
            for cr in CRATES.iter() { println!("{}", cr); }
        },
        _ => {

        }
    }
     */
    match command {
        // Builtins
        Rule::NOTHING => {}
        Rule::Exit => println!("Goodbye!"),
        Rule::Error => match &args.len() {
            0 => println!("Unknown error"),
            _ => println!("Unknown command {:#?}", args[0]),
        },
        Rule::Help => {
            println!("{}", "Crates:".bold());
            for cr in env::CRATES.iter() {
                println!("{}", cr);
            }
        }
        Rule::NewCrate => {
            std::process::Command::new("cargo")
                .arg("new")
                .arg(format!(
                    "{}/{}",
                    env::CRATES_PATH.to_string_lossy(),
                    args[0]
                ))
                .status()
                .expect(&format!("Crate {} failed to start", args[0]));
        }
        Rule::Compile => {
            fn compile(cr: &String) {
                println!("Compiling {}", cr);
                std::process::Command::new("cargo")
                    .arg("build")
                    .arg("--release")
                    .arg(format!(
                        "--manifest-path={}/{}/Cargo.toml",
                        env::CRATES_PATH.to_string_lossy(),
                        cr
                    ))
                    .status()
                    .expect(&format!("Crate {} failed to build", cr));
            }
            match args.len() {
                0 => {
                    for cr in env::CRATES.iter() {
                        compile(cr)
                    }
                }
                1 => compile(&args[0]),
                _ => {
                    for cr in args {
                        compile(cr)
                    }
                }
            }
            print!("\x1B[2J\x1B[1;1H");
            println!("Compiling completed!");
        }

        // Dynamic
        Rule::Crate => {
            std::process::Command::new("cargo")
                .arg("run")
                .arg("-q")
                .arg("--release")
                .arg(format!(
                    "--manifest-path={}/{}/Cargo.toml",
                    env::CRATES_PATH.to_string_lossy(),
                    args[0]
                ))
                .args(args.get(1..).unwrap_or(&[]))
                .status()
                .expect(&format!("Crate {} failed to start", args[0]));
        }
        _ => {}
    }

    command == Rule::Exit
}
*/

fn update_env(env: &Environment) {
    println!("{:?}", env);
    std::io::stdout().flush().unwrap();
    std::env::set_current_dir(&env.current_dir).unwrap();
}

fn main() {
    print!("\x1B[2J\x1B[1;1H");

    let mut env = env::load();

    let commands = env::CRATES.clone().into_iter().collect::<Vec<String>>();
    let completer = Box::new(DefaultCompleter::new_with_wordlen(commands.clone(), 2));
    let mut line_editor = Reedline::create()
        .unwrap()
        .with_highlighter(Box::new(DefaultHighlighter::new(commands.clone())))
        .with_completion_action_handler(Box::new(
            DefaultCompletionActionHandler::default().with_completer(completer),
        ))
        .with_validator(Box::new(DefaultValidator))
        .with_hinter(Box::new(
            DefaultHinter::default()
                .with_completer(Box::new(DefaultCompleter::new_with_wordlen(
                    commands.clone(),
                    2,
                )))
                .with_style(nu_ansi_term::Style::new().fg(nu_ansi_term::Color::DarkGray)),
        ));

    let prompt = reedline::DefaultPrompt::new(6);

    let dir = env::USER_DIRS.home_dir().to_str().unwrap().into();
    std::env::set_current_dir(&dir).unwrap();
    env.current_dir = dir;

    println!("\nWelcome to the Crab Shell! \nWrite {} for a list of the available Crates", "help".bold().color(ORANGE));

    loop {
        if env.debug { println!("{:#?}", env); }
        let sig = line_editor.read_line(&prompt).unwrap();
        match sig {
            Signal::Success(buffer) => {
                if run_command(buffer, &mut env) { break }
                update_env(&env);
            }
            Signal::CtrlC | Signal::CtrlD => {
                line_editor.print_crlf().unwrap();
                break;
            }
            Signal::CtrlL => {}
        }
    }
}
