/*
 * MIT License
 *
 * Copyright (c) 2019 Frank Fischer
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use cplex_sys as cpx;
use cpx::trycpx;

use std::ffi::CString;
use std::os::raw::{c_char, c_int};
use std::ptr::null;

#[test]
fn test_cflp() -> std::result::Result<(), Box<dyn std::error::Error>> {
    let n = 3;
    let m = 5;

    let f = [1000.0, 1000.0, 1000.0];
    let cap = [500.0, 500.0, 500.0];
    let c = [
        [4.0, 5.0, 6.0, 8.0, 10.0], //
        [6.0, 4.0, 3.0, 5.0, 8.0],  //
        [9.0, 7.0, 4.0, 3.0, 4.0],  //
    ];
    let demand = [80.0, 270.0, 250.0, 160.0, 180.0];

    let mut status: c_int = 0;
    let mut env;
    trycpx!({
        env = cpx::openCPLEX(&mut status);
        status
    });
    let mut lp;
    trycpx!({
        let name = CString::new("cflp")?;
        lp = cpx::createprob(env, &mut status, name.as_ptr());
        status
    });

    // variables
    {
        let mut vnames = Vec::with_capacity(n + n * m);
        let mut costs = Vec::with_capacity(n + n * m);
        let mut vtypes = Vec::with_capacity(n + n * m);
        // facility vars
        for i in 0..n {
            vnames.push(CString::new(format!("y#{}", i))?);
            costs.push(f[i]);
            vtypes.push('B' as c_char);
        }
        // transport vars
        for i in 0..n {
            for j in 0..m {
                vnames.push(CString::new(format!("x#{}#{}", i, j))?);
                costs.push(demand[j] * c[i][j]);
                vtypes.push('C' as c_char);
            }
        }
        trycpx!(cpx::newcols(
            env,
            lp,
            vnames.len() as c_int,
            costs.as_ptr(),
            null(),
            null(),
            vtypes.as_ptr(),
            vnames.iter().map(|n| n.as_ptr()).collect::<Vec<_>>().as_ptr()
        ));
    }

    // constraints
    {
        let mut rbeg = Vec::with_capacity(m + n);
        let mut rind = Vec::with_capacity(n * m);
        let mut rval = Vec::with_capacity(n * m);
        let mut rsense = Vec::with_capacity(m + n);
        let mut rhs = Vec::with_capacity(m + n);

        for j in 0..m {
            rbeg.push(rind.len() as c_int);
            rsense.push(b'E' as c_char);
            rhs.push(1.0);
            for i in 0..n {
                rind.push((n + i * m + j) as c_int);
                rval.push(1.0);
            }
        }

        for i in 0..n {
            rbeg.push(rind.len() as c_int);
            rsense.push(b'L' as c_char);
            rhs.push(0.0);
            for j in 0..m {
                rind.push((n + i * m + j) as c_int);
                rval.push(demand[j]);
            }
            rind.push(i as c_int);
            rval.push(-cap[i]);
        }

        trycpx!(cpx::addrows(
            env,
            lp,
            0,
            rbeg.len() as c_int,
            rind.len() as c_int,
            rhs.as_ptr(),
            rsense.as_ptr(),
            rbeg.as_ptr(),
            rind.as_ptr(),
            rval.as_ptr(),
            null(),
            null()
        ));
    }

    trycpx!(cpx::mipopt(env, lp));

    let mut objval = 0.0;
    trycpx!(cpx::getobjval(env, lp, &mut objval));
    assert_eq!(objval, 5610.0);

    let mut ysol = vec![0.0; n];
    let mut xsol = vec![0.0; n * m];
    trycpx!(cpx::getx(env, lp, ysol.as_mut_ptr(), 0, (n - 1) as c_int));
    trycpx!(cpx::getx(
        env,
        lp,
        xsol.as_mut_ptr(),
        n as c_int,
        (n + n * m - 1) as c_int
    ));
    assert!(ysol[0] < 0.5);
    assert!(ysol[1] > 0.5);
    assert!(ysol[2] > 0.5);
    assert!((xsol[m] - 1.0).abs() < 1e-6);
    assert!((xsol[m + 1] - 1.0).abs() < 1e-6);
    assert!((xsol[m + 2] - 0.6).abs() < 1e-6);
    assert!((xsol[2 * m + 2] - 0.4).abs() < 1e-6);
    assert!((xsol[2 * m + 3] - 1.0).abs() < 1e-6);
    assert!((xsol[2 * m + 4] - 1.0).abs() < 1e-6);

    trycpx!(cpx::freeprob(env, &mut lp));
    trycpx!(cpx::closeCPLEX(&mut env));

    Ok(())
}
