/*
 * MIT License
 *
 * Copyright (c) 2017, 2018 Frank Fischer
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Low-level binding for [Cplex][cplex]
//!
//! This crate provides *automatically generated* low-level bindings
//! to [Cplex][cplex]. The build script scans the C-header files and
//! creates appropriate functions and constants.
//!
//! In order to compile this crate, the environment variable `CPLEX_HOME`
//! must be set to a path containing cplex, e.g.
//!
//! ```sh
//! export CPLEX_HOME=/opt/CPLEX_Studio1271/cplex
//! cargo build
//! ```
//!
//! (That path should contain the header files in `include/ilcplex` and the library
//! in `lib/x86-64_linux`).
//!
//! Note that this is *really* low-level. The functions are just
//! plain bindings to the C-API without any additional safe guards.
//!
//! [cplex]: http://www-03.ibm.com/software/products/de/ibmilogcpleoptistud/ "Cplex Homepage"
//!
//! # Example
//!
//! ```
//! #[macro_use]
//! extern crate cplex_sys as cpx;
//! use std::os::raw::{c_char, c_int};
//! use std::ffi::CString;
//! use std::ptr::null;
//!
//! fn main() {
//!     let mut status = 0;
//!     let mut env = unsafe { cpx::openCPLEX(&mut status) };
//!     assert_eq!(status, 0);
//!
//!     let mut lp = unsafe {
//!         cpx::createprob(env, &mut status,
//!                         CString::new("Test LP").unwrap().as_ptr())
//!     };
//!     assert_eq!(status, 0);
//!
//!     let obj     = [1.0,  2.0, 3.0];
//!     let ub      = [40.0, cpx::INFBOUND, cpx::INFBOUND];
//!     let rmatbeg = [0,              3];
//!     let rmatind = [0,    1,   2,   0,   1,    2];
//!     let rmatval = [-1.0, 1.0, 1.0, 1.0, -3.0, 1.0];
//!     let sense   = ['L' as c_char, 'L' as c_char];
//!     let rhs     = [20.0, 30.0];
//!
//!     unwrapcpx!(cpx::newcols(env, lp, 3, obj.as_ptr(), null(), ub.as_ptr(),
//!                             null(), null()));
//!     unwrapcpx!(cpx::addrows(env, lp,
//!                             0, 2, rmatval.len() as c_int,
//!                             rhs.as_ptr(), sense.as_ptr(),
//!                             rmatbeg.as_ptr(), rmatind.as_ptr(), rmatval.as_ptr(),
//!                             null(), null()));
//!     unwrapcpx!(cpx::chgobjsen(env, lp, cpx::MAX));
//!     unwrapcpx!(cpx::lpopt(env, lp));
//!
//!     let mut x = [0.0; 3];
//!     let mut objval = 0.0;
//!     unwrapcpx!(cpx::getobjval(env, lp, &mut objval));
//!     unwrapcpx!(cpx::getx(env, lp, x.as_mut_ptr(), 0, 2));
//!     assert!((objval - 202.5).abs() < 1e-9);
//!     assert!((x[0] - 40.0).abs() < 1e-9);
//!     assert!((x[1] - 17.5).abs() < 1e-9);
//!     assert!((x[2] - 42.5).abs() < 1e-9);
//!
//!     unwrapcpx!(cpx::freeprob(env, &mut lp));
//!     unwrapcpx!(cpx::closeCPLEX(&mut env));
//! }
//! ```

use std::error;
use std::fmt;
use std::os::raw::{c_char, c_double, c_int, c_longlong, c_void};

extern crate libc;
/// Reexport of the standard C file handle.
pub use libc::FILE as File;

pub const MSGBUFSIZE: usize = 1024;
#[allow(dead_code)]

/// Error describing a CPLEX status code.
///
/// This is a wrapper around CPLEX status code as returned by most
/// CPLEX low-level functions. The error struct contains the status
/// code itself along with the error message string returned by
/// `geterrorstring`.
#[derive(Debug)]
pub struct CplexError {
    /// The CPLEX error status code.
    pub code: c_int,
    /// The CPLEX error message associated with the status code.
    pub msg: String,
}

impl CplexError {
    pub fn new(status: c_int) -> CplexError {
        CplexError {
            code: status,
            msg: geterrormessage(status),
        }
    }
}

impl fmt::Display for CplexError {
    fn fmt(&self, fmt: &mut fmt::Formatter) -> Result<(), fmt::Error> {
        write!(fmt, "CPLEX Error ({}): {}", self.code, self.msg)
    }
}

impl error::Error for CplexError {}

include!(concat!(env!("OUT_DIR"), "/cplex-extern.rs"));

/// Objective sense of the problem.
#[derive(Clone, Copy, PartialEq, Eq, Debug)]
pub enum ObjectiveSense {
    Maximize = -1,
    Minimize = 1,
}

impl ObjectiveSense {
    pub fn to_c(self) -> c_int {
        self as c_int
    }
}

pub const MESSAGE_BUF_SIZE: usize = 1024;

/// Return the error message associated with some code.
///
/// This is a simple convenience function that returns the error
/// message as a Rust string.
pub fn geterrormessage(status: c_int) -> String {
    unsafe {
        let mut msg = vec![0; MESSAGE_BUF_SIZE];
        geterrorstring(env(), status, msg.as_mut_ptr());
        let msg = msg
            .into_iter()
            .take_while(|c| *c != 0)
            .map(|c| c as u8)
            .collect::<Vec<_>>();
        String::from_utf8(msg).unwrap()
    }
}

/// Wrapper around CPLEX low-level functions returning a status code.
///
/// This macro is analogous to `try!` but expects as argument a CPLEX
/// low-level function that returns a `c_int` status code.
///
/// `trycpx!` contains an `unsafe` block ... using `trycpx!` is
/// *always* unsafe.
///
/// # Examples
///
/// ```rust,ignore
/// fn myfunc(net : *mut Net) -> Result<(), Error> {
///   trycpx!(CPXNETchgobj(cplex::env(), net, CPX_MAX))
/// }
/// ```
#[macro_export]
macro_rules! trycpx {
    ($e:expr) => {
        unsafe {
            let status = $e;
            if status != 0 {
                return ::std::result::Result::Err(::std::convert::From::from($crate::CplexError::new(status)));
            } else {
                ()
            }
        }
    };
}

/// Wrapper around CPLEX low-level functions returning a status code.
///
/// This macro is like `trycpx!` but simply panics if a non-zero exit
/// code is returned.
///
/// `unwrapcpx!` contains an `unsafe` block ... using `unwrapcpx!` is
/// *always* unsafe.
#[macro_export]
macro_rules! unwrapcpx {
    ($e:expr) => {
        unsafe {
            let status = $e;
            if status != 0 {
                panic!("{}", $crate::CplexError::new(status))
            } else {
                ()
            }
        }
    };
}

/// Globally unique environment.
static mut ENV: *mut Env = 0 as *mut Env;

pub unsafe fn env() -> *mut Env {
    if ENV.is_null() {
        let mut status: c_int = 0;
        ENV = openCPLEX(&mut status);
        if status != 0 {
            panic!("Can't open CPLEX environment");
        }
    }
    ENV
}
