#[cfg(test)]
mod tests {

    use cpclib_common::itertools::Itertools;

    fn test_single_dsk(dsk: &cpclib::disc::edsk::ExtendedDsk) {
        let track = dsk
            .get_track_information(cpclib::disc::edsk::Head::A, 0)
            .unwrap();
        assert_eq!(*track.number_of_sectors(), 9);

        for (sector_idx, sum) in &[
            (0xC1, 21413),
            (0xC6, 60263),
            (0xC2, 22014),
            (0xC7, 49447),
            (0xC3, 85780)
        ] {
            let sector = track.sector(*sector_idx).unwrap();
            let values = sector
                .values()
                .iter()
                .map(|&v| format!("{:x}", v))
                .collect::<Vec<_>>();
            println!("0x{:x} => {:?}", sector_idx, values);
            assert_eq!(values.len(), 512);
            assert_eq!(sector.data_sum(), *sum);
        }

        assert_eq!(track.data_sum(), 484121);
        assert_eq!(
            dsk.get_track_information(cpclib::disc::edsk::Head::A, 41)
                .unwrap()
                .data_sum(),
            329484
        );

        // Check catalgo access

        assert!(dsk.sector(0, 0, 0xC1).is_some());
        assert!(dsk.sector(0, 0, 0xC2).is_some());
        assert!(dsk.sector(0, 0, 0xC3).is_some());
        assert!(dsk.sector(0, 0, 0xC4).is_some());

        assert!(dsk
            .sectors_bytes(
                0,    // track
                0xC1, // sector
                4,    // nb sector
                0.into()
            )
            .is_some());

        assert!(dsk
            .sectors_bytes(
                0,    // track
                0xC1, // sector
                4,    // nb sector
                1.into()
            )
            .is_none());

        assert!(dsk
            .sectors_bytes(
                0,    // track
                0xC1, // sector
                4,    // nb sector
                2.into()
            )
            .is_none());
    }

    #[test]
    fn open_singel_head_edsk() {
        let dsk = cpclib::disc::edsk::ExtendedDsk::open("./tests/dsk/pirate.dsk").unwrap();
        test_single_dsk(&dsk);

        let tmp_file = "/tmp/tmp.dsk";
        dsk.save(tmp_file).unwrap();
        let dsk = cpclib::disc::edsk::ExtendedDsk::open(tmp_file).unwrap();
        test_single_dsk(&dsk);
    }

    fn test_double_head_bf_edsk(dsk: &cpclib::disc::edsk::ExtendedDsk) {
        assert!(dsk.is_double_head());
        assert_eq!(dsk.data_sum(cpclib::disc::edsk::Head::A), 66709468);

        assert_eq!(dsk.data_sum(cpclib::disc::edsk::Head::B), 54340792);
    }

    #[test]
    fn open_double_head_edsk() {
        let dsk = cpclib::disc::edsk::ExtendedDsk::open("./tests/dsk/bf2sides.dsk").unwrap();
        test_double_head_bf_edsk(&dsk);

        let tmp_file = "/tmp/tmp.dsk";
        dsk.save(tmp_file).unwrap();
        let dsk = cpclib::disc::edsk::ExtendedDsk::open(tmp_file).unwrap();
        test_double_head_bf_edsk(&dsk);
    }

    #[test]
    fn save_edsk() {
        let tmp_file = "/tmp/tmp.dsk";
        let dsk1 = cpclib::disc::edsk::ExtendedDsk::open("tests/dsk/pirate.dsk").unwrap();
        dsk1.save(tmp_file).unwrap();
        let _ds2 = cpclib::disc::edsk::ExtendedDsk::open(tmp_file).unwrap();
    }

    #[test]
    fn sector_size() {
        use cpclib::disc::edsk::{
            convert_fdc_sector_size_to_real_sector_size,
            convert_real_sector_size_to_fdc_sector_size
        };

        for real_human_size in [256, 512, 1024, 2048].iter() {
            let computed_fdc_size = convert_real_sector_size_to_fdc_sector_size(*real_human_size);
            let computed_human_size =
                convert_fdc_sector_size_to_real_sector_size(computed_fdc_size);

            assert_eq!(*real_human_size, computed_human_size);
        }
    }

    #[test]
    fn test_harley() {
        use cpclib_common::itertools::Itertools;

        let dsk = cpclib::disc::edsk::ExtendedDsk::open("tests/dsk/harley.dsk").unwrap();

        let tracks_content = [
            673035, 398812, 394382, 391058, 392024, 400595, 382743, 642349, 1055232, 1055232,
            1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232,
            1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232,
            1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232, 1055232,
            1055232, 1055232, 1055232, 1055232, 1055232
        ];

        for (idx, value) in tracks_content.iter().enumerate() {
            assert_eq!(
                dsk.track_bytes(cpclib::disc::edsk::Head::A, idx as u8)
                    .unwrap()
                    .iter()
                    .map(|&v| v as usize)
                    .sum::<usize>(),
                *value
            );
        }

        let sectors_content = [
            0, 0xC1, 512, 512, 104636, 0, 0xC6, 512, 512, 63783, 0, 0xC2, 512, 512, 117248, 0,
            0xC7, 512, 512, 52723, 0, 0xC3, 512, 512, 117248, 0, 0xC8, 512, 512, 18656, 0, 0xC4,
            512, 512, 117248, 0, 0xC9, 512, 512, 39740, 0, 0xC5, 512, 512, 41753, 1, 0xC1, 512,
            512, 61288, 1, 0xC6, 512, 512, 55203, 1, 0xC2, 512, 512, 54618, 1, 0xC7, 512, 512,
            12729, 1, 0xC3, 512, 512, 11887, 1, 0xC8, 512, 512, 38235, 1, 0xC4, 512, 512, 41604, 1,
            0xC9, 512, 512, 61835, 1, 0xC5, 512, 512, 61413, 2, 0xC1, 512, 512, 53164, 2, 0xC6,
            512, 512, 12252, 2, 0xC2, 512, 512, 12422, 2, 0xC7, 512, 512, 42150, 2, 0xC3, 512, 512,
            43795, 2, 0xC8, 512, 512, 60866, 2, 0xC4, 512, 512, 62111, 2, 0xC9, 512, 512, 53609, 2,
            0xC5, 512, 512, 54013, 3, 0xC1, 512, 512, 13335, 3, 0xC6, 512, 512, 42845, 3, 0xC2,
            512, 512, 44481, 3, 0xC7, 512, 512, 60737, 3, 0xC3, 512, 512, 61903, 3, 0xC8, 512, 512,
            53384, 3, 0xC4, 512, 512, 53641, 3, 0xC9, 512, 512, 48237, 3, 0xC5, 512, 512, 12495, 4,
            0xC1, 512, 512, 47844, 4, 0xC6, 512, 512, 55210, 4, 0xC2, 512, 512, 49398, 4, 0xC7,
            512, 512, 32158, 4, 0xC3, 512, 512, 33983, 4, 0xC8, 512, 512, 33977, 4, 0xC4, 512, 512,
            36215, 4, 0xC9, 512, 512, 49994, 4, 0xC5, 512, 512, 53245, 5, 0xC1, 512, 512, 53690, 5,
            0xC6, 512, 512, 29653, 5, 0xC2, 512, 512, 32992, 5, 0xC7, 512, 512, 34133, 5, 0xC3,
            512, 512, 34536, 5, 0xC8, 512, 512, 48383, 5, 0xC4, 512, 512, 52255, 5, 0xC9, 512, 512,
            55470, 5, 0xC5, 512, 512, 59483, 6, 0xC1, 512, 512, 33246, 6, 0xC6, 512, 512, 32272, 6,
            0xC2, 512, 512, 34518, 6, 0xC7, 512, 512, 46384, 6, 0xC3, 512, 512, 49958, 6, 0xC8,
            512, 512, 58711, 6, 0xC4, 512, 512, 59494, 6, 0xC9, 512, 512, 35296, 6, 0xC5, 512, 512,
            32864, 7, 0xC1, 512, 512, 33853, 7, 0xC6, 512, 512, 117248, 7, 0xC2, 512, 512, 50621,
            7, 0xC7, 512, 512, 117248, 7, 0xC3, 512, 512, 55535, 7, 0xC8, 512, 512, 117248, 7,
            0xC4, 512, 512, 33348, 7, 0xC9, 512, 512, 117248, 7, 0xC5, 512, 512, 0, 8, 0xC1, 512,
            512, 117248, 8, 0xC6, 512, 512, 117248, 8, 0xC2, 512, 512, 117248, 8, 0xC7, 512, 512,
            117248, 8, 0xC3, 512, 512, 117248, 8, 0xC8, 512, 512, 117248, 8, 0xC4, 512, 512,
            117248, 8, 0xC9, 512, 512, 117248, 8, 0xC5, 512, 512, 117248, 9, 0xC1, 512, 512,
            117248, 9, 0xC6, 512, 512, 117248, 9, 0xC2, 512, 512, 117248, 9, 0xC7, 512, 512,
            117248, 9, 0xC3, 512, 512, 117248, 9, 0xC8, 512, 512, 117248, 9, 0xC4, 512, 512,
            117248, 9, 0xC9, 512, 512, 117248, 9, 0xC5, 512, 512, 117248, 10, 0xC1, 512, 512,
            117248, 10, 0xC6, 512, 512, 117248, 10, 0xC2, 512, 512, 117248, 10, 0xC7, 512, 512,
            117248, 10, 0xC3, 512, 512, 117248, 10, 0xC8, 512, 512, 117248, 10, 0xC4, 512, 512,
            117248, 10, 0xC9, 512, 512, 117248, 10, 0xC5, 512, 512, 117248, 11, 0xC1, 512, 512,
            117248, 11, 0xC6, 512, 512, 117248, 11, 0xC2, 512, 512, 117248, 11, 0xC7, 512, 512,
            117248, 11, 0xC3, 512, 512, 117248, 11, 0xC8, 512, 512, 117248, 11, 0xC4, 512, 512,
            117248, 11, 0xC9, 512, 512, 117248, 11, 0xC5, 512, 512, 117248, 12, 0xC1, 512, 512,
            117248, 12, 0xC6, 512, 512, 117248, 12, 0xC2, 512, 512, 117248, 12, 0xC7, 512, 512,
            117248, 12, 0xC3, 512, 512, 117248, 12, 0xC8, 512, 512, 117248, 12, 0xC4, 512, 512,
            117248, 12, 0xC9, 512, 512, 117248, 12, 0xC5, 512, 512, 117248, 13, 0xC1, 512, 512,
            117248, 13, 0xC6, 512, 512, 117248, 13, 0xC2, 512, 512, 117248, 13, 0xC7, 512, 512,
            117248, 13, 0xC3, 512, 512, 117248, 13, 0xC8, 512, 512, 117248, 13, 0xC4, 512, 512,
            117248, 13, 0xC9, 512, 512, 117248, 13, 0xC5, 512, 512, 117248, 14, 0xC1, 512, 512,
            117248, 14, 0xC6, 512, 512, 117248, 14, 0xC2, 512, 512, 117248, 14, 0xC7, 512, 512,
            117248, 14, 0xC3, 512, 512, 117248, 14, 0xC8, 512, 512, 117248, 14, 0xC4, 512, 512,
            117248, 14, 0xC9, 512, 512, 117248, 14, 0xC5, 512, 512, 117248, 15, 0xC1, 512, 512,
            117248, 15, 0xC6, 512, 512, 117248, 15, 0xC2, 512, 512, 117248, 15, 0xC7, 512, 512,
            117248, 15, 0xC3, 512, 512, 117248, 15, 0xC8, 512, 512, 117248, 15, 0xC4, 512, 512,
            117248, 15, 0xC9, 512, 512, 117248, 15, 0xC5, 512, 512, 117248, 16, 0xC1, 512, 512,
            117248, 16, 0xC6, 512, 512, 117248, 16, 0xC2, 512, 512, 117248, 16, 0xC7, 512, 512,
            117248, 16, 0xC3, 512, 512, 117248, 16, 0xC8, 512, 512, 117248, 16, 0xC4, 512, 512,
            117248, 16, 0xC9, 512, 512, 117248, 16, 0xC5, 512, 512, 117248, 17, 0xC1, 512, 512,
            117248, 17, 0xC6, 512, 512, 117248, 17, 0xC2, 512, 512, 117248, 17, 0xC7, 512, 512,
            117248, 17, 0xC3, 512, 512, 117248, 17, 0xC8, 512, 512, 117248, 17, 0xC4, 512, 512,
            117248, 17, 0xC9, 512, 512, 117248, 17, 0xC5, 512, 512, 117248, 18, 0xC1, 512, 512,
            117248, 18, 0xC6, 512, 512, 117248, 18, 0xC2, 512, 512, 117248, 18, 0xC7, 512, 512,
            117248, 18, 0xC3, 512, 512, 117248, 18, 0xC8, 512, 512, 117248, 18, 0xC4, 512, 512,
            117248, 18, 0xC9, 512, 512, 117248, 18, 0xC5, 512, 512, 117248, 19, 0xC1, 512, 512,
            117248, 19, 0xC6, 512, 512, 117248, 19, 0xC2, 512, 512, 117248, 19, 0xC7, 512, 512,
            117248, 19, 0xC3, 512, 512, 117248, 19, 0xC8, 512, 512, 117248, 19, 0xC4, 512, 512,
            117248, 19, 0xC9, 512, 512, 117248, 19, 0xC5, 512, 512, 117248, 20, 0xC1, 512, 512,
            117248, 20, 0xC6, 512, 512, 117248, 20, 0xC2, 512, 512, 117248, 20, 0xC7, 512, 512,
            117248, 20, 0xC3, 512, 512, 117248, 20, 0xC8, 512, 512, 117248, 20, 0xC4, 512, 512,
            117248, 20, 0xC9, 512, 512, 117248, 20, 0xC5, 512, 512, 117248, 21, 0xC1, 512, 512,
            117248, 21, 0xC6, 512, 512, 117248, 21, 0xC2, 512, 512, 117248, 21, 0xC7, 512, 512,
            117248, 21, 0xC3, 512, 512, 117248, 21, 0xC8, 512, 512, 117248, 21, 0xC4, 512, 512,
            117248, 21, 0xC9, 512, 512, 117248, 21, 0xC5, 512, 512, 117248, 22, 0xC1, 512, 512,
            117248, 22, 0xC6, 512, 512, 117248, 22, 0xC2, 512, 512, 117248, 22, 0xC7, 512, 512,
            117248, 22, 0xC3, 512, 512, 117248, 22, 0xC8, 512, 512, 117248, 22, 0xC4, 512, 512,
            117248, 22, 0xC9, 512, 512, 117248, 22, 0xC5, 512, 512, 117248, 23, 0xC1, 512, 512,
            117248, 23, 0xC6, 512, 512, 117248, 23, 0xC2, 512, 512, 117248, 23, 0xC7, 512, 512,
            117248, 23, 0xC3, 512, 512, 117248, 23, 0xC8, 512, 512, 117248, 23, 0xC4, 512, 512,
            117248, 23, 0xC9, 512, 512, 117248, 23, 0xC5, 512, 512, 117248, 24, 0xC1, 512, 512,
            117248, 24, 0xC6, 512, 512, 117248, 24, 0xC2, 512, 512, 117248, 24, 0xC7, 512, 512,
            117248, 24, 0xC3, 512, 512, 117248, 24, 0xC8, 512, 512, 117248, 24, 0xC4, 512, 512,
            117248, 24, 0xC9, 512, 512, 117248, 24, 0xC5, 512, 512, 117248, 25, 0xC1, 512, 512,
            117248, 25, 0xC6, 512, 512, 117248, 25, 0xC2, 512, 512, 117248, 25, 0xC7, 512, 512,
            117248, 25, 0xC3, 512, 512, 117248, 25, 0xC8, 512, 512, 117248, 25, 0xC4, 512, 512,
            117248, 25, 0xC9, 512, 512, 117248, 25, 0xC5, 512, 512, 117248, 26, 0xC1, 512, 512,
            117248, 26, 0xC6, 512, 512, 117248, 26, 0xC2, 512, 512, 117248, 26, 0xC7, 512, 512,
            117248, 26, 0xC3, 512, 512, 117248, 26, 0xC8, 512, 512, 117248, 26, 0xC4, 512, 512,
            117248, 26, 0xC9, 512, 512, 117248, 26, 0xC5, 512, 512, 117248, 27, 0xC1, 512, 512,
            117248, 27, 0xC6, 512, 512, 117248, 27, 0xC2, 512, 512, 117248, 27, 0xC7, 512, 512,
            117248, 27, 0xC3, 512, 512, 117248, 27, 0xC8, 512, 512, 117248, 27, 0xC4, 512, 512,
            117248, 27, 0xC9, 512, 512, 117248, 27, 0xC5, 512, 512, 117248, 28, 0xC1, 512, 512,
            117248, 28, 0xC6, 512, 512, 117248, 28, 0xC2, 512, 512, 117248, 28, 0xC7, 512, 512,
            117248, 28, 0xC3, 512, 512, 117248, 28, 0xC8, 512, 512, 117248, 28, 0xC4, 512, 512,
            117248, 28, 0xC9, 512, 512, 117248, 28, 0xC5, 512, 512, 117248, 29, 0xC1, 512, 512,
            117248, 29, 0xC6, 512, 512, 117248, 29, 0xC2, 512, 512, 117248, 29, 0xC7, 512, 512,
            117248, 29, 0xC3, 512, 512, 117248, 29, 0xC8, 512, 512, 117248, 29, 0xC4, 512, 512,
            117248, 29, 0xC9, 512, 512, 117248, 29, 0xC5, 512, 512, 117248, 30, 0xC1, 512, 512,
            117248, 30, 0xC6, 512, 512, 117248, 30, 0xC2, 512, 512, 117248, 30, 0xC7, 512, 512,
            117248, 30, 0xC3, 512, 512, 117248, 30, 0xC8, 512, 512, 117248, 30, 0xC4, 512, 512,
            117248, 30, 0xC9, 512, 512, 117248, 30, 0xC5, 512, 512, 117248, 31, 0xC1, 512, 512,
            117248, 31, 0xC6, 512, 512, 117248, 31, 0xC2, 512, 512, 117248, 31, 0xC7, 512, 512,
            117248, 31, 0xC3, 512, 512, 117248, 31, 0xC8, 512, 512, 117248, 31, 0xC4, 512, 512,
            117248, 31, 0xC9, 512, 512, 117248, 31, 0xC5, 512, 512, 117248, 32, 0xC1, 512, 512,
            117248, 32, 0xC6, 512, 512, 117248, 32, 0xC2, 512, 512, 117248, 32, 0xC7, 512, 512,
            117248, 32, 0xC3, 512, 512, 117248, 32, 0xC8, 512, 512, 117248, 32, 0xC4, 512, 512,
            117248, 32, 0xC9, 512, 512, 117248, 32, 0xC5, 512, 512, 117248, 33, 0xC1, 512, 512,
            117248, 33, 0xC6, 512, 512, 117248, 33, 0xC2, 512, 512, 117248, 33, 0xC7, 512, 512,
            117248, 33, 0xC3, 512, 512, 117248, 33, 0xC8, 512, 512, 117248, 33, 0xC4, 512, 512,
            117248, 33, 0xC9, 512, 512, 117248, 33, 0xC5, 512, 512, 117248, 34, 0xC1, 512, 512,
            117248, 34, 0xC6, 512, 512, 117248, 34, 0xC2, 512, 512, 117248, 34, 0xC7, 512, 512,
            117248, 34, 0xC3, 512, 512, 117248, 34, 0xC8, 512, 512, 117248, 34, 0xC4, 512, 512,
            117248, 34, 0xC9, 512, 512, 117248, 34, 0xC5, 512, 512, 117248, 35, 0xC1, 512, 512,
            117248, 35, 0xC6, 512, 512, 117248, 35, 0xC2, 512, 512, 117248, 35, 0xC7, 512, 512,
            117248, 35, 0xC3, 512, 512, 117248, 35, 0xC8, 512, 512, 117248, 35, 0xC4, 512, 512,
            117248, 35, 0xC9, 512, 512, 117248, 35, 0xC5, 512, 512, 117248, 36, 0xC1, 512, 512,
            117248, 36, 0xC6, 512, 512, 117248, 36, 0xC2, 512, 512, 117248, 36, 0xC7, 512, 512,
            117248, 36, 0xC3, 512, 512, 117248, 36, 0xC8, 512, 512, 117248, 36, 0xC4, 512, 512,
            117248, 36, 0xC9, 512, 512, 117248, 36, 0xC5, 512, 512, 117248, 37, 0xC1, 512, 512,
            117248, 37, 0xC6, 512, 512, 117248, 37, 0xC2, 512, 512, 117248, 37, 0xC7, 512, 512,
            117248, 37, 0xC3, 512, 512, 117248, 37, 0xC8, 512, 512, 117248, 37, 0xC4, 512, 512,
            117248, 37, 0xC9, 512, 512, 117248, 37, 0xC5, 512, 512, 117248, 38, 0xC1, 512, 512,
            117248, 38, 0xC6, 512, 512, 117248, 38, 0xC2, 512, 512, 117248, 38, 0xC7, 512, 512,
            117248, 38, 0xC3, 512, 512, 117248, 38, 0xC8, 512, 512, 117248, 38, 0xC4, 512, 512,
            117248, 38, 0xC9, 512, 512, 117248, 38, 0xC5, 512, 512, 117248, 39, 0xC1, 512, 512,
            117248, 39, 0xC6, 512, 512, 117248, 39, 0xC2, 512, 512, 117248, 39, 0xC7, 512, 512,
            117248, 39, 0xC3, 512, 512, 117248, 39, 0xC8, 512, 512, 117248, 39, 0xC4, 512, 512,
            117248, 39, 0xC9, 512, 512, 117248, 39, 0xC5, 512, 512, 117248, 40, 0xC1, 512, 512,
            117248, 40, 0xC6, 512, 512, 117248, 40, 0xC2, 512, 512, 117248, 40, 0xC7, 512, 512,
            117248, 40, 0xC3, 512, 512, 117248, 40, 0xC8, 512, 512, 117248, 40, 0xC4, 512, 512,
            117248, 40, 0xC9, 512, 512, 117248, 40, 0xC5, 512, 512, 117248, 41, 0xC1, 512, 512,
            117248, 41, 0xC6, 512, 512, 117248, 41, 0xC2, 512, 512, 117248, 41, 0xC7, 512, 512,
            117248, 41, 0xC3, 512, 512, 117248, 41, 0xC8, 512, 512, 117248, 41, 0xC4, 512, 512,
            117248, 41, 0xC9, 512, 512, 117248, 41, 0xC5, 512, 512, 117248
        ];

        for mut chunk in &sectors_content.iter().chunks(5) {
            let track = *chunk.next().unwrap() as u8;
            let sector = *chunk.next().unwrap() as u8;
            let _size = *chunk.next().unwrap() as usize;
            let _real_size = *chunk.next().unwrap() as usize;
            let value = *chunk.next().unwrap() as usize;

            assert_eq!(
                dsk.sectors_bytes(cpclib::disc::edsk::Head::A, track, sector, 1)
                    .unwrap()
                    .iter()
                    .map(|&v| v as usize)
                    .sum::<usize>(),
                value
            );
        }

        // TODO find a way to count the sectors iused
        // assert_eq!(
        // dsk.sectors_used().count(),
        // 65
        // );

        use cpclib::disc::amsdos::*;
        let amsdos = AmsdosManager::new_from_disc(dsk, cpclib::disc::edsk::Head::A);
        let entries = dbg!(amsdos.catalog().to_amsdos_catalog());

        assert_eq!(entries.len(), 1);

        let entry = dbg!(&entries[0]);
        let filename = entry.file_name();

        assert_eq!(filename.name(), "-CED-");

        assert_eq!(filename.extension(), "EXE");

        assert_eq!(filename.user(), 0);

        assert!(!entry.read_only());
        assert!(!entry.system());

        assert_eq!(entry.size(), 32);
    }

    #[test]
    pub fn test_turlogh() {
        use cpclib_common::itertools::Itertools;

        let dsk = cpclib::disc::edsk::ExtendedDsk::open(
            "tests/dsk/Turlogh Le Rodeur (F) (Face A) (1987) [Original] (GAPS).dsk"
        )
        .unwrap();

        let sectors_content = [
            0, 0xC1, 512, 15774, 0, 0xC2, 512, 17835, 0, 0xC3, 512, 25251, 0, 0xC4, 512, 117248, 0,
            0xC5, 512, 54672, 0, 0xC6, 512, 66106, 0, 0xC7, 512, 65803, 0, 0xC8, 512, 66235, 0,
            0xC9, 512, 65355, 1, 0xC1, 512, 68412, 1, 0xC2, 512, 67173, 1, 0xC3, 512, 66520, 1,
            0xC4, 512, 68534, 1, 0xC5, 512, 23856, 1, 0xC6, 512, 60244, 1, 0xC7, 512, 64928, 1,
            0xC8, 512, 65178, 1, 0xC9, 512, 65551, 2, 0xC1, 512, 67519, 2, 0xC2, 512, 67295, 2,
            0xC3, 512, 65866, 2, 0xC4, 512, 66754, 2, 0xC5, 512, 67547, 2, 0xC6, 512, 29766, 2,
            0xC7, 512, 59088, 2, 0xC8, 512, 46802, 2, 0xC9, 512, 58753, 3, 0xC1, 512, 32483, 3,
            0xC2, 512, 55759, 3, 0xC3, 512, 68595, 3, 0xC4, 512, 64215, 3, 0xC5, 512, 66184, 3,
            0xC6, 512, 65053, 3, 0xC7, 512, 67025, 3, 0xC8, 512, 65692, 3, 0xC9, 512, 67960, 4,
            0xC1, 512, 67919, 4, 0xC2, 512, 63107, 4, 0xC3, 512, 63378, 4, 0xC4, 512, 66848, 4,
            0xC5, 512, 66548, 4, 0xC6, 512, 67219, 4, 0xC7, 512, 65944, 4, 0xC8, 512, 68080, 4,
            0xC9, 512, 69762, 5, 0xC1, 512, 61801, 5, 0xC2, 512, 67000, 5, 0xC3, 512, 66421, 5,
            0xC4, 512, 66203, 5, 0xC5, 512, 65580, 5, 0xC6, 512, 68384, 5, 0xC7, 512, 66873, 5,
            0xC8, 512, 64957, 5, 0xC9, 512, 63800, 6, 0xC1, 512, 67565, 6, 0xC2, 512, 66734, 6,
            0xC3, 512, 67852, 6, 0xC4, 512, 67699, 6, 0xC5, 512, 68217, 6, 0xC6, 512, 68271, 6,
            0xC7, 512, 68438, 6, 0xC8, 512, 68025, 6, 0xC9, 512, 58674, 7, 0xC1, 512, 35939, 7,
            0xC2, 512, 55191, 7, 0xC3, 512, 65192, 7, 0xC4, 512, 67416, 7, 0xC5, 512, 67775, 7,
            0xC6, 512, 66523, 7, 0xC7, 512, 65247, 7, 0xC8, 512, 65488, 7, 0xC9, 512, 67415, 8,
            0xC1, 512, 67870, 8, 0xC2, 512, 68211, 8, 0xC3, 512, 62911, 8, 0xC4, 512, 65701, 8,
            0xC5, 512, 64549, 8, 0xC6, 512, 66870, 8, 0xC7, 512, 66016, 8, 0xC8, 512, 66349, 8,
            0xC9, 512, 66494, 9, 0xC1, 512, 63652, 9, 0xC2, 512, 67175, 9, 0xC3, 512, 65221, 9,
            0xC4, 512, 65370, 9, 0xC5, 512, 65595, 9, 0xC6, 512, 64872, 9, 0xC7, 512, 66446, 9,
            0xC8, 512, 68431, 9, 0xC9, 512, 65906, 10, 0xC1, 512, 68115, 10, 0xC2, 512, 67510, 10,
            0xC3, 512, 68095, 10, 0xC4, 512, 69421, 10, 0xC5, 512, 69353, 10, 0xC6, 512, 68030, 10,
            0xC7, 512, 67508, 10, 0xC8, 512, 67122, 10, 0xC9, 512, 57250, 11, 0xC1, 512, 28735, 11,
            0xC2, 512, 54498, 11, 0xC3, 512, 53934, 11, 0xC4, 512, 52136, 11, 0xC5, 512, 58567, 11,
            0xC6, 512, 42726, 11, 0xC7, 512, 20330, 11, 0xC8, 512, 11985, 11, 0xC9, 512, 53934, 12,
            0xC1, 512, 14527, 12, 0xC2, 512, 58567, 12, 0xC3, 512, 44630, 12, 0xC4, 512, 44839, 12,
            0xC5, 512, 78854, 12, 0xC6, 512, 62872, 12, 0xC7, 512, 11286, 12, 0xC8, 512, 24326, 12,
            0xC9, 512, 19785, 13, 0xC1, 512, 24326, 13, 0xC2, 512, 26970, 13, 0xC3, 512, 22813, 13,
            0xC4, 512, 11055, 13, 0xC5, 512, 57018, 13, 0xC6, 512, 46324, 13, 0xC7, 512, 57018, 13,
            0xC8, 512, 57913, 13, 0xC9, 512, 43763, 14, 0xC1, 512, 39783, 14, 0xC2, 512, 11134, 14,
            0xC3, 512, 11358, 14, 0xC4, 512, 57018, 14, 0xC5, 512, 43185, 14, 0xC6, 512, 58775, 14,
            0xC7, 512, 57261, 14, 0xC8, 512, 51413, 14, 0xC9, 512, 65429, 15, 0xC1, 512, 71519, 15,
            0xC2, 512, 65455, 15, 0xC3, 512, 22765, 15, 0xC4, 512, 16324, 15, 0xC5, 512, 38027, 15,
            0xC6, 512, 58303, 15, 0xC7, 512, 57011, 15, 0xC8, 512, 57316, 15, 0xC9, 512, 43098, 16,
            0xC1, 512, 42500, 16, 0xC2, 512, 50200, 16, 0xC3, 512, 53395, 16, 0xC4, 512, 20764, 16,
            0xC5, 512, 43993, 16, 0xC6, 512, 51768, 16, 0xC7, 512, 60665, 16, 0xC8, 512, 50200, 16,
            0xC9, 512, 55522, 17, 0xC1, 512, 69898, 17, 0xC2, 512, 66203, 17, 0xC3, 512, 76170, 17,
            0xC4, 512, 72548, 17, 0xC5, 512, 77009, 17, 0xC6, 512, 67533, 17, 0xC7, 512, 56373, 17,
            0xC8, 512, 57286, 17, 0xC9, 512, 54475, 18, 0xC1, 512, 63460, 18, 0xC2, 512, 65304, 18,
            0xC3, 512, 36378, 18, 0xC4, 512, 56373, 18, 0xC5, 512, 53319, 18, 0xC6, 512, 56377, 18,
            0xC7, 512, 53660, 18, 0xC8, 512, 72256, 18, 0xC9, 512, 56095, 19, 0xC1, 512, 56958, 19,
            0xC2, 512, 23412, 19, 0xC3, 512, 56377, 19, 0xC4, 512, 70638, 19, 0xC5, 512, 87202, 19,
            0xC6, 512, 74659, 19, 0xC7, 512, 84554, 19, 0xC8, 512, 84668, 19, 0xC9, 512, 61046, 20,
            0xC1, 512, 38109, 20, 0xC2, 512, 61046, 20, 0xC3, 512, 59771, 20, 0xC4, 512, 69269, 20,
            0xC5, 512, 58956, 20, 0xC6, 512, 70926, 20, 0xC7, 512, 9470, 20, 0xC8, 512, 61046, 20,
            0xC9, 512, 40130, 21, 0xC1, 512, 58065, 21, 0xC2, 512, 63217, 21, 0xC3, 512, 49912, 21,
            0xC4, 512, 43145, 21, 0xC5, 512, 54118, 21, 0xC6, 512, 38959, 21, 0xC7, 512, 55284, 21,
            0xC8, 512, 49344, 21, 0xC9, 512, 50001, 22, 0xC1, 512, 43600, 22, 0xC2, 512, 46044, 22,
            0xC3, 512, 45364, 22, 0xC4, 512, 64336, 22, 0xC5, 512, 64304, 22, 0xC6, 512, 73415, 22,
            0xC7, 512, 73824, 22, 0xC8, 512, 68011, 22, 0xC9, 512, 54132, 23, 0xC1, 512, 64336, 23,
            0xC2, 512, 36524, 23, 0xC3, 512, 41360, 23, 0xC4, 512, 40042, 23, 0xC5, 512, 40615, 23,
            0xC6, 512, 52794, 23, 0xC7, 512, 62110, 23, 0xC8, 512, 66404, 23, 0xC9, 512, 62110, 24,
            0xC1, 512, 64934, 24, 0xC2, 512, 69304, 24, 0xC3, 512, 61844, 24, 0xC4, 512, 69950, 24,
            0xC5, 512, 65476, 24, 0xC6, 512, 62695, 24, 0xC7, 512, 60316, 24, 0xC8, 512, 55609, 24,
            0xC9, 512, 59508, 25, 0xC1, 512, 59262, 25, 0xC2, 512, 62511, 25, 0xC3, 512, 68742, 25,
            0xC4, 512, 76071, 25, 0xC5, 512, 86005, 25, 0xC6, 512, 76234, 25, 0xC7, 512, 56223, 25,
            0xC8, 512, 73230, 25, 0xC9, 512, 72878, 26, 0xC1, 512, 23793, 26, 0xC2, 512, 86005, 26,
            0xC3, 512, 58783, 26, 0xC4, 512, 69069, 26, 0xC5, 512, 68404, 26, 0xC6, 512, 67587, 26,
            0xC7, 512, 70569, 26, 0xC8, 512, 35382, 26, 0xC9, 512, 42657, 27, 0xC1, 512, 44748, 27,
            0xC2, 512, 48033, 27, 0xC3, 512, 63871, 27, 0xC4, 512, 51414, 27, 0xC5, 512, 29430, 27,
            0xC6, 512, 42324, 27, 0xC7, 512, 44748, 27, 0xC8, 512, 62823, 27, 0xC9, 512, 89063, 28,
            0xC1, 512, 87956, 28, 0xC2, 512, 44222, 28, 0xC3, 512, 42621, 28, 0xC4, 512, 57913, 28,
            0xC5, 512, 54815, 28, 0xC6, 512, 52757, 28, 0xC7, 512, 17720, 28, 0xC8, 512, 44222, 28,
            0xC9, 512, 54153, 29, 0xC1, 512, 54026, 29, 0xC2, 512, 65193, 29, 0xC3, 512, 69269, 29,
            0xC4, 512, 43105, 29, 0xC5, 512, 65150, 29, 0xC6, 512, 39701, 29, 0xC7, 512, 65150, 29,
            0xC8, 512, 59072, 29, 0xC9, 512, 60058, 30, 0xC1, 512, 65383, 30, 0xC2, 512, 82741, 30,
            0xC3, 512, 43275, 30, 0xC4, 512, 59804, 30, 0xC5, 512, 56772, 30, 0xC6, 512, 61708, 30,
            0xC7, 512, 65116, 30, 0xC8, 512, 41276, 30, 0xC9, 512, 25664, 31, 0xC1, 512, 24092, 31,
            0xC2, 512, 28580, 31, 0xC3, 512, 33314, 31, 0xC4, 512, 35249, 31, 0xC5, 512, 50727, 31,
            0xC6, 512, 62727, 31, 0xC7, 512, 43655, 31, 0xC8, 512, 44201, 31, 0xC9, 512, 48118, 32,
            0xC1, 512, 44862, 32, 0xC2, 512, 56973, 32, 0xC3, 512, 59822, 32, 0xC4, 512, 43655, 32,
            0xC5, 512, 63762, 32, 0xC6, 512, 89554, 32, 0xC7, 512, 71664, 32, 0xC8, 512, 52701, 32,
            0xC9, 512, 65121, 33, 0xC1, 512, 46723, 33, 0xC2, 512, 67431, 33, 0xC3, 512, 73832, 33,
            0xC4, 512, 60221, 33, 0xC5, 512, 56261, 33, 0xC6, 512, 68944, 33, 0xC7, 512, 75455, 33,
            0xC8, 512, 41176, 33, 0xC9, 512, 73832, 34, 0xC1, 512, 39195, 34, 0xC2, 512, 51061, 34,
            0xC3, 512, 45616, 34, 0xC4, 512, 42525, 34, 0xC5, 512, 47417, 34, 0xC6, 512, 57762, 34,
            0xC7, 512, 57934, 34, 0xC8, 512, 32699, 34, 0xC9, 512, 39430, 35, 0xC1, 512, 52984, 35,
            0xC2, 512, 50722, 35, 0xC3, 512, 51594, 35, 0xC4, 512, 53681, 35, 0xC5, 512, 57444, 35,
            0xC6, 512, 53762, 35, 0xC7, 512, 24179, 35, 0xC8, 512, 33496, 35, 0xC9, 512, 54881, 36,
            0xC1, 512, 50407, 36, 0xC2, 512, 57397, 36, 0xC3, 512, 52930, 36, 0xC4, 512, 56931, 36,
            0xC5, 512, 45258, 36, 0xC6, 512, 50695, 36, 0xC7, 512, 65902, 36, 0xC8, 512, 67556, 36,
            0xC9, 512, 60602, 37, 0xC1, 512, 54698, 37, 0xC2, 512, 31249, 37, 0xC3, 512, 50695, 37,
            0xC4, 512, 63201, 37, 0xC5, 512, 62659, 37, 0xC6, 512, 62915, 37, 0xC7, 512, 76457, 37,
            0xC8, 512, 91588, 37, 0xC9, 512, 94479, 38, 0xC1, 512, 63067, 38, 0xC2, 512, 62659, 38,
            0xC3, 512, 58145, 38, 0xC4, 512, 76457, 38, 0xC5, 512, 41988, 38, 0xC6, 512, 94479, 38,
            0xC7, 512, 54993, 38, 0xC8, 512, 117248, 38, 0xC9, 512, 117248, 39, 0x41, 512, 117248,
            39, 0x46, 512, 117248, 39, 0x42, 512, 117248, 39, 0x47, 512, 117248, 39, 0x43, 512,
            117248, 39, 0x48, 512, 117248, 39, 0x44, 512, 117248, 39, 0x49, 512, 117248, 39, 0x45,
            512, 117248
        ];

        for mut chunk in &sectors_content.iter().chunks(4) {
            let track = *chunk.next().unwrap() as u8;
            let sector = *chunk.next().unwrap() as u8;
            let size = *chunk.next().unwrap() as usize;
            let value = *chunk.next().unwrap() as usize;

            let sector = dsk
                .sector(cpclib::disc::edsk::Head::A, track, sector)
                .unwrap();

            assert_eq!(size as u16, sector.len());

            assert_eq!(size, sector.values().len());
            let sum = sector.values().iter().map(|&v| v as usize).sum::<usize>();

            assert_eq!(value, sum);
        }

        let tracks_content = [
            494279, 550396, 529390, 552966, 598805, 591019, 601475, 556186, 594971, 592668, 602404,
            376845, 359686, 347200, 395356, 429818, 429007, 597495, 513222, 599514, 468723, 462045,
            533030, 466295, 569636, 631156, 522249, 476454, 456379, 520724, 501739, 370663, 548114,
            563875, 413639, 432743, 507678, 587941, 686284, 1055232
        ];

        for (idx, value) in tracks_content.iter().enumerate() {
            let track = dsk
                .get_track_information(cpclib::disc::edsk::Head::A, idx as u8)
                .unwrap();
            let track_bytes = dsk
                .track_bytes(cpclib::disc::edsk::Head::A, idx as u8)
                .unwrap();
            let obtained = track_bytes.iter().map(|&v| v as usize).sum::<usize>();
            let nb = track_bytes.iter().count();
            let gap = track.gap3_length();
            let track_size = track.compute_track_size();

            dbg!((idx, *value, obtained, nb, track_size, gap));

            assert_eq!(obtained, *value);
        }

        use cpclib::disc::amsdos::*;
        let amsdos = AmsdosManager::new_from_disc(dsk, cpclib::disc::edsk::Head::A);
        let entries = dbg!(amsdos
            .catalog()
            .to_amsdos_catalog()
            .sorted_physically_and_alphabetically());

        assert_eq!(entries.len(), 45);
        let expected_values = [
            (0, "ARMES ", "BIN", "3 Ko"),
            (0, "ASSEM ", "BIN", "3 Ko"),
            (0, "BANC ", "BIN", "1 Ko"),
            (0, "BASIC1 ", "BAS", "18 Ko"),
            (0, "BASIC2 ", "BAS", "18 Ko"),
            (0, "BOOT2 ", "BAS", "1 Ko"),
            (0, "DEMO ", "BAS", "1 Ko"),
            (0, "ENCORE ", "BIN", "3 Ko"),
            (0, "GENERIC ", "BAS", "5 Ko"),
            (0, "ICOCOMB1", "BIN", "2 Ko"),
            (0, "ICOCOMB2", "BIN", "1 Ko"),
            (0, "INTRO ", "BAS", "5 Ko"),
            (0, "LOAD ", "BIN", "1 Ko"),
            (0, "VERSION ", "BIN", "1 Ko"),
            (0, "1", "CPT", "4 Ko"),
            (0, "18", "CPT", "3 Ko"),
            (0, "19", "CPT", "4 Ko"),
            (0, "2", "CPT", "3 Ko"),
            (0, "20", "CPT", "3 Ko"),
            (0, "3", "CPT", "4 Ko"),
            (0, "325", "CPT", "3 Ko"),
            (0, "33", "CPT", "4 Ko"),
            (0, "4", "CPT", "2 Ko"),
            (0, "6", "CPT", "3 Ko"),
            (0, "DES ", "CPT", "1 Ko"),
            (0, "GARDE ", "CPT", "11 Ko"),
            (0, "ICOMENU ", "BIN", "4 Ko"),
            (0, "MUSIC ", "BIN", "3 Ko"),
            (0, "PROTO ", "CPT", "7 Ko"),
            (0, "TURLOG ", "BIN", "3 Ko"),
            (0, "101", "CPT", "3 Ko"),
            (0, "120", "CPT", "3 Ko"),
            (0, "125", "CPT", "4 Ko"),
            (0, "14", "CPT", "3 Ko"),
            (0, "17", "CPT", "4 Ko"),
            (0, "22", "CPT", "3 Ko"),
            (0, "225", "CPT", "4 Ko"),
            (0, "25", "CPT", "4 Ko"),
            (0, "35", "CPT", "3 Ko"),
            (0, "5", "CPT", "3 Ko"),
            (0, "500", "CPT", "4 Ko"),
            (0, "BOOT ", "BAS", "1 Ko"),
            (0, "BOOT1 ", "BAS", "1 Ko"),
            (0, "GENER ", "CPT", "7 Ko"),
            (0, "JEU ", "BAS", "1 Ko")
        ];

        for (idx, expected_entry) in expected_values.iter().enumerate() {
            let entry = &entries[idx];
            assert_eq!(expected_entry.0, entry.file_name().user());

            assert_eq!(expected_entry.1.trim(), entry.file_name().name());

            assert_eq!(expected_entry.2.trim(), entry.file_name().extension());

            assert_eq!(expected_entry.3, format!("{} Ko", entry.size()));

            assert!(entry.system());
            assert!(!entry.read_only());
        }
    }
}
