use crate::NamedNumber::*;
use crate::NamedNumber;
use decimal::d128;

/// Returns the corresponding [`d128`] of a [`NamedNumber`]
pub fn lookup_named_number(named_number: &NamedNumber) -> d128 {
  match named_number {
    Hundred           => d128!(100),
    Thousand          => d128!(1000),
    Million           => d128!(1000000),
    Billion           => d128!(1000000000),
    Trillion          => d128!(1000000000000),
    Quadrillion       => d128!(1000000000000000),
    Quintillion       => d128!(1000000000000000000),
    Sextillion        => d128!(1000000000000000000000),
    Septillion        => d128!(1000000000000000000000000),
    Octillion         => d128!(1000000000000000000000000000),
    Nonillion         => d128!(1000000000000000000000000000000),
    Decillion         => d128!(1000000000000000000000000000000000),
    Undecillion       => d128!(1000000000000000000000000000000000000),
    Duodecillion      => d128!(1E+39),
    Tredecillion      => d128!(1E+42),
    Quattuordecillion => d128!(1E+45),
    Quindecillion     => d128!(1E+48),
    Sexdecillion      => d128!(1E+51),
    Septendecillion   => d128!(1E+54),
    Octodecillion     => d128!(1E+57),
    Novemdecillion    => d128!(1E+60),
    Vigintillion      => d128!(1E+63),
    Googol            => d128!(1E+100),
    Centillion        => d128!(1E+303),
  }
}

/// Returns the factorial of an `i32` as a [`struct@d128`]
pub fn lookup_factorial(n: i32) -> d128 {
  match n {
    0 => d128!(1),
    1 => d128!(1),
    2 => d128!(2),
    3 => d128!(6),
    4 => d128!(24),
    5 => d128!(120),
    6 => d128!(720),
    7 => d128!(5040),
    8 => d128!(40320),
    9 => d128!(362880),
    10 => d128!(3628800),
    11 => d128!(39916800),
    12 => d128!(479001600),
    13 => d128!(6227020800),
    14 => d128!(87178291200),
    15 => d128!(1307674368000),
    16 => d128!(20922789888000),
    17 => d128!(355687428096000),
    18 => d128!(6402373705728000),
    19 => d128!(121645100408832000),
    20 => d128!(2432902008176640000),
    21 => d128!(51090942171709440000),
    22 => d128!(1124000727777607680000),
    23 => d128!(25852016738884976640000),
    24 => d128!(620448401733239439360000),
    25 => d128!(15511210043330985984000000),
    26 => d128!(403291461126605635584000000),
    27 => d128!(10888869450418352160768000000),
    28 => d128!(304888344611713860501504000000),
    29 => d128!(8841761993739701954543616000000),
    30 => d128!(265252859812191058636308480000000),
    31 => d128!(8222838654177922817725562880000000),
    32 => d128!(2.631308369336935301672180121600000E+35),
    33 => d128!(8.683317618811886495518194401280000E+36),
    34 => d128!(2.952327990396041408476186096435200E+38),
    35 => d128!(1.033314796638614492966665133752320E+40),
    36 => d128!(3.719933267899012174679994481508352E+41),
    37 => d128!(1.376375309122634504631597958158090E+43),
    38 => d128!(5.230226174666011117600072241000742E+44),
    39 => d128!(2.039788208119744335864028173990289E+46),
    40 => d128!(8.159152832478977343456112695961156E+47),
    41 => d128!(3.345252661316380710817006205344074E+49),
    42 => d128!(1.405006117752879898543142606244511E+51),
    43 => d128!(6.041526306337383563735513206851397E+52),
    44 => d128!(2.658271574788448768043625811014615E+54),
    45 => d128!(1.196222208654801945619631614956577E+56),
    46 => d128!(5.502622159812088949850305428800254E+57),
    47 => d128!(2.586232415111681806429643551536119E+59),
    48 => d128!(1.241391559253607267086228904737337E+61),
    49 => d128!(6.082818640342675608722521633212951E+62),
    50 => d128!(3.041409320171337804361260816606476E+64),
    51 => d128!(1.551118753287382280224243016469303E+66),
    52 => d128!(8.065817517094387857166063685640376E+67),
    53 => d128!(4.274883284060025564298013753389399E+69),
    54 => d128!(2.308436973392413804720927426830275E+71),
    55 => d128!(1.269640335365827592596510084756651E+73),
    56 => d128!(7.109985878048634518540456474637246E+74),
    57 => d128!(4.052691950487721675568060190543230E+76),
    58 => d128!(2.350561331282878571829474910515073E+78),
    59 => d128!(1.386831185456898357379390197203893E+80),
    60 => d128!(8.320987112741390144276341183223358E+81),
    61 => d128!(5.075802138772247988008568121766248E+83),
    62 => d128!(3.146997326038793752565312235495074E+85),
    63 => d128!(1.982608315404440064116146708361897E+87),
    64 => d128!(1.268869321858841641034333893351614E+89),
    65 => d128!(8.247650592082470666723170306785491E+90),
    66 => d128!(5.443449390774430640037292402478424E+92),
    67 => d128!(3.647111091818868528824985909660544E+94),
    68 => d128!(2.480035542436830599600990418569170E+96),
    69 => d128!(1.711224524281413113724683388812727E+98),
    70 => d128!(1.197857166996989179607278372168909E+100),
    71 => d128!(8.504785885678623175211676442399254E+101),
    72 => d128!(6.123445837688608686152407038527463E+103),
    73 => d128!(4.470115461512684340891257138125048E+105),
    74 => d128!(3.307885441519386412259530282212536E+107),
    75 => d128!(2.480914081139539809194647711659402E+109),
    76 => d128!(1.885494701666050254987932260861146E+111),
    77 => d128!(1.451830920282858696340707840863082E+113),
    78 => d128!(1.132428117820629783145752115873204E+115),
    79 => d128!(8.946182130782975286851441715398312E+116),
    80 => d128!(7.156945704626380229481153372318650E+118),
    81 => d128!(5.797126020747367985879734231578106E+120),
    82 => d128!(4.753643337012841748421382069894047E+122),
    83 => d128!(3.945523969720658651189747118012059E+124),
    84 => d128!(3.314240134565353266999387579130130E+126),
    85 => d128!(2.817104114380550276949479442260610E+128),
    86 => d128!(2.422709538367273238176552320344125E+130),
    87 => d128!(2.107757298379527717213600518699389E+132),
    88 => d128!(1.854826422573984391147968456455462E+134),
    89 => d128!(1.650795516090846108121691926245361E+136),
    90 => d128!(1.485715964481761497309522733620825E+138),
    91 => d128!(1.352001527678402962551665687594951E+140),
    92 => d128!(1.243841405464130725547532432587355E+142),
    93 => d128!(1.156772507081641574759205162306240E+144),
    94 => d128!(1.087366156656743080273652852567866E+146),
    95 => d128!(1.032997848823905926259970209939473E+148),
    96 => d128!(9.916779348709496892095714015418941E+149),
    97 => d128!(9.619275968248211985332842594956373E+151),
    98 => d128!(9.426890448883247745626185743057246E+153),
    99 => d128!(9.332621544394415268169923885626674E+155),
    100 => d128!(9.332621544394415268169923885626674E+157),
    101 => d128!(9.425947759838359420851623124482941E+159),
    102 => d128!(9.614466715035126609268655586972600E+161),
    103 => d128!(9.902900716486180407546715254581778E+163),
    104 => d128!(1.029901674514562762384858386476505E+166),
    105 => d128!(1.081396758240290900504101305800330E+168),
    106 => d128!(1.146280563734708354534347384148350E+170),
    107 => d128!(1.226520203196137939351751701038734E+172),
    108 => d128!(1.324641819451828974499891837121833E+174),
    109 => d128!(1.443859583202493582204882102462798E+176),
    110 => d128!(1.588245541522742940425370312709078E+178),
    111 => d128!(1.762952551090244663872161047107077E+180),
    112 => d128!(1.974506857221074023536820372759926E+182),
    113 => d128!(2.231192748659813646596607021218716E+184),
    114 => d128!(2.543559733472187557120132004189336E+186),
    115 => d128!(2.925093693493015690688151804817736E+188),
    116 => d128!(3.393108684451898201198256093588574E+190),
    117 => d128!(3.969937160808720895401959629498632E+192),
    118 => d128!(4.684525849754290656574312362808386E+194),
    119 => d128!(5.574585761207605881323431711741979E+196),
    120 => d128!(6.689502913449127057588118054090375E+198),
    121 => d128!(8.094298525273443739681622845449354E+200),
    122 => d128!(9.875044200833601362411579871448212E+202),
    123 => d128!(1.214630436702532967576624324188130E+205),
    124 => d128!(1.506141741511140879795014161993281E+207),
    125 => d128!(1.882677176888926099743767702491601E+209),
    126 => d128!(2.372173242880046885677147305139417E+211),
    127 => d128!(3.012660018457659544809977077527060E+213),
    128 => d128!(3.856204823625804217356770659234637E+215),
    129 => d128!(4.974504222477287440390234150412682E+217),
    130 => d128!(6.466855489220473672507304395536487E+219),
    131 => d128!(8.471580690878820510984568758152798E+221),
    132 => d128!(1.118248651196004307449963076076169E+224),
    133 => d128!(1.487270706090685728908450891181305E+226),
    134 => d128!(1.992942746161518876737324194182949E+228),
    135 => d128!(2.690472707318050483595387662146981E+230),
    136 => d128!(3.659042881952548657689727220519894E+232),
    137 => d128!(5.012888748274991661034926292112255E+234),
    138 => d128!(6.917786472619488492228198283114912E+236),
    139 => d128!(9.615723196941089004197195613529728E+238),
    140 => d128!(1.346201247571752460587607385894162E+241),
    141 => d128!(1.898143759076170969428526414110768E+243),
    142 => d128!(2.695364137888162776588507508037291E+245),
    143 => d128!(3.854370717180072770521565736493326E+247),
    144 => d128!(5.550293832739304789551054660550389E+249),
    145 => d128!(8.047926057471991944849029257798064E+251),
    146 => d128!(1.174997204390910823947958271638517E+254),
    147 => d128!(1.727245890454638911203498659308620E+256),
    148 => d128!(2.556323917872865588581178015776758E+258),
    149 => d128!(3.808922637630569726985955243507369E+260),
    150 => d128!(5.713383956445854590478932865261054E+262),
    151 => d128!(8.627209774233240431623188626544192E+264),
    152 => d128!(1.311335885683452545606724671234717E+267),
    153 => d128!(2.006343905095682394778288746989117E+269),
    154 => d128!(3.089769613847350887958564670363240E+271),
    155 => d128!(4.789142901463393876335775239063022E+273),
    156 => d128!(7.471062926282894447083809372938314E+275),
    157 => d128!(1.172956879426414428192158071551315E+278),
    158 => d128!(1.853271869493734796543609753051078E+280),
    159 => d128!(2.946702272495038326504339507351214E+282),
    160 => d128!(4.714723635992061322406943211761942E+284),
    161 => d128!(7.590705053947218729075178570936727E+286),
    162 => d128!(1.229694218739449434110178928491750E+289),
    163 => d128!(2.004401576545302577599591653441552E+291),
    164 => d128!(3.287218585534296227263330311644145E+293),
    165 => d128!(5.423910666131588774984495014212839E+295),
    166 => d128!(9.003691705778437366474261723593313E+297),
    167 => d128!(1.503616514864999040201201707840083E+300),
    168 => d128!(2.526075744973198387538018869171339E+302),
    169 => d128!(4.269068009004705274939251888899563E+304),
    170 => d128!(7.257415615307998967396728211129257E+306),
    171 => d128!(1.241018070217667823424840524103103E+309),
    172 => d128!(2.134551080774388656290725701457337E+311),
    173 => d128!(3.692773369739692375382955463521193E+313),
    174 => d128!(6.425425663347064733166342506526876E+315),
    175 => d128!(1.124449491085736328304109938642203E+318),
    176 => d128!(1.979031104310895937815233492010277E+320),
    177 => d128!(3.502885054630285809932963280858190E+322),
    178 => d128!(6.235135397241908741680674639927578E+324),
    179 => d128!(1.116089236106301664760840760547036E+327),
    180 => d128!(2.008960624991342996569513368984665E+329),
    181 => d128!(3.636218731234330823790819197862244E+331),
    182 => d128!(6.617918090846482099299290940109284E+333),
    183 => d128!(1.211079010624906224171770242039999E+336),
    184 => d128!(2.228385379549827452476057245353598E+338),
    185 => d128!(4.122512952167180787080705903904156E+340),
    186 => d128!(7.667874091030956263970112981261730E+342),
    187 => d128!(1.433892455022788821362411127495944E+345),
    188 => d128!(2.695717815442842984161332919692375E+347),
    189 => d128!(5.094906671186973240064919218218589E+349),
    190 => d128!(9.680322675255249156123346514615319E+351),
    191 => d128!(1.848941630973752588819559184291526E+354),
    192 => d128!(3.549967931469604970533553633839730E+356),
    193 => d128!(6.851438107736337593129758513310679E+358),
    194 => d128!(1.329178992900849493067173151582272E+361),
    195 => d128!(2.591899036156656511480987645585430E+363),
    196 => d128!(5.080122110867046762502735785347443E+365),
    197 => d128!(1.000784055840808212213038949713446E+368),
    198 => d128!(1.981552430564800260181817120432623E+370),
    199 => d128!(3.943289336823952517761816069660920E+372),
    200 => d128!(7.886578673647905035523632139321840E+374),
    201 => d128!(1.585202313403228912140250060003690E+377),
    202 => d128!(3.202108673074522402523305121207454E+379),
    203 => d128!(6.500280606341280477122309396051132E+381),
    204 => d128!(1.326057243693621217332951116794431E+384),
    205 => d128!(2.718417349571923495532549789428584E+386),
    206 => d128!(5.599939740118162400797052566222883E+388),
    207 => d128!(1.159187526204459616964989881208137E+391),
    208 => d128!(2.411110054505276003287178952912925E+393),
    209 => d128!(5.039220013916026846870204011588013E+395),
    210 => d128!(1.058236202922365637842742842433483E+398),
    211 => d128!(2.232878388166191495848187397534649E+400),
    212 => d128!(4.733702182912325971198157282773456E+402),
    213 => d128!(1.008278564960325431865207501230746E+405),
    214 => d128!(2.157716129015096424191544052633796E+407),
    215 => d128!(4.639089677382457312011819713162661E+409),
    216 => d128!(1.002043370314610779394553058043135E+412),
    217 => d128!(2.174434113582705391286180135953603E+414),
    218 => d128!(4.740266367610297753003872696378855E+416),
    219 => d128!(1.038118334506655207907848120506969E+419),
    220 => d128!(2.283860335914641457397265865115332E+421),
    221 => d128!(5.047331342371357620847957561904884E+423),
    222 => d128!(1.120507558006441391828246578742884E+426),
    223 => d128!(2.498731854354364303776989870596631E+428),
    224 => d128!(5.597159353753776040460457310136453E+430),
    225 => d128!(1.259360854594599609103602894780702E+433),
    226 => d128!(2.846155531383795116574142542204387E+435),
    227 => d128!(6.460773056241214914623303570803958E+437),
    228 => d128!(1.473056256822997000534113214143302E+440),
    229 => d128!(3.373298828124663131223119260388162E+442),
    230 => d128!(7.758587304686725201813174298892773E+444),
    231 => d128!(1.792233667382633521618843263044231E+447),
    232 => d128!(4.157982108327709770155716370262616E+449),
    233 => d128!(9.688098312403563764462819142711895E+451),
    234 => d128!(2.267015005102433920884299679394583E+454),
    235 => d128!(5.327485261990719714078104246577270E+456),
    236 => d128!(1.257286521829809852522432602192236E+459),
    237 => d128!(2.979769056736649350478165267195599E+461),
    238 => d128!(7.091850355033225454138033335925526E+463),
    239 => d128!(1.694952234852940883538989967286201E+466),
    240 => d128!(4.067885363647058120493575921486882E+468),
    241 => d128!(9.803603726389410070389517970783386E+470),
    242 => d128!(2.372472101786237237034263348929579E+473),
    243 => d128!(5.765107207340556485993259937898877E+475),
    244 => d128!(1.406686158591095782582355424847326E+478),
    245 => d128!(3.446381088548184667326770790875949E+480),
    246 => d128!(8.478097477828534281623856145554835E+482),
    247 => d128!(2.094090077023647967561092467952044E+485),
    248 => d128!(5.193343391018646959551509320521069E+487),
    249 => d128!(1.293142504363643092928325820809746E+490),
    250 => d128!(3.232856260909107732320814552024365E+492),
    251 => d128!(8.114469214881860408125244525581156E+494),
    252 => d128!(2.044846242150228822847561620446451E+497),
    253 => d128!(5.173460992640078921804330899729521E+499),
    254 => d128!(1.314059092130580046138300048531298E+502),
    255 => d128!(3.350850684932979117652665123754810E+504),
    256 => d128!(8.578177753428426541190822716812314E+506),
    257 => d128!(2.204591682631105621086041438220765E+509),
    258 => d128!(5.687846541188252502401986910609574E+511),
    259 => d128!(1.473152254167757398122114609847880E+514),
    260 => d128!(3.830195860836169235117497985604488E+516),
    261 => d128!(9.996811196782401703656669742427714E+518),
    262 => d128!(2.619164533556989246358047472516061E+521),
    263 => d128!(6.888402723254881717921664852717240E+523),
    264 => d128!(1.818538318939288773531319521117351E+526),
    265 => d128!(4.819126545189115249857996730960980E+528),
    266 => d128!(1.281887661020304656462227130435621E+531),
    267 => d128!(3.422640054924213432754146438263108E+533),
    268 => d128!(9.172675347196891999781112454545129E+535),
    269 => d128!(2.467449668395963947941119250272640E+538),
    270 => d128!(6.662114104669102659441021975736128E+540),
    271 => d128!(1.805432922365326820708516955424491E+543),
    272 => d128!(4.910777548833688952327166118754616E+545),
    273 => d128!(1.340642270831597083985316350420010E+548),
    274 => d128!(3.673359822078576010119766800150827E+550),
    275 => d128!(1.010173951071608402782935870041477E+553),
    276 => d128!(2.788080104957639191680903001314477E+555),
    277 => d128!(7.722981890732660560956101313641101E+557),
    278 => d128!(2.146988965623679635945796165192226E+560),
    279 => d128!(5.990099214090066184288771300886311E+562),
    280 => d128!(1.677227779945218531600855964248167E+565),
    281 => d128!(4.713010061646064073798405259537349E+567),
    282 => d128!(1.329068837384190068811150283189532E+570),
    283 => d128!(3.761264809797257894735555301426376E+572),
    284 => d128!(1.068199205982421242104897705605091E+575),
    285 => d128!(3.044367737049900539998958460974509E+577),
    286 => d128!(8.706891727962715544397021198387096E+579),
    287 => d128!(2.498877925925299361241945083937097E+582),
    288 => d128!(7.196768426664862160376801841738839E+584),
    289 => d128!(2.079866075306145164348895732262524E+587),
    290 => d128!(6.031611618387820976611797623561320E+589),
    291 => d128!(1.755198980950855904194033108456344E+592),
    292 => d128!(5.125181024376499240246576676692524E+594),
    293 => d128!(1.501678040142314277392246966270910E+597),
    294 => d128!(4.414933438018403975533206080836475E+599),
    295 => d128!(1.302405364215429172782295793846760E+602),
    296 => d128!(3.855119878077670351435595549786410E+604),
    297 => d128!(1.144970603789068094376371878286564E+607),
    298 => d128!(3.412012399291422921241588197293961E+609),
    299 => d128!(1.020191707388135453451234870990894E+612),
    300 => d128!(3.060575122164406360353704612972682E+614),
    301 => d128!(9.212331117714863144664650885047773E+616),
    302 => d128!(2.782123997549888669688724567284427E+619),
    303 => d128!(8.429835712576162669156835438871814E+621),
    304 => d128!(2.562670056623153451423677973417031E+624),
    305 => d128!(7.816143672700618026842217818921945E+626),
    306 => d128!(2.391739963846389116213718652590115E+629),
    307 => d128!(7.342641689008414586776116263451653E+631),
    308 => d128!(2.261533640214591692727043809143109E+634),
    309 => d128!(6.988138948263088330526565370252207E+636),
    310 => d128!(2.166323073961557382463235264778184E+639),
    311 => d128!(6.737264760020443459460661673460152E+641),
    312 => d128!(2.102026605126378359351726442119567E+644),
    313 => d128!(6.579343274045564264770903763834245E+646),
    314 => d128!(2.065913788050307179138063781843953E+649),
    315 => d128!(6.507628432358467614284900912808452E+651),
    316 => d128!(2.056410584625275766114028688447471E+654),
    317 => d128!(6.518821553262124178581470942378483E+656),
    318 => d128!(2.072985253937355488788907759676358E+659),
    319 => d128!(6.612822960060164009236615753367582E+661),
    320 => d128!(2.116103347219252482955717041077626E+664),
    321 => d128!(6.792691744573800470287851701859179E+666),
    322 => d128!(2.187246741752763751432688247998656E+669),
    323 => d128!(7.064806975861426917127583041035659E+671),
    324 => d128!(2.288997460179102321149336905295554E+674),
    325 => d128!(7.439241745582082543735344942210550E+676),
    326 => d128!(2.425192809059758909257722451160639E+679),
    327 => d128!(7.930380485625411633272752415295290E+681),
    328 => d128!(2.601164799285135015713462792216855E+684),
    329 => d128!(8.557832189648094201697292586393453E+686),
    330 => d128!(2.824084622583871086560106553509839E+689),
    331 => d128!(9.347720100752613296513952692117567E+691),
    332 => d128!(3.103443073449867614442632293783032E+694),
    333 => d128!(1.033446543458805915609396553829750E+697),
    334 => d128!(3.451711455152411758135384489791365E+699),
    335 => d128!(1.156323337476057938975353804080107E+702),
    336 => d128!(3.885246413919554674957188781709160E+704),
    337 => d128!(1.309328041490889925460572619435987E+707),
    338 => d128!(4.425528780239207948056735453693636E+709),
    339 => d128!(1.500254256501091494391233318802143E+712),
    340 => d128!(5.100864472103711080930193283927286E+714),
    341 => d128!(1.739394784987365478597195909819205E+717),
    342 => d128!(5.948730164656789936802410011581681E+719),
    343 => d128!(2.040414446477278948323226633972517E+722),
    344 => d128!(7.019025695881839582231899620865458E+724),
    345 => d128!(2.421563865079234655870005369198583E+727),
    346 => d128!(8.378610973174151909310218577427097E+729),
    347 => d128!(2.907378007691430712530645846367203E+732),
    348 => d128!(1.011767546676617887960664754535787E+735),
    349 => d128!(3.531068737901396428982719993329897E+737),
    350 => d128!(1.235874058265488750143951997665464E+740),
    351 => d128!(4.337917944511865513005271511805779E+742),
    352 => d128!(1.526947116468176660577855572155634E+745),
    353 => d128!(5.390123321132663611839830169709388E+747),
    354 => d128!(1.908103655680962918591299880077123E+750),
    355 => d128!(6.773767977667418360999114574273787E+752),
    356 => d128!(2.411461400049600936515684788441468E+755),
    357 => d128!(8.608917198177075343360994694736041E+757),
    358 => d128!(3.081992356947392972923236100715503E+760),
    359 => d128!(1.106435256144114077279441760156866E+763),
    360 => d128!(3.983166922118810678205990336564718E+765),
    361 => d128!(1.437923258884890654832362511499863E+768),
    362 => d128!(5.205282197163304170493152291629504E+770),
    363 => d128!(1.889517437570279413889014281861510E+773),
    364 => d128!(6.877843472755817066556011985975896E+775),
    365 => d128!(2.510412867555873229292944374881202E+778),
    366 => d128!(9.188111095254496019212176412065199E+780),
    367 => d128!(3.372036771958400039050868743227928E+783),
    368 => d128!(1.240909532080691214370719697507878E+786),
    369 => d128!(4.578956173377750581027955683804070E+788),
    370 => d128!(1.694213784149767714980343603007506E+791),
    371 => d128!(6.285533139195638222577074767157847E+793),
    372 => d128!(2.338218327780777418798671813382719E+796),
    373 => d128!(8.721554362622299772119045863917542E+798),
    374 => d128!(3.261861331620740114772523153105161E+801),
    375 => d128!(1.223197999357777543039696182414435E+804),
    376 => d128!(4.599224477585243561829257645878276E+806),
    377 => d128!(1.733907628049636822809630132496110E+809),
    378 => d128!(6.554170834027627190220401900835296E+811),
    379 => d128!(2.484030746096470705093532320416577E+814),
    380 => d128!(9.439316835166588679355422817582993E+816),
    381 => d128!(3.596379714198470286834416093499120E+819),
    382 => d128!(1.373817050823815649570746947716664E+822),
    383 => d128!(5.261719304655213937855960809754823E+824),
    384 => d128!(2.020500212987602152136688950945852E+827),
    385 => d128!(7.778925820002268285726252461141530E+829),
    386 => d128!(3.002665366520875558290333450000631E+832),
    387 => d128!(1.162031496843578841058359045150244E+835),
    388 => d128!(4.508682207753085903306433095182947E+837),
    389 => d128!(1.753877378815950416386202474026166E+840),
    390 => d128!(6.840121777382206623906189648702047E+842),
    391 => d128!(2.674487614956442789947320152642500E+845),
    392 => d128!(1.048399145062925573659349499835860E+848),
    393 => d128!(4.120208640097297504481243534354930E+850),
    394 => d128!(1.623362204198335216765609952535842E+853),
    395 => d128!(6.412280706583424106224159312516576E+855),
    396 => d128!(2.539263159807035946064767087756564E+858),
    397 => d128!(1.008087474443393270587712533839356E+861),
    398 => d128!(4.012188148284705216939095884680637E+863),
    399 => d128!(1.600863071165597381558699257987574E+866),
    400 => d128!(6.403452284662389526234797031950296E+868),
    401 => d128!(2.567784366149618200020153609812069E+871),
    402 => d128!(1.032249315192146516408101751144452E+874),
    403 => d128!(4.159964740224350461124650057112142E+876),
    404 => d128!(1.680625755050637586294358623073305E+879),
    405 => d128!(6.806534307955082224492152423446885E+881),
    406 => d128!(2.763452929029763383143813883919435E+884),
    407 => d128!(1.124725342115113696939532250755210E+887),
    408 => d128!(4.588879395829663883513291583081257E+889),
    409 => d128!(1.876851672894332528356936257480234E+892),
    410 => d128!(7.695091858866763366263438655668959E+894),
    411 => d128!(3.162682753994239743534273287479942E+897),
    412 => d128!(1.303025294645626774336120594441736E+900),
    413 => d128!(5.381494466886438578008178055044370E+902),
    414 => d128!(2.227938709290985571295385714788369E+905),
    415 => d128!(9.245945643557590120875850716371731E+907),
    416 => d128!(3.846313387719957490284353898010640E+910),
    417 => d128!(1.603912682679222273448575575470437E+913),
    418 => d128!(6.704355013599149103015045905466427E+915),
    419 => d128!(2.809124750698043474163304234390433E+918),
    420 => d128!(1.179832395293178259148587778443982E+921),
    421 => d128!(4.967094384184280471015554547249164E+923),
    422 => d128!(2.096113830125766358768564018939147E+926),
    423 => d128!(8.866561501431991697591025800112592E+928),
    424 => d128!(3.759422076607164479778594939247739E+931),
    425 => d128!(1.597754382558044903905902849180289E+934),
    426 => d128!(6.806433669697271290639146137508031E+936),
    427 => d128!(2.906347176960734841102915400715929E+939),
    428 => d128!(1.243916591739194511992047791506418E+942),
    429 => d128!(5.336402178561144456445885025562533E+944),
    430 => d128!(2.294652936781292116271730560991889E+947),
    431 => d128!(9.889954157527369021131158717875042E+949),
    432 => d128!(4.272460196051823417128660566122018E+952),
    433 => d128!(1.849975264890439539616710025130834E+955),
    434 => d128!(8.028892649624507601936521509067820E+957),
    435 => d128!(3.492568302586660806842386856444502E+960),
    436 => d128!(1.522759779927784111783280669409803E+963),
    437 => d128!(6.654460238284416568492936525320839E+965),
    438 => d128!(2.914653584368574456999906198090527E+968),
    439 => d128!(1.279532923537804186622958820961741E+971),
    440 => d128!(5.629944863566338421141018812231660E+973),
    441 => d128!(2.482805684832755243723189296194162E+976),
    442 => d128!(1.097400112696077817725649668917820E+979),
    443 => d128!(4.861482499243624732524628033305943E+981),
    444 => d128!(2.158498229664169381240934846787839E+984),
    445 => d128!(9.605317122005553746522160068205884E+986),
    446 => d128!(4.283971436414476970948883390419824E+989),
    447 => d128!(1.914935232077271206014150875517661E+992),
    448 => d128!(8.578909839706175002943395922319121E+994),
    449 => d128!(3.851930518028072576321584769121285E+997),
    450 => d128!(1.733368733112632659344713146104578E+1000),
    451 => d128!(7.817492986337973293644656288931647E+1002),
    452 => d128!(3.533506829824763928727384642597104E+1005),
    453 => d128!(1.600678593910618059713505243096488E+1008),
    454 => d128!(7.267080816354205991099313803658056E+1010),
    455 => d128!(3.306521771441163725950187780664415E+1013),
    456 => d128!(1.507773927777170659033285627982973E+1016),
    457 => d128!(6.890526849941669911782115319882187E+1018),
    458 => d128!(3.155861297273284819596208816506042E+1021),
    459 => d128!(1.448540335448437732194659846776273E+1024),
    460 => d128!(6.663285543062813568095435295170856E+1026),
    461 => d128!(3.071774635351957054891995671073765E+1029),
    462 => d128!(1.419159881532604159360102000036079E+1032),
    463 => d128!(6.570710251495957257837272260167046E+1034),
    464 => d128!(3.048809556694124167636494328717509E+1037),
    465 => d128!(1.417696443862767737950969862853642E+1040),
    466 => d128!(6.606465428400497658851519560897972E+1042),
    467 => d128!(3.085219355063032406683659634939353E+1045),
    468 => d128!(1.443882658169499166327952709151617E+1048),
    469 => d128!(6.771809666814951090078098205921084E+1050),
    470 => d128!(3.182750543403027012336706156782909E+1053),
    471 => d128!(1.499075505942825722810588599844750E+1056),
    472 => d128!(7.075636388050137411665978191267220E+1058),
    473 => d128!(3.346776011547714995718007684469395E+1061),
    474 => d128!(1.586371829473616907970335642438493E+1064),
    475 => d128!(7.535266189999680312859094301582842E+1066),
    476 => d128!(3.586786706439847828920928887553433E+1069),
    477 => d128!(1.710897258971807414395283079362988E+1072),
    478 => d128!(8.178088897885239440809453119355083E+1074),
    479 => d128!(3.917304582087029692147728044171085E+1077),
    480 => d128!(1.880306199401774252230909461202121E+1080),
    481 => d128!(9.044272819122534153230674508382202E+1082),
    482 => d128!(4.359339498817061461857185113040221E+1085),
    483 => d128!(2.105560977928640686077020409598427E+1088),
    484 => d128!(1.019091513317462092061277878245639E+1091),
    485 => d128!(4.942593839589691146497197709491349E+1093),
    486 => d128!(2.402100606040589897197638086812796E+1096),
    487 => d128!(1.169822995141767279935249748277832E+1099),
    488 => d128!(5.708736216291824326084018771595820E+1101),
    489 => d128!(2.791572009766702095455085179310356E+1104),
    490 => d128!(1.367870284785684026772991737862074E+1107),
    491 => d128!(6.716243098297708571455389432902783E+1109),
    492 => d128!(3.304391604362472617156051600988169E+1112),
    493 => d128!(1.629065060950699000257933439287167E+1115),
    494 => d128!(8.047581401096453061274191190078605E+1117),
    495 => d128!(3.983552793542744265330724639088909E+1120),
    496 => d128!(1.975842185597201155604039420988099E+1123),
    497 => d128!(9.819935662418089743352075922310852E+1125),
    498 => d128!(4.890327959884208692189333809310804E+1128),
    499 => d128!(2.440273651982220137402477570846091E+1131),
    500 => d128!(1.220136825991110068701238785423046E+1134),
    501 => d128!(6.112885498215461444193206314969460E+1136),
    502 => d128!(3.068668520104161644984989570114669E+1139),
    503 => d128!(1.543540265612393307427449753767679E+1142),
    504 => d128!(7.779442938686462269434346758989102E+1144),
    505 => d128!(3.928618684036663446064345113289497E+1147),
    506 => d128!(1.987881054122551703708558627324485E+1150),
    507 => d128!(1.007855694440133713780239224053514E+1153),
    508 => d128!(5.119906927755879266003615258191851E+1155),
    509 => d128!(2.606032626227742546395840166419652E+1158),
    510 => d128!(1.329076639376148698661878484874023E+1161),
    511 => d128!(6.791581627212119850162199057706258E+1163),
    512 => d128!(3.477289793132605363283045917545604E+1166),
    513 => d128!(1.783849663877026551364202555700895E+1169),
    514 => d128!(9.168987272327916474012001136302600E+1171),
    515 => d128!(4.722028445248876984116180585195839E+1174),
    516 => d128!(2.436566677748420523803949181961053E+1177),
    517 => d128!(1.259704972395933410806641727073864E+1180),
    518 => d128!(6.525271757010935067978404146242616E+1182),
    519 => d128!(3.386616041888675300280791751899918E+1185),
    520 => d128!(1.761040341782111156146011710987957E+1188),
    521 => d128!(9.175020180684799123520721014247256E+1190),
    522 => d128!(4.789360534317465142477816369437068E+1193),
    523 => d128!(2.504835559448034269515897961215587E+1196),
    524 => d128!(1.312533833150769957226330531676968E+1199),
    525 => d128!(6.890802624041542275438235291304082E+1201),
    526 => d128!(3.624562180245851236880511763225947E+1204),
    527 => d128!(1.910144268989563601836029699220074E+1207),
    528 => d128!(1.008556174026489581769423681188199E+1210),
    529 => d128!(5.335262160600129887560251273485573E+1212),
    530 => d128!(2.827688945118068840406933174947354E+1215),
    531 => d128!(1.501502829857694554256081515897045E+1218),
    532 => d128!(7.987995054842935028642353664572279E+1220),
    533 => d128!(4.257601364231284370266374503217025E+1223),
    534 => d128!(2.273559128499505853722243984717891E+1226),
    535 => d128!(1.216354133747235631741400531824072E+1229),
    536 => d128!(6.519658156885182986133906850577026E+1231),
    537 => d128!(3.501056430247343263553907978759863E+1234),
    538 => d128!(1.883568359473070675792002492572806E+1237),
    539 => d128!(1.015243345755985094251889343496742E+1240),
    540 => d128!(5.482314067082319508960202454882407E+1242),
    541 => d128!(2.965931910291534854347469528091382E+1245),
    542 => d128!(1.607535095378011891056328484225529E+1248),
    543 => d128!(8.728915567902604568435863669344622E+1250),
    544 => d128!(4.748530068939016885229109836123474E+1253),
    545 => d128!(2.587948887571764202449864860687293E+1256),
    546 => d128!(1.413020092614183254537626213935262E+1259),
    547 => d128!(7.729219906599582402320815390225883E+1261),
    548 => d128!(4.235612508816571156471806833843784E+1264),
    549 => d128!(2.325351267340297564903021951780237E+1267),
    550 => d128!(1.278943197037163660696662073479130E+1270),
    551 => d128!(7.046977015674771770438608024870006E+1272),
    552 => d128!(3.889931312652474017282111629728243E+1275),
    553 => d128!(2.151132015896818131557007731239718E+1278),
    554 => d128!(1.191727136806837244882582283106804E+1281),
    555 => d128!(6.614085609277946709098331671242762E+1283),
    556 => d128!(3.677431598758538370258672409210976E+1286),
    557 => d128!(2.048329400508505872234080531930514E+1289),
    558 => d128!(1.142967805483746276706616936817227E+1292),
    559 => d128!(6.389190032654141686789988676808299E+1294),
    560 => d128!(3.577946418286319344602393659012647E+1297),
    561 => d128!(2.007227940658625152321942842706095E+1300),
    562 => d128!(1.128062102650147335604931877600825E+1303),
    563 => d128!(6.350989637920329499455766470892645E+1305),
    564 => d128!(3.581958155787065837693052289583452E+1308),
    565 => d128!(2.023806358019692198296574543614650E+1311),
    566 => d128!(1.145474398639145784235861191685892E+1314),
    567 => d128!(6.494839840283956596617332956859008E+1316),
    568 => d128!(3.689069029281287346878645119495917E+1319),
    569 => d128!(2.099080277661052500373949072993177E+1322),
    570 => d128!(1.196475758266799925213150971606111E+1325),
    571 => d128!(6.831876579703427572967092047870894E+1327),
    572 => d128!(3.907833403590360571737176651382151E+1330),
    573 => d128!(2.239188540257276607605402221241973E+1333),
    574 => d128!(1.285294222107676772765500874992893E+1336),
    575 => d128!(7.390441777119141443401630031209135E+1338),
    576 => d128!(4.256894463620625471399338897976462E+1341),
    577 => d128!(2.456228105509100896997418544132419E+1344),
    578 => d128!(1.419699844984260318464507918508538E+1347),
    579 => d128!(8.220062102458867243909500848164435E+1349),
    580 => d128!(4.767636019426143001467510491935372E+1352),
    581 => d128!(2.769996527286589083852623595814451E+1355),
    582 => d128!(1.612137978880794846802226932764010E+1358),
    583 => d128!(9.398764416875033956856983018014178E+1360),
    584 => d128!(5.488878419455019830804478082520280E+1363),
    585 => d128!(3.210993875381186601020619678274364E+1366),
    586 => d128!(1.881642410973375348198083131468777E+1369),
    587 => d128!(1.104524095241371329392274798172172E+1372),
    588 => d128!(6.494601680019263416826575813252371E+1374),
    589 => d128!(3.825320389531346152510853154005647E+1377),
    590 => d128!(2.256939029823494229981403360863332E+1380),
    591 => d128!(1.333850966625685089919009386270229E+1383),
    592 => d128!(7.896397722424055732320535566719756E+1385),
    593 => d128!(4.682563849397465049266077591064815E+1388),
    594 => d128!(2.781442926542094239264050089092500E+1391),
    595 => d128!(1.654958541292546072362109803010038E+1394),
    596 => d128!(9.863552906103574591278174425939826E+1396),
    597 => d128!(5.888541084943834030993070132286076E+1399),
    598 => d128!(3.521347568796412750533855939107073E+1402),
    599 => d128!(2.109287193709051237569779707525137E+1405),
    600 => d128!(1.265572316225430742541867824515082E+1408),
    601 => d128!(7.606089620514838762676625625335643E+1410),
    602 => d128!(4.578865951549932935131328626452057E+1413),
    603 => d128!(2.761056168784609559884191161750590E+1416),
    604 => d128!(1.667677925945904174170051461697356E+1419),
    605 => d128!(1.008945145197272025372881134326900E+1422),
    606 => d128!(6.114207579895468473759659674021014E+1424),
    607 => d128!(3.711324000996549363572113422130755E+1427),
    608 => d128!(2.256484992605902013051844960655499E+1430),
    609 => d128!(1.374199360496994325948573581039199E+1433),
    610 => d128!(8.382616099031665388286298844339114E+1435),
    611 => d128!(5.121778436508347552242928593891199E+1438),
    612 => d128!(3.134528403143108701972672299461414E+1441),
    613 => d128!(1.921465911126725634309248119569847E+1444),
    614 => d128!(1.179780069431809539465878345415886E+1447),
    615 => d128!(7.255647427005628667715151824307699E+1449),
    616 => d128!(4.469478815035467259312533523773543E+1452),
    617 => d128!(2.757668428876883298995833184168276E+1455),
    618 => d128!(1.704239089045913878779424907815995E+1458),
    619 => d128!(1.054923996119420690964464017938101E+1461),
    620 => d128!(6.540528775940408283979676911216226E+1463),
    621 => d128!(4.061668369858993544351379361865276E+1466),
    622 => d128!(2.526357726052293984586557963080202E+1469),
    623 => d128!(1.573920863330579152397425610998966E+1472),
    624 => d128!(9.821266187182813910959935812633548E+1474),
    625 => d128!(6.138291366989258694349959882895968E+1477),
    626 => d128!(3.842570395735275942663074886692876E+1480),
    627 => d128!(2.409291638126018016049747953956433E+1483),
    628 => d128!(1.513035148743139314079241715084640E+1486),
    629 => d128!(9.516991085594346285558430387882386E+1488),
    630 => d128!(5.995704383924438159901811144365903E+1491),
    631 => d128!(3.783289466256320478898042832094885E+1494),
    632 => d128!(2.391038942673994542663563069883967E+1497),
    633 => d128!(1.513527650712638545506035423236551E+1500),
    634 => d128!(9.595765305518128378508264583319733E+1502),
    635 => d128!(6.093310969004011520352748010408030E+1505),
    636 => d128!(3.875345776286551326944347734619507E+1508),
    637 => d128!(2.468595259494533195263549506952626E+1511),
    638 => d128!(1.574963775557512178578144585435775E+1514),
    639 => d128!(1.006401852581250282111434390093460E+1517),
    640 => d128!(6.440971856520001805513180096598144E+1519),
    641 => d128!(4.128662960029321157333948441919410E+1522),
    642 => d128!(2.650601620338824183008394899712261E+1525),
    643 => d128!(1.704336841877863949674397920514984E+1528),
    644 => d128!(1.097592926169344383590312260811650E+1531),
    645 => d128!(7.079474373792271274157514082235142E+1533),
    646 => d128!(4.573340445469807243105754097123902E+1536),
    647 => d128!(2.958951268218965286289422900839165E+1539),
    648 => d128!(1.917400421805889505515546039743779E+1542),
    649 => d128!(1.244392873752022289079589379793713E+1545),
    650 => d128!(8.088553679388144879017330968659134E+1547),
    651 => d128!(5.265648445281682316240282460597096E+1550),
    652 => d128!(3.433202786323656870188664164309307E+1553),
    653 => d128!(2.241881419469347936233197699293977E+1556),
    654 => d128!(1.466190448332953550296511295338261E+1559),
    655 => d128!(9.603547436580845754442148984465610E+1561),
    656 => d128!(6.299927118397034814914049733809440E+1564),
    657 => d128!(4.139052116786851873398530675112802E+1567),
    658 => d128!(2.723496292845748532696233184224224E+1570),
    659 => d128!(1.794784056985348283046817668403764E+1573),
    660 => d128!(1.184557477610329866810899661146484E+1576),
    661 => d128!(7.829924927004280419620046760178259E+1578),
    662 => d128!(5.183410301676833637788470955238007E+1581),
    663 => d128!(3.436601030011740701853756243322799E+1584),
    664 => d128!(2.281903083927795826030894145566339E+1587),
    665 => d128!(1.517465550811984224310544606801615E+1590),
    666 => d128!(1.010632056840781493390822708129876E+1593),
    667 => d128!(6.740915819128012560916787463226273E+1595),
    668 => d128!(4.502931767177512390692414025435150E+1598),
    669 => d128!(3.012461352241755789373224983016115E+1601),
    670 => d128!(2.018349106001976378880060738620797E+1604),
    671 => d128!(1.354312250127326150228520755614555E+1607),
    672 => d128!(9.100978320855631729535659477729810E+1609),
    673 => d128!(6.124958409935840153977498828512162E+1612),
    674 => d128!(4.128221968296756263780834210417197E+1615),
    675 => d128!(2.786549828600310478052063092031608E+1618),
    676 => d128!(1.883707684133809883163194650213367E+1621),
    677 => d128!(1.275270102158589290901482778194449E+1624),
    678 => d128!(8.646331292635235392312053236158364E+1626),
    679 => d128!(5.870858947699324831379884147351529E+1629),
    680 => d128!(3.992184084435540885338321220199040E+1632),
    681 => d128!(2.718677361500603342915396750955546E+1635),
    682 => d128!(1.854137960543411479868300584151682E+1638),
    683 => d128!(1.266376227051150040750049298975599E+1641),
    684 => d128!(8.662013393029866278730337204993097E+1643),
    685 => d128!(5.933479174225458400930280985420271E+1646),
    686 => d128!(4.070366713518664463038172755998306E+1649),
    687 => d128!(2.796341932187322486107224683370836E+1652),
    688 => d128!(1.923883249344877870441770582159135E+1655),
    689 => d128!(1.325555558798620852734379931107644E+1658),
    690 => d128!(9.146333355710483883867221524642744E+1660),
    691 => d128!(6.320116348795944363752250073528136E+1663),
    692 => d128!(4.373520513366793499716557050881470E+1666),
    693 => d128!(3.030849715763187895303574036260859E+1669),
    694 => d128!(2.103409702739652399340680381165036E+1672),
    695 => d128!(1.461869743404058417541772864909700E+1675),
    696 => d128!(1.017461341409224658609073913977151E+1678),
    697 => d128!(7.091705549622295870505245180420742E+1680),
    698 => d128!(4.950010473636362517612661135933678E+1683),
    699 => d128!(3.460057321071817399811250134017641E+1686),
    700 => d128!(2.422040124750272179867875093812349E+1689),
    701 => d128!(1.697850127449940798087380440762457E+1692),
    702 => d128!(1.191890789469858440257341069415245E+1695),
    703 => d128!(8.378992249973104835009107717989172E+1697),
    704 => d128!(5.898810543981065803846411833464377E+1700),
    705 => d128!(4.158661433506651391711720342592386E+1703),
    706 => d128!(2.936014972055695882548474561870225E+1706),
    707 => d128!(2.075762585243376988961771515242249E+1709),
    708 => d128!(1.469639910352310908184934232791512E+1712),
    709 => d128!(1.041974696439788433903118371049182E+1715),
    710 => d128!(7.398020344722497880712140434449192E+1717),
    711 => d128!(5.259992465097695993186331848893376E+1720),
    712 => d128!(3.745114635149559547148668276412084E+1723),
    713 => d128!(2.670266734861635957117000481081816E+1726),
    714 => d128!(1.906570448691208073381538343492417E+1729),
    715 => d128!(1.363197870814213772467799915597078E+1732),
    716 => d128!(9.760496755029770610869447395675078E+1734),
    717 => d128!(6.998276173356345527993393782699031E+1737),
    718 => d128!(5.024762292469856089099256735977904E+1740),
    719 => d128!(3.612804088285826528062365593168113E+1743),
    720 => d128!(2.601218943565795100204903227081041E+1746),
    721 => d128!(1.875478858310938267247735226725431E+1749),
    722 => d128!(1.354095735700497428952864833695761E+1752),
    723 => d128!(9.790112169114596411329212747620352E+1754),
    724 => d128!(7.088041210438967801802350029277135E+1757),
    725 => d128!(5.138829877568251656306703771225923E+1760),
    726 => d128!(3.730790491114550702478666937910020E+1763),
    727 => d128!(2.712284687040278360701990863860585E+1766),
    728 => d128!(1.974543252165322646591049348890506E+1769),
    729 => d128!(1.439442030828520209364874975341179E+1772),
    730 => d128!(1.050792682504819752836358731999061E+1775),
    731 => d128!(7.681294509110232393233782330913136E+1777),
    732 => d128!(5.622707580668690111847128666228416E+1780),
    733 => d128!(4.121444656630149851983945312345429E+1783),
    734 => d128!(3.025140377966529991356215859261545E+1786),
    735 => d128!(2.223478177805399543646818656557236E+1789),
    736 => d128!(1.636479938864774064124058531226126E+1792),
    737 => d128!(1.206085714943338485259431137513655E+1795),
    738 => d128!(8.900912576281838021214601794850774E+1797),
    739 => d128!(6.577774393872278297677590726394722E+1800),
    740 => d128!(4.867553051465485940281417137532094E+1803),
    741 => d128!(3.606856811135925081748530098911282E+1806),
    742 => d128!(2.676287753862856410657409333392171E+1809),
    743 => d128!(1.988481801120102313118455134710383E+1812),
    744 => d128!(1.479430460033356120960130620224525E+1815),
    745 => d128!(1.102175692724850310115297312067271E+1818),
    746 => d128!(8.222230667727383313460117948021842E+1820),
    747 => d128!(6.142006308792355335154708107172316E+1823),
    748 => d128!(4.594220718976681790695721664164892E+1826),
    749 => d128!(3.441071318513534661231095526459504E+1829),
    750 => d128!(2.580803488885150995923321644844628E+1832),
    751 => d128!(1.938183420152748397938414555278316E+1835),
    752 => d128!(1.457513931954866795249687745569294E+1838),
    753 => d128!(1.097507990762014696823014872413678E+1841),
    754 => d128!(8.275210250345590814045532137999132E+1843),
    755 => d128!(6.247783739010921064604376764189345E+1846),
    756 => d128!(4.723324506692256324840908833727145E+1849),
    757 => d128!(3.575556651566038037904567987131449E+1852),
    758 => d128!(2.710271941887056832731662534245638E+1855),
    759 => d128!(2.057096403892276136043331863492439E+1858),
    760 => d128!(1.563393266958129863392932216254254E+1861),
    761 => d128!(1.189742276155136826042021416569487E+1864),
    762 => d128!(9.065836144302142614440203194259491E+1866),
    763 => d128!(6.917232978102534814817875037219992E+1869),
    764 => d128!(5.284765995270336598520856528436074E+1872),
    765 => d128!(4.042845986381807497868455244253597E+1875),
    766 => d128!(3.096820025568464543367236717098255E+1878),
    767 => d128!(2.375260959611012304762670562014362E+1881),
    768 => d128!(1.824200416981257450057730991627030E+1884),
    769 => d128!(1.402810120658586979094395132561186E+1887),
    770 => d128!(1.080163792907111973902684252072113E+1890),
    771 => d128!(8.328062843313833318789695583475991E+1892),
    772 => d128!(6.429264515038279322105644990443465E+1895),
    773 => d128!(4.969821470124589915987663577612798E+1898),
    774 => d128!(3.846641817876432594974451609072306E+1901),
    775 => d128!(2.981147408854235261105199997031037E+1904),
    776 => d128!(2.313370389270886562617635197696085E+1907),
    777 => d128!(1.797488792463478859153902548609858E+1910),
    778 => d128!(1.398446280536586552421736182818470E+1913),
    779 => d128!(1.089389652538000924336532486415588E+1916),
    780 => d128!(8.497239289796407209824953394041586E+1918),
    781 => d128!(6.636343885330994030873288600746479E+1921),
    782 => d128!(5.189620918328837332142911685783747E+1924),
    783 => d128!(4.063473179051479631067899849968674E+1927),
    784 => d128!(3.185762972376360030757233482375440E+1930),
    785 => d128!(2.500823933315442624144428283664720E+1933),
    786 => d128!(1.965647611585937902577520630960470E+1936),
    787 => d128!(1.546964670318133129328508736565890E+1939),
    788 => d128!(1.219008160210688905910864884413921E+1942),
    789 => d128!(9.617974384062335467636723938025837E+1944),
    790 => d128!(7.598199763409245019433011911040411E+1947),
    791 => d128!(6.010176012856712810371512421632965E+1950),
    792 => d128!(4.760059402182516545814237837933308E+1953),
    793 => d128!(3.774727105930735620830690605481113E+1956),
    794 => d128!(2.997133322109004082939568340752004E+1959),
    795 => d128!(2.382720991076658245936956830897843E+1962),
    796 => d128!(1.896645908897019963765817637394683E+1965),
    797 => d128!(1.511626789390924911121356657003562E+1968),
    798 => d128!(1.206278177933958079074842612288842E+1971),
    799 => d128!(9.638162641692325051807992472187848E+1973),
    800 => d128!(7.710530113353860041446393977750278E+1976),
    801 => d128!(6.176134620796441893198561576177973E+1979),
    802 => d128!(4.953259965878746398345246384094734E+1982),
    803 => d128!(3.977467752600633357871232846428071E+1985),
    804 => d128!(3.197884073090909219728471208528169E+1988),
    805 => d128!(2.574296678838181921881419322865176E+1991),
    806 => d128!(2.074883123143574629036423974229332E+1994),
    807 => d128!(1.674430680376864725632394147203071E+1997),
    808 => d128!(1.352939989744506698310974470940081E+2000),
    809 => d128!(1.094528451703305918933578346990526E+2003),
    810 => d128!(8.865680458796777943361984610623261E+2005),
    811 => d128!(7.190066852084186912066569519215465E+2008),
    812 => d128!(5.838334283892359772598054449602958E+2011),
    813 => d128!(4.746565772804488495122218267527205E+2014),
    814 => d128!(3.863704539062853635029485669767145E+2017),
    815 => d128!(3.148919199336225712549030820860223E+2020),
    816 => d128!(2.569518066658360181440009149821942E+2023),
    817 => d128!(2.099296260459880268236487475404527E+2026),
    818 => d128!(1.717224341056182059417446754880903E+2029),
    819 => d128!(1.406406735325013106662888892247460E+2032),
    820 => d128!(1.153253522966510747463568891642917E+2035),
    821 => d128!(9.468211423555053236675900600388349E+2037),
    822 => d128!(7.782869790162253760547590293519223E+2040),
    823 => d128!(6.405301837303534844930666811566321E+2043),
    824 => d128!(5.277968713938112712222869452730649E+2046),
    825 => d128!(4.354324188998942987583867298502785E+2049),
    826 => d128!(3.596671780113126907744274388563300E+2052),
    827 => d128!(2.974447562153555952704514919341849E+2055),
    828 => d128!(2.462842581463144328839338353215051E+2058),
    829 => d128!(2.041696500032946648607811494815277E+2061),
    830 => d128!(1.694608095027345718344483540696680E+2064),
    831 => d128!(1.408219326967724291944265822318941E+2067),
    832 => d128!(1.171638480037146610897629164169359E+2070),
    833 => d128!(9.759748538709431268777250937530760E+2072),
    834 => d128!(8.139630281283665678160227281900654E+2075),
    835 => d128!(6.796591284871860841263789780387046E+2078),
    836 => d128!(5.681950314152875663296528256403570E+2081),
    837 => d128!(4.755792412945956930179194150609788E+2084),
    838 => d128!(3.985354042048711907490164698211002E+2087),
    839 => d128!(3.343712041278869290384248181799031E+2090),
    840 => d128!(2.808718114674250203922768472711186E+2093),
    841 => d128!(2.362131934441044421499048285550107E+2096),
    842 => d128!(1.988915088799359402902198656433190E+2099),
    843 => d128!(1.676655419857859976646553467373179E+2102),
    844 => d128!(1.415097174360033820289691126462963E+2105),
    845 => d128!(1.195757112334228578144789001861204E+2108),
    846 => d128!(1.011610517034757377110491495574579E+2111),
    847 => d128!(8.568341079284394984125862967516684E+2113),
    848 => d128!(7.265953235233166946538731796454148E+2116),
    849 => d128!(6.168794296712958737611383295189572E+2119),
    850 => d128!(5.243475152206014926969675800911136E+2122),
    851 => d128!(4.462197354527318702851194106575377E+2125),
    852 => d128!(3.801792146057275534829217378802221E+2128),
    853 => d128!(3.242928700586856031209322424118295E+2131),
    854 => d128!(2.769461110301175050652761350197024E+2134),
    855 => d128!(2.367889249307504668308110954418456E+2137),
    856 => d128!(2.026913197407223996071742976982198E+2140),
    857 => d128!(1.737064610177990964633483731273744E+2143),
    858 => d128!(1.490401435532716247655529041432872E+2146),
    859 => d128!(1.280254833122603256736099446590837E+2149),
    860 => d128!(1.101019156485438800793045524068120E+2152),
    861 => d128!(9.479774937339628074828121962226513E+2154),
    862 => d128!(8.171565995986759400501841131439254E+2157),
    863 => d128!(7.052061454536573362633088896432076E+2160),
    864 => d128!(6.092981096719599385314988806517314E+2163),
    865 => d128!(5.270428648662453468297465317637477E+2166),
    866 => d128!(4.564191209741684703545604965074055E+2169),
    867 => d128!(3.957153778846040637974039504719206E+2172),
    868 => d128!(3.434809480038363273761466290096271E+2175),
    869 => d128!(2.984849438153337684898714206093659E+2178),
    870 => d128!(2.596819011193403785861881359301483E+2181),
    871 => d128!(2.261829358749454697485698663951592E+2184),
    872 => d128!(1.972315200829524496207529234965788E+2187),
    873 => d128!(1.721831170324174885189173022125133E+2190),
    874 => d128!(1.504880442863328849655337221337366E+2193),
    875 => d128!(1.316770387505412743448420068670195E+2196),
    876 => d128!(1.153490859454741563260815980155091E+2199),
    877 => d128!(1.011611483741808350979735614596015E+2202),
    878 => d128!(8.881948827253077321602078696153012E+2204),
    879 => d128!(7.807233019155454965688227173918498E+2207),
    880 => d128!(6.870365056856800369805639913048278E+2210),
    881 => d128!(6.052791615090841125798768763395533E+2213),
    882 => d128!(5.338562204510121872954514049314860E+2216),
    883 => d128!(4.713950426582437613818835905545021E+2219),
    884 => d128!(4.167132177098874850615850940501799E+2222),
    885 => d128!(3.687911976732504242795028082344092E+2225),
    886 => d128!(3.267490011384998759116394880956866E+2228),
    887 => d128!(2.898263640098493899336242259408740E+2231),
    888 => d128!(2.573658112407462582610583126354961E+2234),
    889 => d128!(2.287982061930234235940808399329560E+2237),
    890 => d128!(2.036304035117908469987319475403308E+2240),
    891 => d128!(1.814346895290056446758701652584347E+2243),
    892 => d128!(1.618397430598730350508761874105238E+2246),
    893 => d128!(1.445228905524666203004324353575978E+2249),
    894 => d128!(1.292034641539051585485865972096924E+2252),
    895 => d128!(1.156371004177451169009850045026747E+2255),
    896 => d128!(1.036108419742996247432825640343965E+2258),
    897 => d128!(9.293892525094676339472445993885366E+2260),
    898 => d128!(8.345915487535019352846256502509059E+2263),
    899 => d128!(7.502978023293982398208784595755644E+2266),
    900 => d128!(6.752680220964584158387906136180080E+2269),
    901 => d128!(6.084164879089090326707503428698252E+2272),
    902 => d128!(5.487916720938359474690168092685823E+2275),
    903 => d128!(4.955588799007338605645221787695298E+2278),
    904 => d128!(4.479852274302634099503280496076549E+2281),
    905 => d128!(4.054266308243883860050468848949277E+2284),
    906 => d128!(3.673165275268958777205724777148045E+2287),
    907 => d128!(3.331560904668945610925592372873277E+2290),
    908 => d128!(3.025057301439402614720437874568936E+2293),
    909 => d128!(2.749777087008416976780878027983163E+2296),
    910 => d128!(2.502297149177659448870599005464678E+2299),
    911 => d128!(2.279592702900847757921115693978322E+2302),
    912 => d128!(2.078988545045573155224057512908230E+2305),
    913 => d128!(1.898116541626608290719564509285214E+2308),
    914 => d128!(1.734878519046719977717681961486686E+2311),
    915 => d128!(1.587413844927748779611678994760318E+2314),
    916 => d128!(1.454071081953817882124297959200451E+2317),
    917 => d128!(1.333383182151650997907981228586814E+2320),
    918 => d128!(1.224045761215215616079526767842695E+2323),
    919 => d128!(1.124898054556783151177085099647437E+2326),
    920 => d128!(1.034906210192240499082918291675642E+2329),
    921 => d128!(9.531486195870534996553677466332663E+2331),
    922 => d128!(8.788030272592633266822490623958715E+2334),
    923 => d128!(8.111351941603000505277158845913894E+2337),
    924 => d128!(7.494889194041172466876094773624438E+2340),
    925 => d128!(6.932772504488084531860387665602605E+2343),
    926 => d128!(6.419747339155966276502718978348012E+2346),
    927 => d128!(5.951105783397580738318020492928607E+2349),
    928 => d128!(5.522626166992954925159123017437747E+2352),
    929 => d128!(5.130519709136455125472825283199667E+2355),
    930 => d128!(4.771383329496903266689727513375690E+2358),
    931 => d128!(4.442157879761616941288136314952767E+2361),
    932 => d128!(4.140091143937826989280543045535979E+2364),
    933 => d128!(3.862705037293992580998746661485068E+2367),
    934 => d128!(3.607766504832589070652829381827054E+2370),
    935 => d128!(3.373261682018470781060395472008295E+2373),
    936 => d128!(3.157372934369288651072530161799764E+2376),
    937 => d128!(2.958458439504023466054960761606379E+2379),
    938 => d128!(2.775034016254774011159553194386784E+2382),
    939 => d128!(2.605756941263232796478820449529190E+2385),
    940 => d128!(2.449411524787438828690091222557439E+2388),
    941 => d128!(2.304896244824979937797375840426550E+2391),
    942 => d128!(2.171212262625131101405128041681810E+2394),
    943 => d128!(2.047453163655498628625035743305947E+2397),
    944 => d128!(1.932795786490790705422033741680814E+2400),
    945 => d128!(1.826492018233797216623821885888369E+2403),
    946 => d128!(1.727861449249172166926135504050397E+2406),
    947 => d128!(1.636284792438966042079050322335726E+2409),
    948 => d128!(1.551197983232139807890939705574268E+2412),
    949 => d128!(1.472086886087300677688501780589980E+2415),
    950 => d128!(1.398482541782935643804076691560481E+2418),
    951 => d128!(1.329956897235571797257676933674017E+2421),
    952 => d128!(1.266118966168264350989308440857664E+2424),
    953 => d128!(1.206611374758355926492810944137354E+2427),
    954 => d128!(1.151107251519471553874141640707036E+2430),
    955 => d128!(1.099307425201095333949805266875219E+2433),
    956 => d128!(1.050937898492247139256013835132709E+2436),
    957 => d128!(1.005747568857080512268005240222003E+2439),
    958 => d128!(9.635061709650831307527490201326789E+2441),
    959 => d128!(9.240024179555147223918863103072391E+2444),
    960 => d128!(8.870423212372941334962108578949495E+2447),
    961 => d128!(8.524476707090396622898586344370465E+2450),
    962 => d128!(8.200546592220961551228440063284387E+2453),
    963 => d128!(7.897126368308785973832987780942865E+2456),
    964 => d128!(7.612829819049669678775000220828922E+2459),
    965 => d128!(7.346380775382931240017875213099910E+2462),
    966 => d128!(7.096603829019911577857267455854513E+2465),
    967 => d128!(6.862415902662254495787977629811314E+2468),
    968 => d128!(6.642818593777062351922762345657352E+2471),
    969 => d128!(6.436891217369973419013156712941974E+2474),
    970 => d128!(6.243784480848874216442762011553715E+2477),
    971 => d128!(6.062714730904256864165921913218657E+2480),
    972 => d128!(5.892958718438937671969276099648535E+2483),
    973 => d128!(5.733848833041086354826105644958025E+2486),
    974 => d128!(5.584768763382018109600626898189116E+2489),
    975 => d128!(5.445149544297467656860611225734388E+2492),
    976 => d128!(5.314465955234328433095956556316763E+2495),
    977 => d128!(5.192233238263938879134749555521477E+2498),
    978 => d128!(5.078004107022132223793785065300005E+2501),
    979 => d128!(4.971366020774667447094115578928705E+2504),
    980 => d128!(4.871938700359174098152233267350131E+2507),
    981 => d128!(4.779371865052349790287340835270479E+2510),
    982 => d128!(4.693343171481407494062168700235610E+2513),
    983 => d128!(4.613556337566223566663111832331605E+2516),
    984 => d128!(4.539739436165163989596502043014299E+2519),
    985 => d128!(4.471643344622686529752554512369085E+2522),
    986 => d128!(4.409040337797968918336018749195918E+2525),
    987 => d128!(4.351722813406595322397650505456371E+2528),
    988 => d128!(4.299502139645716178528878699390895E+2531),
    989 => d128!(4.252207616109613300565061033697595E+2534),
    990 => d128!(4.209685539948517167559410423360619E+2537),
    991 => d128!(4.171798370088980513051375729550373E+2540),
    992 => d128!(4.138423983128268668946964723713970E+2543),
    993 => d128!(4.109455015246370788264335970647972E+2546),
    994 => d128!(4.084798285154892563534749954824084E+2549),
    995 => d128!(4.064374293729118100717076205049964E+2552),
    996 => d128!(4.048116796554201628314207900229764E+2555),
    997 => d128!(4.035972446164539023429265276529075E+2558),
    998 => d128!(4.027900501272209945382406745976017E+2561),
    999 => d128!(4.023872600770937735437024339230041E+2564),
    1000 => d128!(4.023872600770937735437024339230041E+2567),
    _ => d128!("NaN"),
  }
}
