#![allow(unused)]
#![macro_use]

/// `input` macro for parsing input from all CP platforms
///
/// # Examples
/// ```ignore
/// input! {
///     source = s,
///     t: usize,
///     k: [(usize, usize); t] 
/// }
/// ``` 
///
/// Omit source when submitting, while developing, you can save the input in file
/// and use this
/// ```
/// use std::fs::File;
/// use std::io::Read;
/// let mut sc = File::open("test.txt").unwrap();
/// let mut s= String::new();
/// sc.read_to_string(&mut s).unwrap();
/// ```
/// 
#[macro_export]
macro_rules! input {
    (source = $s:expr, $($r:tt)*) => {
        let mut iter = $s.split_whitespace();
        input_inner!{iter, $($r)*}
    };
    ($($r:tt)*) => {
        let mut s = {
            use std::io::Read;
            let mut s = String::new();
            std::io::stdin().read_to_string(&mut s).unwrap();
            s
        };
        let mut iter = s.split_whitespace();
        input_inner!{iter, $($r)*}
    };
}

macro_rules! input_inner {
    ($iter:expr) => {};
    ($iter:expr, ) => {};

    ($iter:expr, $var:ident : $t:tt $($r:tt)*) => {
        let $var = read_value!($iter, $t);
        input_inner!{$iter $($r)*}
    };
}

macro_rules! read_value {
    ($iter:expr, ( $($t:tt),* )) => {
        ( $(read_value!($iter, $t)),* )
    };

    ($iter:expr, [ $t:tt ; $len:expr ]) => {
        (0..$len).map(|_| read_value!($iter, $t)).collect::<Vec<_>>()
    };

    ($iter:expr, chars) => {
        read_value!($iter, String).chars().collect::<Vec<char>>()
    };

    ($iter:expr, usize1) => {
        read_value!($iter, usize) - 1
    };

    ($iter:expr, $t:ty) => {
        $iter.next().unwrap().parse::<$t>().expect("Parse error")
    };
}


#[cfg(test)]
mod tests {

    #[test]
    fn test_input_macro() {
        use std::fs::File;
        use std::io::Read;
        let mut sc = File::open("test.txt").unwrap();
        let mut s= String::new();
        sc.read_to_string(&mut s).unwrap();
    
        input! {
            source = s,
            t: usize,
            k: [(usize, usize); t] 
        }

        assert_eq!(t, 3);
        let k_original = [(10usize,20usize), (30,40), (50,60)];
        for res in k.iter().zip(k_original.iter()) {
            assert_eq!(res.0, res.1)
        }
    }

}