//! # Tricks for competitive programming
//! Most used ones are 
//! - Macro [input!] for input parsing all at a time
//! - [Scanner]/[UnsafeScanner] for parsing on demand

#![allow(unused)]
mod macros;

use std::str::{self};
use std::io::{stdin, stdout, BufRead, BufWriter, Write};

pub struct Scanner<R> {
    reader: R,
    buffer: Vec<String>,
}

impl<R: BufRead> Scanner<R> {
    pub fn new(reader: R) -> Self {
        Self {
            reader,
            buffer: vec![],
        }
    }

    pub fn next<T: str::FromStr>(&mut self) -> T {
        loop {
            if let Some(token) = self.buffer.pop() {
                return token.parse().ok().expect("Failed to parse");
            }
            let mut input = String::new();
            self.reader.read_line(&mut input).expect("Failed to read");
            self.buffer = input.split_whitespace().rev().map(String::from).collect();
        }
    }
}
pub struct UnsafeScanner<R> {
    reader: R,
    buf_str: Vec<u8>,
    buf_iter: str::SplitAsciiWhitespace<'static>,
}

impl<R: BufRead> UnsafeScanner<R> {
    pub fn new(reader: R) -> Self {
        Self {
            reader,
            buf_str: vec![],
            buf_iter: "".split_ascii_whitespace(),
        }
    }

    /// Taken from [here](https://github.com/EbTech/rust-algorithms/blob/master/src/scanner.rs)
    pub fn token<T: str::FromStr>(&mut self) -> T {
        loop {
            if let Some(token) = self.buf_iter.next() {
                return token.parse().ok().expect("Failed parse");
            }
            self.buf_str.clear();
            self.reader
                .read_until(b'\n', &mut self.buf_str)
                .expect("Failed read");
            self.buf_iter = unsafe {
                let slice = str::from_utf8_unchecked(&self.buf_str);
                // Pointer that points to "" (empty slice) now points to 
                // slice created from above
                std::mem::transmute(slice.split_ascii_whitespace())
            }
        }
    }
}

/// ## Get random values without `rand` crate
/// ```
/// extern "C" {
/// 	fn rand() -> i32;
/// 	fn time(time: *mut isize) -> isize;
/// 	fn srand(seed: u32);
/// }
/// ```
pub fn rand_c(){}

/// ## Using `entry` in Hashmap for inplace manipulation
/// ```
/// use std::collections::HashMap;

/// let mut letters = HashMap::new();

/// for ch in "a short treatise on fungi".chars() {
///     let counter = letters.entry(ch).or_insert(0);
///     *counter += 1;
/// }

/// assert_eq!(letters[&'s'], 2);
/// assert_eq!(letters[&'t'], 3);
/// assert_eq!(letters[&'u'], 1);
/// assert_eq!(letters.get(&'y'), None);
/// ```
pub fn hashmap(){}

/// ## Use `splice` or `copy_from_slice` for copying data from one vec to another
/// ```
/// let mut _a = vec![5i32,4,3,2,1];
/// let b = vec![1i32,2,3,4,5];
/// let v: Vec<_> = _a.splice(1..3, b[0..2].iter().cloned()).collect();
/// 
/// let mut a = vec![5i32,4,3,2,1];
/// let b = vec![1i32,2,3,4,5];
/// a[1..3].copy_from_slice(&b[0..2]);
/// assert_eq!(_a, a);
/// ```
pub fn copy_splice(){}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_new_scanner() {
        let input: &[u8] = b"50 30";
        let mut scan = Scanner::new(input);
        let mut out = vec![];
        let x = scan.next::<i32>();
        let y = scan.next::<i32>();
        writeln!(out, "{} - {} = {}", x, y, x - y).ok();
        assert_eq!(out, b"50 - 30 = 20\n")
    }

    #[test]
    fn test_unsafe_scanner() {
        let input: &[u8] = b"50 30";
        let mut scan = UnsafeScanner::new(input);
        let mut out = vec![];
        let x = scan.token::<i32>();
        let y = scan.token::<i32>();
        writeln!(out, "{} - {} = {}", x, y, x - y).ok();
        assert_eq!(out, b"50 - 30 = 20\n");
    }

    
}