use std::env;
use std::fs::File;
use std::io::prelude::*;
use std::path::PathBuf;
use std::time::{SystemTime, UNIX_EPOCH};

#[cfg(not(debug_assertions))]
use dirs::home_dir;

use crate::otp::otp_element::OTPElement;

pub fn get_db_path() -> PathBuf {
    match env::var("COTP_DB_PATH") {
        Ok(value) => PathBuf::from(value),
        Err(_e) => get_default_db_path(),
    }
}

// Pushing an absolute path to a PathBuf replaces the entire PathBuf: https://doc.rust-lang.org/std/path/struct.PathBuf.html#method.push
pub fn get_default_db_path() -> PathBuf {
    let result: Option<PathBuf> = {
        #[cfg(not(debug_assertions))]{
            home_dir()
        }
        #[cfg(debug_assertions)]
        Some(PathBuf::from("."))
    };
    match result {
        Some(home) => home,
        None => {
            let current_dir = PathBuf::from(".");
            if let Some(str_dir) = current_dir.to_str(){
                eprintln!("Cannot get home folder, using: {}",str_dir);
            }
            else{
                eprintln!("Cannot get home folder, using");
            }
            current_dir
        },
    }.join(".cotp/db.cotp")
}

pub fn create_db_if_needed() -> Result<bool, ()> {
    let db_path = get_db_path();
    let db_dir = db_path.parent().unwrap();
    if !db_dir.exists() {
        match std::fs::create_dir(db_dir) {
            Ok(()) => {}
            Err(_e) => {}
        }
    }
    if !db_path.exists() {
        return match std::fs::File::create(db_path) {
            Ok(_f) => Ok(true),
            Err(_e) => Err(()),
        }
    }
    Ok(false)
}

pub fn delete_db() -> std::io::Result<()> {
    std::fs::remove_file(get_db_path())
}

pub fn write_to_file(content: &str, file: &mut File) -> Result<(), std::io::Error> {
    file.write_all(content.as_bytes())?;
    file.sync_all()
}

pub fn check_elements(id: usize, elements: &[OTPElement]) -> Result<(), String> {
    if elements.is_empty() {
        return Err(String::from("there are no elements in your database. Type \"cotp -h\" to get help."));
    }

    if id >= elements.len() {
        return Err(format!("{} is a bad index", id + 1));
    }

    Ok(())
}

pub fn millis_before_next_step() -> u64 {
    let now = SystemTime::now();
    let since_the_epoch = now.duration_since(UNIX_EPOCH).unwrap();
    let in_ms = since_the_epoch.as_secs() * 1000 + since_the_epoch.subsec_nanos() as u64 / 1000000;
    in_ms % 30000
}

pub fn percentage() -> u16 {
    (millis_before_next_step() * 100 / 30000) as u16
}

#[cfg(test)]
mod tests {
    use super::create_db_if_needed;

    #[test]
    fn test_db_creation() {
        assert_eq!(Ok(true), create_db_if_needed());
    }
}