/// The ten least positive integers k such that 2ⁿ-k is a prime.
/// In each case these are proven primes (proven using UBASIC's  APRT-CL).
///
/// Note that 2n-k will be an n bit number (for these k's).
/// see https://primes.utm.edu/lists/2small/0bit.html
pub fn closest_primes_to_power_of_2(n: usize) -> anyhow::Result<[u32; 10]> {
    anyhow::ensure!(n >= 2, "The minimum number of bits is 2, but got {}", n);

    anyhow::ensure!(
        n <= 400,
        "The maximum prime handled is 2^400 - 593, but got {}",
        n
    );

    Ok(match n {
        2 => [1, 2, 2, 2, 2, 2, 2, 2, 2, 2],
        3 => [1, 3, 5, 6, 6, 6, 6, 6, 6, 6],
        4 => [3, 5, 9, 11, 13, 14, 14, 14, 14, 14],
        5 => [1, 3, 9, 13, 15, 19, 21, 25, 27, 29],
        6 => [3, 5, 11, 17, 21, 23, 27, 33, 35, 41],
        7 => [1, 15, 19, 21, 25, 27, 31, 39, 45, 49],
        8 => [5, 15, 17, 23, 27, 29, 33, 45, 57, 59],
        9 => [3, 9, 13, 21, 25, 33, 45, 49, 51, 55],
        10 => [3, 5, 11, 15, 27, 33, 41, 47, 53, 57],
        11 => [9, 19, 21, 31, 37, 45, 49, 51, 55, 61],
        12 => [3, 5, 17, 23, 39, 45, 47, 69, 75, 77],
        13 => [1, 13, 21, 25, 31, 45, 69, 75, 81, 91],
        14 => [3, 15, 21, 23, 35, 45, 51, 65, 83, 111],
        15 => [19, 49, 51, 55, 61, 75, 81, 115, 121, 135],
        16 => [15, 17, 39, 57, 87, 89, 99, 113, 117, 123],
        17 => [1, 9, 13, 31, 49, 61, 63, 85, 91, 99],
        18 => [5, 11, 17, 23, 33, 35, 41, 65, 75, 93],
        19 => [1, 19, 27, 31, 45, 57, 67, 69, 85, 87],
        20 => [3, 5, 17, 27, 59, 69, 129, 143, 153, 185],
        21 => [9, 19, 21, 55, 61, 69, 105, 111, 121, 129],
        22 => [3, 17, 27, 33, 57, 87, 105, 113, 117, 123],
        23 => [15, 21, 27, 37, 61, 69, 135, 147, 157, 159],
        24 => [3, 17, 33, 63, 75, 77, 89, 95, 117, 167],
        25 => [39, 49, 61, 85, 91, 115, 141, 159, 165, 183],
        26 => [5, 27, 45, 87, 101, 107, 111, 117, 125, 135],
        27 => [39, 79, 111, 115, 135, 187, 199, 219, 231, 235],
        28 => [57, 89, 95, 119, 125, 143, 165, 183, 213, 273],
        29 => [3, 33, 43, 63, 73, 75, 93, 99, 121, 133],
        30 => [35, 41, 83, 101, 105, 107, 135, 153, 161, 173],
        31 => [1, 19, 61, 69, 85, 99, 105, 151, 159, 171],
        32 => [5, 17, 65, 99, 107, 135, 153, 185, 209, 267],
        33 => [9, 25, 49, 79, 105, 285, 301, 303, 321, 355],
        34 => [41, 77, 113, 131, 143, 165, 185, 207, 227, 281],
        35 => [31, 49, 61, 69, 79, 121, 141, 247, 309, 325],
        36 => [5, 17, 23, 65, 117, 137, 159, 173, 189, 233],
        37 => [25, 31, 45, 69, 123, 141, 199, 201, 351, 375],
        38 => [45, 87, 107, 131, 153, 185, 191, 227, 231, 257],
        39 => [7, 19, 67, 91, 135, 165, 219, 231, 241, 301],
        40 => [87, 167, 195, 203, 213, 285, 293, 299, 389, 437],
        41 => [21, 31, 55, 63, 73, 75, 91, 111, 133, 139],
        42 => [11, 17, 33, 53, 65, 143, 161, 165, 215, 227],
        43 => [57, 67, 117, 175, 255, 267, 291, 309, 319, 369],
        44 => [17, 117, 119, 129, 143, 149, 287, 327, 359, 377],
        45 => [55, 69, 81, 93, 121, 133, 139, 159, 193, 229],
        46 => [21, 57, 63, 77, 167, 197, 237, 287, 305, 311],
        47 => [115, 127, 147, 279, 297, 339, 435, 541, 619, 649],
        48 => [59, 65, 89, 93, 147, 165, 189, 233, 243, 257],
        49 => [81, 111, 123, 139, 181, 201, 213, 265, 283, 339],
        50 => [27, 35, 51, 71, 113, 117, 131, 161, 195, 233],
        51 => [129, 139, 165, 231, 237, 247, 355, 391, 397, 439],
        52 => [47, 143, 173, 183, 197, 209, 269, 285, 335, 395],
        53 => [111, 145, 231, 265, 315, 339, 343, 369, 379, 421],
        54 => [33, 53, 131, 165, 195, 245, 255, 257, 315, 327],
        55 => [55, 67, 99, 127, 147, 169, 171, 199, 207, 267],
        56 => [5, 27, 47, 57, 89, 93, 147, 177, 189, 195],
        57 => [13, 25, 49, 61, 69, 111, 195, 273, 363, 423],
        58 => [27, 57, 63, 137, 141, 147, 161, 203, 213, 251],
        59 => [55, 99, 225, 427, 517, 607, 649, 687, 861, 871],
        60 => [93, 107, 173, 179, 257, 279, 369, 395, 399, 453],
        61 => [1, 31, 45, 229, 259, 283, 339, 391, 403, 465],
        62 => [57, 87, 117, 143, 153, 167, 171, 195, 203, 273],
        63 => [25, 165, 259, 301, 375, 387, 391, 409, 457, 471],
        64 => [59, 83, 95, 179, 189, 257, 279, 323, 353, 363],
        65 => [49, 79, 115, 141, 163, 229, 301, 345, 453, 493],
        66 => [5, 45, 173, 203, 275, 297, 387, 401, 443, 495],
        67 => [19, 31, 49, 57, 61, 75, 81, 165, 181, 237],
        68 => [23, 83, 125, 147, 149, 167, 285, 315, 345, 357],
        69 => [19, 91, 93, 103, 129, 153, 165, 201, 255, 385],
        70 => [35, 71, 167, 215, 263, 267, 273, 447, 473, 585],
        71 => [231, 325, 411, 435, 441, 465, 559, 577, 601, 721],
        72 => [93, 107, 129, 167, 249, 269, 329, 347, 429, 473],
        73 => [69, 181, 199, 273, 319, 433, 475, 501, 523, 645],
        74 => [35, 45, 57, 135, 153, 237, 257, 275, 461, 465],
        75 => [97, 207, 231, 271, 279, 289, 325, 381, 409, 427],
        76 => [15, 63, 117, 123, 143, 189, 215, 267, 285, 347],
        77 => [33, 43, 145, 163, 195, 261, 295, 379, 433, 451],
        78 => [11, 95, 111, 123, 147, 153, 191, 263, 303, 507],
        79 => [67, 199, 249, 277, 355, 367, 405, 447, 477, 511],
        80 => [65, 93, 117, 143, 285, 317, 549, 645, 765, 933],
        81 => [51, 63, 163, 205, 333, 349, 429, 433, 481, 553],
        82 => [57, 113, 185, 315, 363, 365, 375, 453, 623, 635],
        83 => [55, 97, 117, 121, 139, 285, 307, 405, 429, 561],
        84 => [35, 69, 213, 215, 333, 399, 525, 563, 587, 597],
        85 => [19, 61, 181, 295, 411, 433, 469, 519, 531, 823],
        86 => [35, 41, 65, 71, 113, 255, 261, 293, 357, 461],
        87 => [67, 129, 181, 195, 201, 217, 261, 277, 289, 339],
        88 => [299, 455, 483, 563, 605, 719, 735, 743, 753, 797],
        89 => [1, 21, 31, 49, 69, 99, 103, 265, 321, 441],
        90 => [33, 41, 53, 75, 227, 263, 273, 291, 297, 317],
        91 => [45, 81, 111, 201, 315, 339, 567, 619, 655, 771],
        92 => [83, 149, 197, 317, 363, 419, 485, 497, 519, 537],
        93 => [25, 51, 79, 105, 273, 405, 489, 553, 571, 579],
        94 => [3, 11, 105, 173, 273, 297, 321, 395, 407, 431],
        95 => [15, 37, 211, 339, 387, 415, 441, 447, 555, 561],
        96 => [17, 87, 93, 147, 165, 189, 237, 243, 315, 347],
        97 => [141, 165, 349, 399, 453, 595, 729, 741, 859, 885],
        98 => [51, 65, 107, 117, 141, 227, 273, 363, 471, 525],
        99 => [115, 145, 247, 319, 381, 427, 675, 717, 1207, 1231],
        100 => [15, 99, 153, 183, 267, 285, 357, 479, 603, 833],
        101 => [69, 99, 213, 259, 363, 483, 493, 573, 619, 703],
        102 => [33, 63, 417, 447, 473, 483, 671, 681, 707, 801],
        103 => [97, 127, 237, 339, 441, 571, 619, 631, 697, 727],
        104 => [17, 29, 309, 369, 395, 587, 639, 657, 717, 857],
        105 => [13, 139, 151, 163, 231, 279, 313, 385, 469, 541],
        106 => [117, 407, 431, 593, 623, 717, 791, 813, 941, 965],
        107 => [1, 171, 321, 369, 441, 489, 721, 745, 919, 1077],
        108 => [59, 137, 227, 233, 269, 383, 467, 483, 545, 633],
        109 => [31, 91, 165, 399, 493, 735, 811, 943, 945, 1215],
        110 => [21, 135, 143, 195, 215, 443, 467, 531, 533, 623],
        111 => [37, 69, 175, 261, 429, 471, 487, 565, 765, 831],
        112 => [75, 189, 269, 297, 327, 519, 537, 623, 699, 875],
        113 => [133, 211, 463, 765, 831, 979, 1023, 1069, 1261, 1351],
        114 => [11, 35, 53, 75, 153, 161, 195, 227, 237, 257],
        115 => [67, 85, 97, 129, 277, 369, 379, 711, 865, 985],
        116 => [3, 69, 83, 89, 117, 197, 303, 353, 359, 429],
        117 => [279, 319, 373, 391, 451, 495, 541, 555, 685, 793],
        118 => [5, 111, 273, 375, 447, 497, 563, 593, 735, 741],
        119 => [69, 169, 231, 565, 745, 799, 801, 867, 871, 981],
        120 => [119, 395, 515, 555, 615, 765, 873, 899, 1049, 1143],
        121 => [73, 165, 225, 229, 433, 531, 565, 655, 685, 741],
        122 => [3, 113, 167, 203, 341, 825, 843, 951, 1011, 1085],
        123 => [67, 79, 135, 241, 471, 735, 847, 891, 1029, 1069],
        124 => [59, 99, 155, 165, 327, 389, 503, 563, 615, 719],
        125 => [9, 135, 385, 405, 415, 579, 633, 685, 853, 985],
        126 => [137, 203, 237, 261, 335, 341, 465, 663, 671, 783],
        127 => [1, 25, 39, 295, 309, 507, 511, 577, 697, 735],
        128 => [159, 173, 233, 237, 275, 357, 675, 713, 797, 1193],
        129 => [25, 315, 403, 613, 735, 741, 805, 1113, 1185, 1365],
        130 => [5, 27, 113, 173, 315, 417, 425, 447, 455, 585],
        131 => [69, 181, 271, 315, 385, 421, 427, 615, 625, 681],
        132 => [347, 363, 405, 465, 527, 585, 725, 735, 915, 923],
        133 => [99, 103, 183, 259, 453, 475, 631, 973, 1011, 1039],
        134 => [45, 117, 185, 197, 345, 495, 603, 695, 957, 1115],
        135 => [45, 55, 105, 205, 301, 409, 705, 795, 819, 915],
        136 => [113, 243, 257, 297, 299, 335, 365, 369, 377, 423],
        137 => [13, 33, 111, 231, 265, 303, 339, 421, 553, 555],
        138 => [105, 203, 237, 261, 297, 423, 513, 765, 983, 1071],
        139 => [187, 301, 397, 447, 517, 619, 837, 951, 997, 1245],
        140 => [27, 57, 77, 129, 147, 357, 623, 759, 773, 843],
        141 => [9, 103, 111, 165, 171, 193, 349, 523, 595, 625],
        142 => [111, 351, 365, 375, 453, 585, 885, 887, 1215, 1383],
        143 => [69, 127, 229, 351, 355, 381, 391, 465, 469, 471],
        144 => [83, 167, 279, 369, 377, 387, 395, 413, 425, 473],
        145 => [151, 295, 355, 453, 481, 501, 543, 783, 831, 973],
        146 => [153, 335, 573, 695, 1013, 1037, 1301, 1371, 1643, 1701],
        147 => [145, 325, 387, 427, 507, 615, 657, 915, 955, 1137],
        148 => [167, 197, 207, 347, 375, 585, 669, 675, 935, 1029],
        149 => [31, 33, 99, 199, 313, 381, 405, 433, 631, 679],
        150 => [3, 5, 185, 713, 803, 905, 1173, 1175, 1193, 1293],
        151 => [195, 295, 309, 319, 445, 465, 517, 735, 775, 829],
        152 => [17, 23, 209, 233, 303, 539, 623, 707, 989, 1199],
        153 => [69, 145, 171, 399, 451, 553, 555, 639, 735, 885],
        154 => [243, 371, 393, 437, 675, 773, 795, 897, 1023, 1077],
        155 => [31, 49, 289, 381, 405, 441, 499, 511, 825, 867],
        156 => [143, 167, 173, 195, 587, 597, 609, 767, 1017, 1343],
        157 => [19, 133, 181, 213, 259, 333, 435, 451, 753, 861],
        158 => [15, 155, 213, 255, 321, 527, 621, 665, 731, 755],
        159 => [91, 241, 301, 441, 459, 475, 535, 651, 657, 777],
        160 => [47, 57, 75, 189, 285, 383, 465, 543, 659, 843],
        161 => [159, 399, 493, 685, 709, 765, 973, 1011, 1099, 1263],
        162 => [101, 317, 447, 627, 767, 861, 1077, 1095, 1113, 1197],
        163 => [55, 69, 577, 589, 621, 645, 759, 867, 915, 1111],
        164 => [63, 155, 485, 735, 779, 957, 1037, 1085, 1175, 1253],
        165 => [25, 61, 115, 271, 313, 361, 391, 501, 511, 699],
        166 => [5, 237, 633, 705, 753, 935, 1133, 1341, 1343, 1665],
        167 => [135, 471, 577, 591, 729, 771, 801, 1015, 1221, 1231],
        168 => [257, 585, 609, 633, 909, 1143, 1253, 1385, 1467, 1649],
        169 => [643, 675, 841, 973, 1011, 1131, 1201, 1203, 1251, 1275],
        170 => [143, 153, 195, 255, 357, 525, 605, 761, 825, 897],
        171 => [19, 187, 231, 649, 661, 669, 691, 831, 861, 1015],
        172 => [95, 155, 227, 525, 537, 629, 825, 855, 879, 927],
        173 => [55, 103, 303, 313, 343, 421, 441, 511, 531, 609],
        174 => [3, 17, 143, 153, 161, 215, 275, 411, 431, 465],
        175 => [229, 267, 727, 837, 847, 1039, 1069, 1089, 1107, 1299],
        176 => [233, 327, 359, 455, 495, 533, 563, 657, 743, 825],
        177 => [339, 609, 619, 819, 859, 873, 919, 1029, 1281, 1299],
        178 => [41, 305, 405, 557, 563, 827, 951, 1277, 1295, 1331],
        179 => [49, 159, 205, 381, 469, 471, 507, 639, 679, 687],
        180 => [47, 107, 189, 447, 569, 887, 929, 1287, 1547, 1823],
        181 => [165, 199, 271, 283, 399, 511, 675, 813, 825, 969],
        182 => [161, 233, 267, 281, 413, 605, 681, 755, 995, 1211],
        183 => [147, 319, 357, 609, 805, 945, 985, 1125, 1159, 1225],
        184 => [33, 59, 287, 309, 323, 963, 1145, 1187, 1203, 1233],
        185 => [303, 321, 339, 429, 651, 723, 789, 819, 915, 1029],
        186 => [371, 485, 677, 723, 773, 923, 1371, 1445, 1497, 1577],
        187 => [85, 105, 391, 477, 511, 589, 805, 861, 957, 1057],
        188 => [125, 143, 167, 173, 363, 705, 765, 855, 873, 1017],
        189 => [25, 301, 469, 583, 591, 711, 819, 921, 951, 973],
        190 => [11, 33, 95, 273, 651, 717, 851, 1083, 1163, 1251],
        191 => [19, 51, 69, 139, 201, 237, 325, 769, 771, 945],
        192 => [237, 333, 399, 489, 527, 663, 915, 945, 1059, 1143],
        193 => [31, 123, 423, 909, 1195, 1245, 1261, 1365, 1369, 1435],
        194 => [33, 75, 317, 393, 411, 705, 801, 987, 1185, 1191],
        195 => [135, 915, 975, 979, 1005, 1081, 1201, 1225, 1227, 1299],
        196 => [15, 47, 215, 459, 507, 533, 585, 663, 699, 885],
        197 => [75, 111, 169, 439, 775, 853, 1051, 1155, 1431, 1779],
        198 => [17, 45, 417, 485, 615, 627, 801, 1263, 1265, 1295],
        199 => [49, 189, 445, 459, 735, 909, 1021, 1065, 1707, 2047],
        200 => [75, 117, 285, 383, 387, 635, 827, 1275, 1307, 1317],
        201 => [55, 111, 313, 381, 385, 439, 481, 663, 679, 865],
        202 => [183, 195, 273, 585, 845, 977, 1043, 1067, 1161, 1355],
        203 => [159, 187, 237, 271, 495, 741, 829, 931, 957, 1101],
        204 => [167, 249, 857, 1125, 1137, 1169, 1487, 1515, 1683, 1815],
        205 => [81, 229, 325, 343, 369, 441, 753, 825, 969, 1089],
        206 => [5, 63, 261, 297, 393, 497, 567, 813, 885, 1017],
        207 => [91, 157, 297, 369, 387, 429, 459, 481, 517, 871],
        208 => [299, 423, 563, 675, 1383, 1505, 1583, 1593, 1695, 1749],
        209 => [33, 273, 301, 321, 439, 451, 493, 735, 873, 885],
        210 => [47, 65, 165, 171, 203, 213, 503, 555, 741, 755],
        211 => [175, 481, 495, 601, 837, 1099, 1281, 1329, 1357, 1917],
        212 => [23, 29, 99, 149, 357, 413, 479, 497, 533, 695],
        213 => [3, 61, 123, 513, 595, 793, 1315, 1329, 1335, 1683],
        214 => [185, 255, 395, 447, 623, 633, 707, 743, 813, 1035],
        215 => [157, 237, 309, 765, 897, 1081, 1147, 1189, 1377, 1477],
        216 => [377, 479, 683, 707, 843, 1167, 1215, 1253, 1343, 1377],
        217 => [61, 229, 369, 441, 519, 675, 729, 811, 999, 1033],
        218 => [33, 117, 233, 243, 257, 275, 297, 341, 623, 747],
        219 => [121, 261, 277, 355, 397, 471, 615, 685, 1105, 1189],
        220 => [77, 167, 395, 473, 483, 585, 587, 609, 923, 963],
        221 => [3, 133, 309, 373, 411, 573, 759, 855, 979, 999],
        222 => [117, 263, 335, 437, 641, 647, 875, 1035, 1103, 1125],
        223 => [235, 259, 549, 715, 777, 1125, 1221, 1225, 1417, 1621],
        224 => [63, 363, 573, 719, 773, 857, 1025, 1223, 1227, 1253],
        225 => [49, 81, 103, 261, 445, 489, 609, 625, 741, 825],
        226 => [5, 77, 155, 203, 215, 417, 533, 573, 767, 833],
        227 => [405, 441, 625, 721, 835, 855, 1017, 1149, 1479, 1507],
        228 => [93, 149, 185, 455, 537, 549, 723, 899, 1209, 1283],
        229 => [91, 291, 475, 531, 565, 685, 733, 775, 939, 1093],
        230 => [27, 77, 345, 351, 831, 917, 1221, 1245, 1247, 1335],
        231 => [165, 217, 235, 249, 345, 391, 399, 439, 481, 559],
        232 => [567, 665, 833, 875, 1163, 1197, 1253, 1403, 1917, 2097],
        233 => [3, 159, 289, 373, 511, 531, 595, 615, 759, 1113],
        234 => [83, 243, 257, 341, 503, 581, 593, 683, 1157, 1803],
        235 => [15, 151, 181, 259, 451, 537, 561, 679, 727, 1167],
        236 => [209, 455, 513, 569, 657, 875, 915, 1203, 1317, 1349],
        237 => [181, 649, 765, 829, 949, 1449, 1515, 1633, 1689, 1711],
        238 => [161, 215, 383, 425, 665, 731, 791, 825, 1263, 1313],
        239 => [87, 199, 421, 531, 939, 1147, 1395, 1701, 2001, 2187],
        240 => [467, 797, 887, 1493, 1529, 2027, 2093, 2253, 2495, 2589],
        241 => [39, 819, 1383, 2065, 2113, 2133, 2215, 2253, 2295, 2505],
        242 => [63, 267, 281, 527, 777, 945, 971, 1077, 1223, 1487],
        243 => [9, 31, 199, 475, 507, 699, 1117, 1125, 1179, 1417],
        244 => [189, 329, 767, 1065, 1289, 1553, 1565, 1923, 2105, 2505],
        245 => [163, 303, 489, 759, 843, 861, 1039, 1321, 1389, 1413],
        246 => [107, 171, 243, 483, 567, 797, 945, 1155, 1617, 1697],
        247 => [81, 309, 361, 411, 525, 571, 729, 939, 1047, 1149],
        248 => [237, 387, 485, 603, 605, 765, 887, 1097, 1223, 1515],
        249 => [75, 279, 385, 403, 531, 583, 595, 693, 999, 1021],
        250 => [207, 407, 1053, 1391, 1655, 1671, 1785, 1793, 2111, 2327],
        251 => [9, 325, 355, 369, 451, 465, 615, 619, 1141, 1339],
        252 => [129, 143, 413, 549, 705, 749, 839, 1077, 1133, 1175],
        253 => [273, 391, 421, 1615, 1723, 1725, 1791, 1879, 1951, 2091],
        254 => [245, 521, 701, 933, 941, 1223, 1311, 1427, 1491, 1527],
        255 => [19, 31, 475, 735, 765, 921, 949, 1285, 1311, 1351],
        256 => [189, 357, 435, 587, 617, 923, 1053, 1299, 1539, 1883],
        257 => [93, 363, 493, 679, 813, 819, 1861, 2113, 2211, 2233],
        258 => [87, 1017, 1203, 1355, 1385, 1547, 1773, 2411, 2415, 2747],
        259 => [361, 417, 561, 745, 885, 987, 1069, 1071, 1159, 1305],
        260 => [149, 597, 687, 689, 803, 983, 995, 1247, 1373, 1419],
        261 => [223, 261, 361, 609, 1251, 1263, 1629, 1791, 2023, 2095],
        262 => [71, 287, 453, 515, 711, 1013, 1127, 1187, 1365, 1415],
        263 => [747, 819, 939, 1017, 1101, 1297, 1665, 1767, 2337, 2679],
        264 => [275, 363, 567, 1245, 1257, 1355, 1419, 1505, 1635, 2013],
        265 => [49, 115, 139, 211, 489, 601, 1063, 1281, 1285, 1399],
        266 => [3, 213, 365, 1115, 1611, 1851, 2171, 2177, 2273, 2477],
        267 => [265, 427, 481, 517, 555, 595, 1099, 1381, 1449, 1797],
        268 => [77, 329, 719, 825, 1305, 1635, 1749, 2009, 2259, 2273],
        269 => [241, 343, 603, 615, 709, 1155, 1281, 1431, 1713, 1759],
        270 => [53, 611, 1071, 1251, 1397, 1691, 1847, 1853, 2133, 2343],
        271 => [169, 441, 967, 1221, 1419, 2289, 2295, 2617, 2775, 2817],
        272 => [237, 287, 689, 905, 1253, 1443, 2159, 2355, 2367, 2397],
        273 => [205, 321, 345, 795, 963, 1101, 1269, 1321, 1329, 1725],
        274 => [305, 351, 707, 785, 801, 843, 1077, 1263, 1793, 1833],
        275 => [129, 199, 205, 651, 721, 939, 999, 1017, 1051, 1629],
        276 => [89, 453, 473, 483, 609, 675, 915, 1133, 1287, 1295],
        277 => [103, 181, 859, 1221, 1603, 2071, 2115, 2289, 2343, 2409],
        278 => [93, 623, 653, 665, 831, 857, 947, 1493, 1553, 1563],
        279 => [69, 231, 535, 751, 781, 1159, 1509, 1629, 1839, 2035],
        280 => [47, 105, 195, 725, 1077, 1415, 2105, 2199, 2805, 3029],
        281 => [139, 259, 601, 615, 633, 663, 729, 853, 931, 1195],
        282 => [83, 93, 237, 245, 425, 443, 503, 557, 647, 1431],
        283 => [45, 1225, 1239, 1431, 1455, 1615, 2107, 2355, 2485, 2667],
        284 => [173, 323, 2009, 2109, 2159, 2333, 2417, 3045, 3227, 3713],
        285 => [9, 33, 735, 1081, 1243, 1641, 1839, 1879, 1965, 2245],
        286 => [165, 521, 533, 593, 617, 915, 983, 1265, 1433, 1797],
        287 => [115, 435, 771, 835, 937, 951, 1401, 1441, 1759, 1861],
        288 => [167, 525, 567, 1355, 1473, 1595, 1875, 2349, 2537, 2765],
        289 => [493, 843, 945, 1383, 1489, 1621, 2083, 2143, 2575, 2649],
        290 => [47, 83, 503, 825, 1221, 1553, 1587, 1641, 1697, 1791],
        291 => [19, 315, 427, 861, 907, 987, 1017, 1195, 1329, 2085],
        292 => [167, 197, 207, 657, 789, 923, 977, 1623, 1655, 1967],
        293 => [601, 1095, 1389, 1455, 1459, 1693, 1849, 1863, 2071, 2163],
        294 => [35, 135, 177, 245, 315, 515, 633, 917, 1047, 1125],
        295 => [171, 421, 459, 511, 585, 1291, 1461, 1579, 1611, 2077],
        296 => [285, 785, 1007, 1287, 1389, 1709, 1727, 2117, 2295, 2477],
        297 => [123, 171, 285, 333, 619, 693, 703, 859, 1273, 1501],
        298 => [341, 483, 953, 1223, 1251, 1307, 1475, 1805, 1845, 1907],
        299 => [69, 405, 429, 675, 1015, 1017, 1105, 1137, 1167, 1539],
        300 => [153, 185, 383, 413, 483, 539, 693, 779, 1047, 1097],
        301 => [265, 523, 861, 1249, 1299, 1309, 1365, 1869, 1963, 1981],
        302 => [267, 293, 647, 1043, 1307, 1581, 1601, 2037, 2141, 2241],
        303 => [121, 207, 241, 249, 321, 381, 549, 637, 835, 955],
        304 => [75, 1403, 1685, 1707, 1739, 1883, 1937, 2219, 2255, 2307],
        305 => [103, 429, 453, 1195, 1339, 1581, 2019, 2581, 2989, 3043],
        306 => [503, 947, 1017, 1797, 2063, 2093, 2175, 2597, 2655, 2817],
        307 => [99, 147, 187, 255, 319, 369, 511, 631, 879, 1021],
        308 => [159, 189, 615, 777, 959, 1025, 1079, 1139, 1409, 1443],
        309 => [493, 2673, 2733, 2949, 3243, 3525, 3595, 3799, 3969, 4051],
        310 => [77, 255, 635, 645, 755, 1097, 1127, 1235, 1377, 1533],
        311 => [45, 75, 181, 199, 271, 297, 471, 789, 885, 1071],
        312 => [203, 689, 753, 1385, 1449, 1547, 1575, 1655, 1683, 1799],
        313 => [139, 343, 439, 669, 675, 759, 915, 1893, 1993, 2371],
        314 => [113, 453, 677, 695, 981, 1013, 1037, 1067, 1121, 1277],
        315 => [465, 691, 1041, 1417, 1479, 1515, 1795, 2217, 2769, 2839],
        316 => [57, 243, 267, 293, 353, 399, 507, 899, 1259, 1323],
        317 => [33, 379, 523, 609, 1009, 1065, 1351, 1521, 1773, 1821],
        318 => [165, 275, 681, 807, 1337, 1427, 1617, 1931, 2075, 2295],
        319 => [795, 865, 1191, 1911, 2059, 2475, 2989, 3435, 3855, 4045],
        320 => [197, 743, 825, 843, 873, 1007, 1017, 1217, 1815, 2955],
        321 => [9, 45, 609, 769, 1053, 1165, 1389, 1395, 1431, 1675],
        322 => [11, 185, 923, 1121, 1233, 1307, 2307, 2373, 2583, 2717],
        323 => [141, 247, 309, 919, 1269, 1345, 1431, 1629, 1731, 1951],
        324 => [23, 177, 777, 923, 1055, 1299, 1329, 1385, 1449, 1799],
        325 => [399, 433, 1051, 1219, 1261, 1485, 1701, 2013, 2349, 2413],
        326 => [101, 117, 215, 731, 1731, 1823, 1833, 1995, 2603, 2807],
        327 => [595, 951, 979, 1077, 1297, 1549, 1939, 2667, 2961, 3055],
        328 => [155, 437, 629, 693, 843, 977, 1019, 1119, 1475, 1503],
        329 => [139, 243, 273, 339, 489, 531, 1665, 1963, 2145, 2343],
        330 => [255, 437, 927, 1001, 1217, 1445, 1595, 1911, 3185, 3435],
        331 => [61, 459, 745, 969, 1261, 1449, 1599, 1771, 2029, 2737],
        332 => [707, 1077, 1109, 1217, 1293, 1485, 1487, 1613, 2789, 2987],
        333 => [483, 741, 805, 913, 955, 1041, 1083, 1131, 1243, 1353],
        334 => [243, 603, 711, 981, 1085, 1425, 1637, 1901, 1953, 2423],
        335 => [321, 511, 541, 759, 871, 1227, 1399, 1821, 2131, 2337],
        336 => [3, 17, 303, 585, 627, 707, 723, 1047, 1413, 1479],
        337 => [75, 1443, 1579, 1615, 1659, 1693, 1741, 1785, 1915, 2793],
        338 => [15, 275, 315, 401, 467, 603, 623, 843, 951, 1053],
        339 => [147, 415, 751, 865, 901, 921, 1311, 1375, 1881, 1911],
        340 => [293, 299, 437, 699, 1187, 1203, 1209, 1217, 1389, 1439],
        341 => [229, 631, 765, 861, 951, 1071, 1101, 1203, 1263, 1503],
        342 => [65, 237, 263, 317, 707, 1017, 1085, 1353, 1517, 1755],
        343 => [199, 291, 495, 501, 561, 589, 691, 841, 915, 1377],
        344 => [119, 219, 527, 569, 617, 845, 993, 1079, 1107, 1253],
        345 => [475, 493, 859, 951, 1083, 1113, 1521, 2425, 2443, 2491],
        346 => [45, 57, 197, 363, 407, 1467, 1565, 1601, 1623, 1995],
        347 => [211, 367, 451, 1065, 1417, 1455, 1507, 1659, 1777, 2109],
        348 => [117, 143, 257, 359, 629, 759, 1085, 1295, 1469, 1533],
        349 => [285, 433, 579, 673, 859, 895, 1009, 2071, 2175, 2419],
        350 => [113, 131, 401, 425, 513, 555, 923, 1097, 1461, 2073],
        351 => [61, 135, 291, 429, 691, 1159, 2139, 2155, 2901, 3517],
        352 => [657, 879, 963, 999, 1263, 1305, 1323, 1439, 1449, 1865],
        353 => [139, 489, 523, 595, 621, 735, 1029, 1489, 1879, 1951],
        354 => [153, 251, 525, 537, 573, 783, 1371, 1563, 1605, 1691],
        355 => [49, 369, 589, 717, 957, 1221, 1609, 2185, 2197, 2355],
        356 => [173, 227, 393, 759, 803, 855, 989, 1385, 1497, 1623],
        357 => [243, 541, 739, 1033, 1075, 1105, 1621, 1725, 2575, 2739],
        358 => [671, 1005, 1161, 1211, 1247, 1365, 1491, 1793, 2681, 2957],
        359 => [411, 855, 901, 925, 1275, 1791, 2179, 2277, 2995, 3027],
        360 => [719, 1155, 1307, 1515, 1785, 1823, 2117, 2207, 2345, 2669],
        361 => [369, 663, 1779, 2539, 2691, 2793, 3501, 3751, 3909, 4173],
        362 => [605, 663, 1073, 1563, 1571, 1955, 2297, 2327, 2385, 2553],
        363 => [75, 97, 105, 111, 1287, 2251, 2341, 2391, 2481, 2511],
        364 => [923, 1239, 1653, 1725, 1917, 1949, 2009, 2045, 2339, 2609],
        365 => [169, 261, 295, 325, 369, 493, 639, 723, 2079, 2241],
        366 => [167, 507, 635, 657, 747, 837, 1115, 1511, 1785, 1787],
        367 => [487, 801, 1279, 1407, 1509, 1827, 2365, 2409, 2467, 3049],
        368 => [315, 419, 785, 1139, 1289, 1953, 2567, 2595, 2799, 3107],
        369 => [25, 195, 445, 825, 1339, 1551, 1599, 1729, 1869, 1959],
        370 => [495, 813, 1797, 1977, 2001, 2063, 2303, 2315, 2505, 2913],
        371 => [741, 747, 925, 1077, 1485, 1647, 1815, 2115, 2305, 2319],
        372 => [177, 207, 563, 797, 1749, 1997, 2247, 2475, 2615, 2873],
        373 => [333, 489, 735, 843, 1813, 2295, 2491, 3189, 3513, 3519],
        374 => [65, 153, 371, 495, 855, 987, 1433, 1983, 2235, 2247],
        375 => [679, 979, 1089, 1119, 1669, 1767, 2065, 2547, 2635, 2697],
        376 => [57, 473, 519, 629, 1035, 1239, 1929, 2085, 2237, 2985],
        377 => [259, 321, 511, 663, 1485, 1693, 1791, 2253, 2409, 2553],
        378 => [417, 447, 825, 1991, 2207, 2693, 2993, 3101, 3177, 3287],
        379 => [19, 99, 319, 355, 679, 711, 795, 925, 1069, 1455],
        380 => [65, 483, 677, 723, 1133, 1139, 1175, 1263, 1347, 1469],
        381 => [313, 735, 1203, 2209, 2629, 3033, 3055, 3091, 3315, 4003],
        382 => [105, 227, 635, 677, 963, 1083, 1641, 2085, 2243, 2411],
        383 => [31, 187, 367, 421, 471, 567, 607, 747, 1035, 1699],
        384 => [317, 1437, 1557, 1617, 2147, 2319, 2729, 3087, 3093, 3273],
        385 => [265, 1215, 1911, 2005, 2101, 2401, 2449, 2455, 2565, 2679],
        386 => [231, 747, 881, 923, 1287, 1611, 2493, 2735, 2837, 3633],
        387 => [615, 621, 795, 919, 1021, 1111, 1255, 1287, 1689, 2095],
        388 => [45, 63, 269, 347, 483, 525, 717, 837, 875, 959],
        389 => [21, 51, 103, 313, 843, 871, 883, 1141, 1179, 1743],
        390 => [137, 293, 383, 461, 1017, 1283, 1773, 2057, 2693, 3047],
        391 => [105, 127, 399, 645, 675, 1671, 1791, 2085, 2227, 2275],
        392 => [107, 299, 623, 1035, 1389, 1437, 1589, 1827, 2285, 2607],
        393 => [93, 331, 549, 1113, 1411, 2523, 2551, 2719, 3393, 4291],
        394 => [377, 447, 987, 1161, 1497, 1571, 2043, 2187, 2303, 2615],
        395 => [531, 1137, 2161, 2205, 2215, 2907, 3055, 3151, 3481, 3655],
        396 => [605, 695, 909, 945, 1229, 1365, 1997, 2397, 2837, 3285],
        397 => [81, 463, 673, 1071, 1165, 1425, 1905, 1999, 2061, 2133],
        398 => [131, 231, 777, 951, 1457, 1547, 1673, 1893, 2145, 2241],
        399 => [91, 219, 937, 1731, 2749, 2875, 3201, 3489, 3729, 4149],
        400 => [593, 663, 767, 879, 1205, 2457, 3107, 3195, 3263, 3267],
        _ => panic!("closest prime < 2^{} is not recorded here", n),
    })
}

#[cfg(test)]
mod tests {

    // generate primes the ways it is laid out in primes.rs
    #[test]
    #[ignore]
    pub fn generate_primes() {
        let primes = [
            2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83,
            89, 97, 101, 103, 107, 109, 113, 127, 131, 137, 139, 149, 151, 157, 163, 167, 173, 179,
            181, 191, 193, 197, 199, 211, 223, 227, 229, 233, 239, 241, 251, 257,
        ];

        for i in 2..9 {
            let base = 1_u32 << i;
            //fin index of first prime below base
            let mut index = 0;
            for (j, p) in primes.iter().enumerate() {
                if *p > base {
                    break
                }
                index = j;
            }
            print!("{} => [", i);
            for j in 0..10 {
                if j != 0 {
                    print!(", ")
                };
                let idx = if index > j { index - j } else { 0 };
                print!("{}", base - primes[idx]);
            }
            println!("],");
        }
    }
}
