"""
sdn.py defines services to start Open vSwitch and the Ryu SDN Controller.
"""

import re
from typing import Tuple

from core.nodes.base import CoreNode
from core.services.coreservices import CoreService


class SdnService(CoreService):
    """
    Parent class for SDN services.
    """

    group: str = "SDN"

    @classmethod
    def generate_config(cls, node: CoreNode, filename: str) -> str:
        return ""


class OvsService(SdnService):
    name: str = "OvsService"
    group: str = "SDN"
    executables: Tuple[str, ...] = ("ovs-ofctl", "ovs-vsctl")
    dirs: Tuple[str, ...] = (
        "/etc/openvswitch",
        "/var/run/openvswitch",
        "/var/log/openvswitch",
    )
    configs: Tuple[str, ...] = ("OvsService.sh",)
    startup: Tuple[str, ...] = ("bash OvsService.sh",)
    shutdown: Tuple[str, ...] = ("killall ovs-vswitchd", "killall ovsdb-server")

    @classmethod
    def generate_config(cls, node: CoreNode, filename: str) -> str:
        # Check whether the node is running zebra
        has_zebra = 0
        for s in node.services:
            if s.name == "zebra":
                has_zebra = 1

        cfg = "#!/bin/sh\n"
        cfg += "# auto-generated by OvsService (OvsService.py)\n"
        cfg += "## First make sure that the ovs services are up and running\n"
        cfg += "/etc/init.d/openvswitch-switch start < /dev/null\n\n"
        cfg += "## create the switch itself, set the fail mode to secure, \n"
        cfg += "## this stops it from routing traffic without defined flows.\n"
        cfg += "## remove the -- and everything after if you want it to act as a regular switch\n"
        cfg += "ovs-vsctl add-br ovsbr0 -- set Bridge ovsbr0 fail-mode=secure\n"
        cfg += "\n## Now add all our interfaces as ports to the switch\n"

        portnum = 1
        for iface in node.get_ifaces(control=False):
            ifnumstr = re.findall(r"\d+", iface.name)
            ifnum = ifnumstr[0]

            # create virtual interfaces
            cfg += "## Create a veth pair to send the data to\n"
            cfg += "ip link add rtr%s type veth peer name sw%s\n" % (ifnum, ifnum)

            # remove ip address of eths because quagga/zebra will assign same IPs to rtr interfaces
            # or assign them manually to rtr interfaces if zebra is not running
            for ip4 in iface.ip4s:
                cfg += "ip addr del %s dev %s\n" % (ip4.ip, iface.name)
                if has_zebra == 0:
                    cfg += "ip addr add %s dev rtr%s\n" % (ip4.ip, ifnum)
            for ip6 in iface.ip6s:
                cfg += "ip -6 addr del %s dev %s\n" % (ip6.ip, iface.name)
                if has_zebra == 0:
                    cfg += "ip -6 addr add %s dev rtr%s\n" % (ip6.ip, ifnum)

            # add interfaces to bridge
            # Make port numbers explicit so they're easier to follow in reading the script
            cfg += "## Add the CORE interface to the switch\n"
            cfg += (
                "ovs-vsctl add-port ovsbr0 eth%s -- set Interface eth%s ofport_request=%d\n"
                % (ifnum, ifnum, portnum)
            )
            cfg += "## And then add its sibling veth interface\n"
            cfg += (
                "ovs-vsctl add-port ovsbr0 sw%s -- set Interface sw%s ofport_request=%d\n"
                % (ifnum, ifnum, portnum + 1)
            )
            cfg += "## start them up so we can send/receive data\n"
            cfg += "ovs-ofctl mod-port ovsbr0 eth%s up\n" % ifnum
            cfg += "ovs-ofctl mod-port ovsbr0 sw%s up\n" % ifnum
            cfg += "## Bring up the lower part of the veth pair\n"
            cfg += "ip link set dev rtr%s up\n" % ifnum
            portnum += 2

        # Add rule for default controller if there is one local (even if the controller is not local, it finds it)
        cfg += "\n## We assume there will be an SDN controller on the other end of this, \n"
        cfg += "## but it will still function if there's not\n"
        cfg += "ovs-vsctl set-controller ovsbr0 tcp:127.0.0.1:6633\n"

        cfg += "\n## Now to create some default flows, \n"
        cfg += "## if the above controller will be present then you probably want to delete them\n"
        # Setup default flows
        portnum = 1
        for iface in node.get_ifaces(control=False):
            cfg += "## Take the data from the CORE interface and put it on the veth and vice versa\n"
            cfg += (
                "ovs-ofctl add-flow ovsbr0 priority=1000,in_port=%d,action=output:%d\n"
                % (portnum, portnum + 1)
            )
            cfg += (
                "ovs-ofctl add-flow ovsbr0 priority=1000,in_port=%d,action=output:%d\n"
                % (portnum + 1, portnum)
            )
            portnum += 2
        return cfg


class RyuService(SdnService):
    name: str = "ryuService"
    group: str = "SDN"
    executables: Tuple[str, ...] = ("ryu-manager",)
    configs: Tuple[str, ...] = ("ryuService.sh",)
    startup: Tuple[str, ...] = ("bash ryuService.sh",)
    shutdown: Tuple[str, ...] = ("killall ryu-manager",)

    @classmethod
    def generate_config(cls, node: CoreNode, filename: str) -> str:
        """
        Return a string that will be written to filename, or sent to the
        GUI for user customization.
        """
        cfg = "#!/bin/sh\n"
        cfg += "# auto-generated by ryuService (ryuService.py)\n"
        cfg += (
            "ryu-manager --observe-links ryu.app.ofctl_rest ryu.app.rest_topology &\n"
        )
        return cfg
