use crate::network_map_snapshot::{CertPath, NetworkHostAndPort, NodeInfo, PartyAndCertificate};
use crate::types::*;
use oasis_amqp::{amqp, ser};

#[test]
fn encode() {
    let envelope = Envelope {
        obj: ObjectList(amqp::List::default()),
        schema: Schema {
            types: vec![TypeNotation::RestrictedType(RestrictedType {
                name: "java.util.List<java.lang.Object>",
                label: None,
                provides: amqp::List::default(),
                source: "list",
                descriptor: Descriptor {
                    name: Some("net.corda:1BLPJgNvsxdvPcbrIQd87g==".into()),
                    code: None,
                },
                choices: amqp::List::default(),
            })]
            .into(),
        },
        transforms_schema: None,
    };

    let mut body = vec![];
    body.extend_from_slice(CORDA_MAGIC);
    body.push(SectionId::DataAndStop as u8);
    ser::into_bytes(&envelope, &mut body).unwrap();
    assert_eq!(
        body,
        vec![
            99, 111, 114, 100, 97, 1, 0, 0, 0, 128, 197, 98, 0, 0, 0, 0, 0, 1, 208, 0, 0, 0, 213,
            0, 0, 0, 3, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 49, 66, 76, 80,
            74, 103, 78, 118, 115, 120, 100, 118, 80, 99, 98, 114, 73, 81, 100, 56, 55, 103, 61,
            61, 208, 0, 0, 0, 4, 0, 0, 0, 0, 0, 128, 197, 98, 0, 0, 0, 0, 0, 2, 208, 0, 0, 0, 147,
            0, 0, 0, 1, 208, 0, 0, 0, 138, 0, 0, 0, 1, 0, 128, 197, 98, 0, 0, 0, 0, 0, 6, 208, 0,
            0, 0, 119, 0, 0, 0, 6, 161, 32, 106, 97, 118, 97, 46, 117, 116, 105, 108, 46, 76, 105,
            115, 116, 60, 106, 97, 118, 97, 46, 108, 97, 110, 103, 46, 79, 98, 106, 101, 99, 116,
            62, 64, 208, 0, 0, 0, 4, 0, 0, 0, 0, 161, 4, 108, 105, 115, 116, 0, 128, 197, 98, 0, 0,
            0, 0, 0, 3, 208, 0, 0, 0, 41, 0, 0, 0, 2, 163, 34, 110, 101, 116, 46, 99, 111, 114,
            100, 97, 58, 49, 66, 76, 80, 74, 103, 78, 118, 115, 120, 100, 118, 80, 99, 98, 114, 73,
            81, 100, 56, 55, 103, 61, 61, 64, 208, 0, 0, 0, 4, 0, 0, 0, 0, 64
        ]
    );
}

#[test]
fn decode() {
    let body = [
        99, 111, 114, 100, 97, 1, 0, 0, 0, 128, 197, 98, 0, 0, 0, 0, 0, 1, 208, 0, 0, 35, 222, 0,
        0, 0, 3, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 101, 43, 113, 115, 87,
        47, 99, 74, 52, 97, 106, 71, 112, 98, 56, 89, 107, 74, 87, 66, 49, 65, 61, 61, 208, 0, 0,
        30, 158, 0, 0, 0, 1, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 49, 66, 76,
        80, 74, 103, 78, 118, 115, 120, 100, 118, 80, 99, 98, 114, 73, 81, 100, 56, 55, 103, 61,
        61, 208, 0, 0, 30, 112, 0, 0, 0, 3, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97,
        58, 110, 99, 85, 99, 90, 122, 118, 84, 57, 89, 71, 110, 48, 73, 116, 100, 111, 87, 87, 51,
        81, 81, 61, 61, 208, 0, 0, 10, 14, 0, 0, 0, 4, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114,
        100, 97, 58, 57, 120, 80, 84, 77, 65, 121, 103, 108, 49, 112, 71, 84, 65, 88, 82, 112, 71,
        86, 116, 105, 65, 61, 61, 192, 57, 1, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97,
        58, 73, 65, 43, 53, 100, 55, 43, 85, 118, 79, 54, 121, 116, 115, 54, 119, 68, 122, 114, 56,
        54, 81, 61, 61, 192, 17, 2, 161, 9, 108, 111, 99, 97, 108, 104, 111, 115, 116, 113, 0, 0,
        39, 21, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 97, 87, 101, 56, 106, 50,
        97, 65, 100, 110, 47, 75, 50, 49, 97, 50, 65, 51, 67, 75, 109, 65, 61, 61, 208, 0, 0, 9,
        117, 0, 0, 0, 1, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 71, 97, 80, 112,
        113, 47, 114, 76, 57, 75, 116, 102, 84, 79, 81, 68, 78, 57, 90, 67, 98, 65, 61, 61, 208, 0,
        0, 9, 71, 0, 0, 0, 1, 0, 163, 37, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 106, 97,
        118, 97, 46, 115, 101, 99, 117, 114, 105, 116, 121, 46, 99, 101, 114, 116, 46, 67, 101,
        114, 116, 80, 97, 116, 104, 208, 0, 0, 9, 22, 0, 0, 0, 2, 176, 0, 0, 9, 6, 48, 130, 9, 2,
        48, 130, 2, 9, 48, 130, 1, 176, 160, 3, 2, 1, 2, 2, 8, 112, 87, 180, 169, 203, 106, 74,
        231, 48, 10, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3,
        12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82, 111, 111, 116, 32, 67, 65,
        49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2, 82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99,
        111, 114, 100, 97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111, 110, 49,
        11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13, 49, 55, 48, 53, 50, 50, 48, 48,
        48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 88, 49,
        27, 48, 25, 6, 3, 85, 4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82,
        111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2, 82, 51, 49, 14, 48, 12,
        6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100, 97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76,
        111, 110, 100, 111, 110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 89, 48, 19, 6,
        7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 66, 0, 4, 105, 102,
        232, 177, 71, 173, 89, 51, 126, 229, 7, 138, 125, 250, 38, 185, 181, 105, 67, 26, 173, 127,
        205, 81, 149, 240, 242, 127, 32, 74, 34, 102, 177, 147, 229, 80, 74, 101, 39, 231, 255,
        151, 54, 76, 147, 46, 68, 100, 244, 237, 9, 200, 134, 223, 138, 103, 56, 253, 30, 136, 199,
        127, 37, 48, 163, 100, 48, 98, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 124, 174, 169, 223,
        185, 72, 1, 43, 19, 137, 11, 154, 230, 69, 133, 28, 57, 23, 7, 115, 48, 11, 6, 3, 85, 29,
        15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7,
        3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 15, 6, 3, 85, 29, 19, 1, 1,
        255, 4, 5, 48, 3, 1, 1, 255, 48, 10, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 3, 71, 0, 48, 68,
        2, 32, 13, 162, 248, 74, 11, 138, 176, 215, 147, 79, 180, 158, 82, 65, 93, 160, 32, 64, 9,
        225, 188, 26, 163, 184, 51, 82, 165, 126, 41, 169, 133, 12, 2, 32, 40, 170, 211, 72, 67,
        106, 116, 249, 91, 198, 211, 46, 161, 59, 24, 129, 174, 166, 87, 255, 59, 192, 0, 6, 90,
        86, 173, 72, 58, 186, 37, 1, 48, 130, 2, 94, 48, 130, 1, 251, 160, 3, 2, 1, 2, 2, 8, 29,
        86, 250, 193, 221, 209, 30, 18, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134,
        72, 206, 61, 3, 1, 7, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3, 12, 18, 67, 111, 114, 100,
        97, 32, 78, 111, 100, 101, 32, 82, 111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4,
        10, 12, 2, 82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100, 97, 49, 15,
        48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111, 110, 49, 11, 48, 9, 6, 3, 85, 4, 6,
        19, 2, 85, 75, 48, 30, 23, 13, 49, 56, 48, 55, 49, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13,
        50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 99, 49, 11, 48, 9, 6, 3, 85, 4, 6,
        19, 2, 85, 83, 49, 17, 48, 15, 6, 3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89, 111, 114, 107,
        49, 14, 48, 12, 6, 3, 85, 4, 11, 19, 5, 67, 111, 114, 100, 97, 49, 22, 48, 20, 6, 3, 85, 4,
        10, 19, 13, 82, 51, 32, 72, 111, 108, 100, 67, 111, 32, 76, 76, 67, 49, 25, 48, 23, 6, 3,
        85, 4, 3, 19, 16, 67, 111, 114, 100, 97, 32, 68, 111, 111, 114, 109, 97, 110, 32, 67, 65,
        48, 89, 48, 19, 6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3,
        66, 0, 4, 3, 203, 222, 160, 38, 225, 102, 102, 179, 151, 34, 5, 90, 49, 49, 5, 223, 43,
        187, 147, 152, 116, 85, 189, 103, 214, 67, 228, 21, 96, 253, 219, 48, 116, 160, 184, 118,
        81, 204, 29, 239, 236, 67, 132, 241, 22, 105, 24, 54, 167, 251, 13, 52, 237, 120, 251, 93,
        29, 62, 196, 111, 82, 198, 96, 163, 129, 153, 48, 129, 150, 48, 29, 6, 3, 85, 29, 14, 4,
        22, 4, 20, 235, 238, 46, 48, 21, 41, 64, 174, 25, 152, 30, 216, 111, 227, 125, 127, 7, 162,
        194, 19, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85, 29,
        15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7,
        3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24,
        48, 22, 128, 20, 124, 174, 169, 223, 185, 72, 1, 43, 19, 137, 11, 154, 230, 69, 133, 28,
        57, 23, 7, 115, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 1, 48, 20,
        6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68,
        2, 32, 25, 179, 65, 122, 103, 10, 143, 94, 3, 25, 33, 192, 203, 116, 108, 26, 245, 67, 64,
        15, 37, 113, 116, 42, 224, 81, 176, 80, 86, 1, 174, 146, 2, 32, 96, 199, 161, 188, 27, 210,
        35, 236, 31, 62, 203, 174, 72, 180, 178, 149, 114, 206, 202, 135, 164, 93, 158, 190, 127,
        215, 48, 127, 200, 252, 31, 174, 48, 130, 2, 155, 48, 130, 2, 55, 160, 3, 2, 1, 2, 2, 8,
        18, 34, 64, 127, 5, 156, 72, 141, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42,
        134, 72, 206, 61, 3, 1, 7, 48, 99, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 17,
        48, 15, 6, 3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89, 111, 114, 107, 49, 14, 48, 12, 6, 3,
        85, 4, 11, 19, 5, 67, 111, 114, 100, 97, 49, 22, 48, 20, 6, 3, 85, 4, 10, 19, 13, 82, 51,
        32, 72, 111, 108, 100, 67, 111, 32, 76, 76, 67, 49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 67,
        111, 114, 100, 97, 32, 68, 111, 111, 114, 109, 97, 110, 32, 67, 65, 48, 30, 23, 13, 49, 57,
        49, 48, 51, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48,
        48, 48, 90, 48, 62, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85,
        4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109, 49, 27, 48, 25, 6, 3, 85, 4, 10,
        12, 18, 86, 97, 108, 117, 101, 88, 32, 45, 32, 68, 105, 114, 101, 99, 116, 111, 114, 121,
        48, 89, 48, 19, 6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3,
        66, 0, 4, 169, 76, 198, 30, 14, 219, 148, 134, 190, 174, 10, 230, 192, 22, 132, 137, 188,
        173, 92, 210, 165, 52, 51, 66, 123, 6, 81, 187, 121, 51, 246, 96, 50, 35, 170, 165, 236,
        117, 214, 23, 227, 25, 35, 132, 40, 126, 168, 34, 238, 193, 150, 201, 172, 230, 0, 156,
        215, 52, 226, 97, 67, 240, 104, 219, 163, 129, 239, 48, 129, 236, 48, 29, 6, 3, 85, 29, 14,
        4, 22, 4, 20, 91, 158, 182, 50, 210, 58, 215, 60, 146, 130, 224, 89, 244, 248, 194, 220,
        81, 27, 114, 187, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6,
        3, 85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1,
        5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29,
        35, 4, 24, 48, 22, 128, 20, 235, 238, 46, 48, 21, 41, 64, 174, 25, 152, 30, 216, 111, 227,
        125, 127, 7, 162, 194, 19, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1,
        4, 48, 84, 6, 3, 85, 29, 30, 1, 1, 255, 4, 74, 48, 72, 160, 68, 48, 66, 164, 64, 48, 62,
        49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65,
        109, 115, 116, 101, 114, 100, 97, 109, 49, 27, 48, 25, 6, 3, 85, 4, 10, 12, 18, 86, 97,
        108, 117, 101, 88, 32, 45, 32, 68, 105, 114, 101, 99, 116, 111, 114, 121, 161, 0, 48, 20,
        6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 72, 0, 48, 69,
        2, 33, 0, 200, 120, 214, 1, 175, 235, 86, 246, 185, 52, 166, 113, 42, 191, 85, 120, 57,
        216, 16, 155, 143, 108, 164, 67, 222, 69, 80, 67, 192, 34, 1, 17, 2, 32, 118, 129, 25, 26,
        56, 103, 24, 170, 222, 237, 143, 34, 201, 182, 21, 142, 51, 170, 112, 151, 5, 161, 216,
        113, 207, 232, 245, 101, 146, 136, 91, 234, 48, 130, 1, 240, 48, 130, 1, 141, 160, 3, 2, 1,
        2, 2, 8, 74, 176, 187, 161, 30, 162, 152, 112, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2,
        6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 48, 62, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76,
        49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109, 49, 27,
        48, 25, 6, 3, 85, 4, 10, 12, 18, 86, 97, 108, 117, 101, 88, 32, 45, 32, 68, 105, 114, 101,
        99, 116, 111, 114, 121, 48, 30, 23, 13, 49, 57, 49, 48, 51, 48, 48, 48, 48, 48, 48, 48, 90,
        23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 62, 49, 11, 48, 9, 6, 3,
        85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101,
        114, 100, 97, 109, 49, 27, 48, 25, 6, 3, 85, 4, 10, 12, 18, 86, 97, 108, 117, 101, 88, 32,
        45, 32, 68, 105, 114, 101, 99, 116, 111, 114, 121, 48, 42, 48, 5, 6, 3, 43, 101, 112, 3,
        33, 0, 231, 105, 89, 66, 81, 164, 219, 134, 117, 107, 156, 156, 32, 81, 60, 106, 124, 23,
        254, 152, 154, 203, 163, 57, 168, 233, 72, 108, 206, 14, 197, 169, 163, 129, 153, 48, 129,
        150, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 49, 60, 64, 155, 27, 23, 226, 152, 129, 212,
        27, 106, 89, 240, 56, 249, 72, 155, 227, 1, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48,
        3, 1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 2, 132, 48, 35, 6, 3, 85, 29, 37, 4,
        28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37,
        0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 91, 158, 182, 50, 210, 58, 215, 60,
        146, 130, 224, 89, 244, 248, 194, 220, 81, 27, 114, 187, 48, 17, 6, 10, 43, 6, 1, 4, 1,
        131, 138, 98, 1, 1, 4, 3, 2, 1, 6, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42,
        134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68, 2, 32, 69, 239, 48, 104, 93, 212, 203, 178,
        187, 67, 63, 154, 210, 49, 85, 209, 76, 188, 21, 250, 253, 7, 96, 52, 170, 89, 36, 162, 78,
        223, 155, 47, 2, 32, 124, 251, 64, 23, 22, 28, 147, 141, 122, 57, 19, 88, 60, 178, 189, 44,
        180, 196, 80, 110, 116, 198, 103, 81, 244, 77, 56, 117, 245, 141, 34, 222, 161, 5, 88, 46,
        53, 48, 57, 84, 6, 129, 0, 0, 1, 112, 11, 30, 96, 48, 0, 163, 34, 110, 101, 116, 46, 99,
        111, 114, 100, 97, 58, 110, 99, 85, 99, 90, 122, 118, 84, 57, 89, 71, 110, 48, 73, 116,
        100, 111, 87, 87, 51, 81, 81, 61, 61, 208, 0, 0, 9, 223, 0, 0, 0, 4, 0, 163, 34, 110, 101,
        116, 46, 99, 111, 114, 100, 97, 58, 57, 120, 80, 84, 77, 65, 121, 103, 108, 49, 112, 71,
        84, 65, 88, 82, 112, 71, 86, 116, 105, 65, 61, 61, 192, 57, 1, 0, 163, 34, 110, 101, 116,
        46, 99, 111, 114, 100, 97, 58, 73, 65, 43, 53, 100, 55, 43, 85, 118, 79, 54, 121, 116, 115,
        54, 119, 68, 122, 114, 56, 54, 81, 61, 61, 192, 17, 2, 161, 9, 108, 111, 99, 97, 108, 104,
        111, 115, 116, 113, 0, 0, 39, 24, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58,
        97, 87, 101, 56, 106, 50, 97, 65, 100, 110, 47, 75, 50, 49, 97, 50, 65, 51, 67, 75, 109,
        65, 61, 61, 208, 0, 0, 9, 70, 0, 0, 0, 1, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100,
        97, 58, 71, 97, 80, 112, 113, 47, 114, 76, 57, 75, 116, 102, 84, 79, 81, 68, 78, 57, 90,
        67, 98, 65, 61, 61, 208, 0, 0, 9, 24, 0, 0, 0, 1, 0, 163, 37, 110, 101, 116, 46, 99, 111,
        114, 100, 97, 58, 106, 97, 118, 97, 46, 115, 101, 99, 117, 114, 105, 116, 121, 46, 99, 101,
        114, 116, 46, 67, 101, 114, 116, 80, 97, 116, 104, 208, 0, 0, 8, 231, 0, 0, 0, 2, 176, 0,
        0, 8, 215, 48, 130, 8, 211, 48, 130, 2, 9, 48, 130, 1, 176, 160, 3, 2, 1, 2, 2, 8, 112, 87,
        180, 169, 203, 106, 74, 231, 48, 10, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 48, 88, 49, 27,
        48, 25, 6, 3, 85, 4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82, 111,
        111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2, 82, 51, 49, 14, 48, 12, 6, 3,
        85, 4, 11, 12, 5, 99, 111, 114, 100, 97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111,
        110, 100, 111, 110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13, 49, 55,
        48, 53, 50, 50, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48,
        48, 48, 90, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78,
        111, 100, 101, 32, 82, 111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2,
        82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100, 97, 49, 15, 48, 13, 6,
        3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111, 110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85,
        75, 48, 89, 48, 19, 6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7,
        3, 66, 0, 4, 105, 102, 232, 177, 71, 173, 89, 51, 126, 229, 7, 138, 125, 250, 38, 185, 181,
        105, 67, 26, 173, 127, 205, 81, 149, 240, 242, 127, 32, 74, 34, 102, 177, 147, 229, 80, 74,
        101, 39, 231, 255, 151, 54, 76, 147, 46, 68, 100, 244, 237, 9, 200, 134, 223, 138, 103, 56,
        253, 30, 136, 199, 127, 37, 48, 163, 100, 48, 98, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20,
        124, 174, 169, 223, 185, 72, 1, 43, 19, 137, 11, 154, 230, 69, 133, 28, 57, 23, 7, 115, 48,
        11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8,
        43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 15, 6, 3,
        85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 10, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2,
        3, 71, 0, 48, 68, 2, 32, 13, 162, 248, 74, 11, 138, 176, 215, 147, 79, 180, 158, 82, 65,
        93, 160, 32, 64, 9, 225, 188, 26, 163, 184, 51, 82, 165, 126, 41, 169, 133, 12, 2, 32, 40,
        170, 211, 72, 67, 106, 116, 249, 91, 198, 211, 46, 161, 59, 24, 129, 174, 166, 87, 255, 59,
        192, 0, 6, 90, 86, 173, 72, 58, 186, 37, 1, 48, 130, 2, 94, 48, 130, 1, 251, 160, 3, 2, 1,
        2, 2, 8, 29, 86, 250, 193, 221, 209, 30, 18, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2,
        6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3, 12, 18, 67,
        111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82, 111, 111, 116, 32, 67, 65, 49, 11, 48, 9,
        6, 3, 85, 4, 10, 12, 2, 82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100,
        97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111, 110, 49, 11, 48, 9, 6,
        3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13, 49, 56, 48, 55, 49, 48, 48, 48, 48, 48, 48, 48,
        90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 99, 49, 11, 48, 9, 6,
        3, 85, 4, 6, 19, 2, 85, 83, 49, 17, 48, 15, 6, 3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89,
        111, 114, 107, 49, 14, 48, 12, 6, 3, 85, 4, 11, 19, 5, 67, 111, 114, 100, 97, 49, 22, 48,
        20, 6, 3, 85, 4, 10, 19, 13, 82, 51, 32, 72, 111, 108, 100, 67, 111, 32, 76, 76, 67, 49,
        25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 67, 111, 114, 100, 97, 32, 68, 111, 111, 114, 109, 97,
        110, 32, 67, 65, 48, 89, 48, 19, 6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206,
        61, 3, 1, 7, 3, 66, 0, 4, 3, 203, 222, 160, 38, 225, 102, 102, 179, 151, 34, 5, 90, 49, 49,
        5, 223, 43, 187, 147, 152, 116, 85, 189, 103, 214, 67, 228, 21, 96, 253, 219, 48, 116, 160,
        184, 118, 81, 204, 29, 239, 236, 67, 132, 241, 22, 105, 24, 54, 167, 251, 13, 52, 237, 120,
        251, 93, 29, 62, 196, 111, 82, 198, 96, 163, 129, 153, 48, 129, 150, 48, 29, 6, 3, 85, 29,
        14, 4, 22, 4, 20, 235, 238, 46, 48, 21, 41, 64, 174, 25, 152, 30, 216, 111, 227, 125, 127,
        7, 162, 194, 19, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3,
        85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5,
        5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35,
        4, 24, 48, 22, 128, 20, 124, 174, 169, 223, 185, 72, 1, 43, 19, 137, 11, 154, 230, 69, 133,
        28, 57, 23, 7, 115, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 1, 48,
        20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48,
        68, 2, 32, 25, 179, 65, 122, 103, 10, 143, 94, 3, 25, 33, 192, 203, 116, 108, 26, 245, 67,
        64, 15, 37, 113, 116, 42, 224, 81, 176, 80, 86, 1, 174, 146, 2, 32, 96, 199, 161, 188, 27,
        210, 35, 236, 31, 62, 203, 174, 72, 180, 178, 149, 114, 206, 202, 135, 164, 93, 158, 190,
        127, 215, 48, 127, 200, 252, 31, 174, 48, 130, 2, 132, 48, 130, 2, 31, 160, 3, 2, 1, 2, 2,
        8, 4, 193, 40, 158, 84, 33, 190, 222, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8,
        42, 134, 72, 206, 61, 3, 1, 7, 48, 99, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49,
        17, 48, 15, 6, 3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89, 111, 114, 107, 49, 14, 48, 12, 6,
        3, 85, 4, 11, 19, 5, 67, 111, 114, 100, 97, 49, 22, 48, 20, 6, 3, 85, 4, 10, 19, 13, 82,
        51, 32, 72, 111, 108, 100, 67, 111, 32, 76, 76, 67, 49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16,
        67, 111, 114, 100, 97, 32, 68, 111, 111, 114, 109, 97, 110, 32, 67, 65, 48, 30, 23, 13, 49,
        57, 49, 48, 51, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48,
        48, 48, 48, 90, 48, 50, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3,
        85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109, 49, 15, 48, 13, 6, 3, 85, 4,
        10, 12, 6, 83, 101, 108, 108, 101, 114, 48, 89, 48, 19, 6, 7, 42, 134, 72, 206, 61, 2, 1,
        6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 66, 0, 4, 67, 28, 133, 95, 43, 216, 205, 81, 61,
        140, 73, 20, 27, 161, 42, 13, 242, 109, 92, 2, 57, 79, 103, 168, 60, 193, 68, 7, 120, 101,
        55, 73, 161, 205, 172, 50, 26, 180, 137, 34, 223, 52, 236, 42, 175, 208, 38, 191, 26, 222,
        245, 179, 176, 182, 7, 158, 136, 104, 138, 252, 227, 133, 217, 92, 163, 129, 227, 48, 129,
        224, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 168, 88, 110, 18, 26, 134, 51, 194, 208, 119,
        70, 226, 234, 168, 129, 90, 220, 76, 235, 97, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5,
        48, 3, 1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37,
        4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29,
        37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 235, 238, 46, 48, 21, 41, 64, 174,
        25, 152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131,
        138, 98, 1, 1, 4, 3, 2, 1, 4, 48, 72, 6, 3, 85, 29, 30, 1, 1, 255, 4, 62, 48, 60, 160, 56,
        48, 54, 164, 52, 48, 50, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6,
        3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109, 49, 15, 48, 13, 6, 3, 85, 4,
        10, 12, 6, 83, 101, 108, 108, 101, 114, 161, 0, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3,
        2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 73, 0, 48, 70, 2, 33, 0, 216, 218, 93, 78, 59,
        31, 39, 40, 53, 73, 71, 131, 159, 143, 33, 206, 75, 10, 54, 221, 62, 96, 183, 127, 155,
        113, 115, 247, 193, 91, 57, 27, 2, 33, 0, 176, 50, 185, 180, 60, 225, 75, 111, 117, 80,
        118, 183, 84, 182, 69, 24, 251, 191, 207, 173, 219, 57, 150, 97, 171, 45, 134, 136, 125, 0,
        50, 96, 48, 130, 1, 216, 48, 130, 1, 117, 160, 3, 2, 1, 2, 2, 8, 82, 143, 125, 134, 40,
        127, 21, 57, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1,
        7, 48, 50, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7,
        12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109, 49, 15, 48, 13, 6, 3, 85, 4, 10, 12, 6,
        83, 101, 108, 108, 101, 114, 48, 30, 23, 13, 49, 57, 49, 48, 51, 48, 48, 48, 48, 48, 48,
        48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 50, 49, 11, 48, 9,
        6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116,
        101, 114, 100, 97, 109, 49, 15, 48, 13, 6, 3, 85, 4, 10, 12, 6, 83, 101, 108, 108, 101,
        114, 48, 42, 48, 5, 6, 3, 43, 101, 112, 3, 33, 0, 63, 120, 115, 97, 250, 206, 11, 217, 65,
        199, 136, 105, 13, 17, 36, 105, 206, 236, 173, 211, 22, 213, 174, 119, 165, 236, 203, 169,
        34, 140, 1, 178, 163, 129, 153, 48, 129, 150, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 153,
        126, 14, 158, 38, 185, 194, 178, 44, 167, 168, 64, 215, 179, 57, 178, 236, 168, 242, 241,
        48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4,
        4, 3, 2, 2, 132, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6,
        8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22,
        128, 20, 168, 88, 110, 18, 26, 134, 51, 194, 208, 119, 70, 226, 234, 168, 129, 90, 220, 76,
        235, 97, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 6, 48, 20, 6, 8,
        42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68, 2,
        32, 37, 7, 247, 198, 67, 28, 180, 5, 62, 115, 22, 223, 130, 171, 177, 27, 194, 184, 151,
        48, 27, 132, 243, 113, 33, 5, 2, 20, 126, 147, 11, 175, 2, 32, 61, 228, 82, 18, 136, 29,
        221, 50, 48, 32, 119, 14, 197, 91, 208, 33, 6, 243, 123, 231, 65, 22, 17, 29, 205, 107,
        124, 123, 219, 37, 197, 155, 161, 5, 88, 46, 53, 48, 57, 84, 4, 129, 0, 0, 1, 110, 28, 150,
        180, 64, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 110, 99, 85, 99, 90,
        122, 118, 84, 57, 89, 71, 110, 48, 73, 116, 100, 111, 87, 87, 51, 81, 81, 61, 61, 208, 0,
        0, 10, 1, 0, 0, 0, 4, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 57, 120,
        80, 84, 77, 65, 121, 103, 108, 49, 112, 71, 84, 65, 88, 82, 112, 71, 86, 116, 105, 65, 61,
        61, 192, 57, 1, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 73, 65, 43, 53,
        100, 55, 43, 85, 118, 79, 54, 121, 116, 115, 54, 119, 68, 122, 114, 56, 54, 81, 61, 61,
        192, 17, 2, 161, 9, 108, 111, 99, 97, 108, 104, 111, 115, 116, 113, 0, 0, 39, 18, 0, 163,
        34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 97, 87, 101, 56, 106, 50, 97, 65, 100,
        110, 47, 75, 50, 49, 97, 50, 65, 51, 67, 75, 109, 65, 61, 61, 208, 0, 0, 9, 104, 0, 0, 0,
        1, 0, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 71, 97, 80, 112, 113, 47, 114,
        76, 57, 75, 116, 102, 84, 79, 81, 68, 78, 57, 90, 67, 98, 65, 61, 61, 208, 0, 0, 9, 58, 0,
        0, 0, 1, 0, 163, 37, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 106, 97, 118, 97, 46,
        115, 101, 99, 117, 114, 105, 116, 121, 46, 99, 101, 114, 116, 46, 67, 101, 114, 116, 80,
        97, 116, 104, 208, 0, 0, 9, 9, 0, 0, 0, 2, 176, 0, 0, 8, 249, 48, 130, 8, 245, 48, 130, 2,
        9, 48, 130, 1, 176, 160, 3, 2, 1, 2, 2, 8, 112, 87, 180, 169, 203, 106, 74, 231, 48, 10, 6,
        8, 42, 134, 72, 206, 61, 4, 3, 2, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3, 12, 18, 67, 111,
        114, 100, 97, 32, 78, 111, 100, 101, 32, 82, 111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6,
        3, 85, 4, 10, 12, 2, 82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100, 97,
        49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111, 110, 49, 11, 48, 9, 6, 3,
        85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13, 49, 55, 48, 53, 50, 50, 48, 48, 48, 48, 48, 48,
        90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 88, 49, 27, 48, 25, 6,
        3, 85, 4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82, 111, 111, 116,
        32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2, 82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11,
        12, 5, 99, 111, 114, 100, 97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100,
        111, 110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 89, 48, 19, 6, 7, 42, 134, 72,
        206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 66, 0, 4, 105, 102, 232, 177, 71,
        173, 89, 51, 126, 229, 7, 138, 125, 250, 38, 185, 181, 105, 67, 26, 173, 127, 205, 81, 149,
        240, 242, 127, 32, 74, 34, 102, 177, 147, 229, 80, 74, 101, 39, 231, 255, 151, 54, 76, 147,
        46, 68, 100, 244, 237, 9, 200, 134, 223, 138, 103, 56, 253, 30, 136, 199, 127, 37, 48, 163,
        100, 48, 98, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 124, 174, 169, 223, 185, 72, 1, 43,
        19, 137, 11, 154, 230, 69, 133, 28, 57, 23, 7, 115, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2,
        1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43,
        6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3,
        1, 1, 255, 48, 10, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 3, 71, 0, 48, 68, 2, 32, 13, 162,
        248, 74, 11, 138, 176, 215, 147, 79, 180, 158, 82, 65, 93, 160, 32, 64, 9, 225, 188, 26,
        163, 184, 51, 82, 165, 126, 41, 169, 133, 12, 2, 32, 40, 170, 211, 72, 67, 106, 116, 249,
        91, 198, 211, 46, 161, 59, 24, 129, 174, 166, 87, 255, 59, 192, 0, 6, 90, 86, 173, 72, 58,
        186, 37, 1, 48, 130, 2, 94, 48, 130, 1, 251, 160, 3, 2, 1, 2, 2, 8, 29, 86, 250, 193, 221,
        209, 30, 18, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1,
        7, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100,
        101, 32, 82, 111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2, 82, 51, 49,
        14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100, 97, 49, 15, 48, 13, 6, 3, 85, 4, 7,
        12, 6, 76, 111, 110, 100, 111, 110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30,
        23, 13, 49, 56, 48, 55, 49, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48,
        48, 48, 48, 48, 48, 48, 90, 48, 99, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 17,
        48, 15, 6, 3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89, 111, 114, 107, 49, 14, 48, 12, 6, 3,
        85, 4, 11, 19, 5, 67, 111, 114, 100, 97, 49, 22, 48, 20, 6, 3, 85, 4, 10, 19, 13, 82, 51,
        32, 72, 111, 108, 100, 67, 111, 32, 76, 76, 67, 49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 67,
        111, 114, 100, 97, 32, 68, 111, 111, 114, 109, 97, 110, 32, 67, 65, 48, 89, 48, 19, 6, 7,
        42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 66, 0, 4, 3, 203, 222,
        160, 38, 225, 102, 102, 179, 151, 34, 5, 90, 49, 49, 5, 223, 43, 187, 147, 152, 116, 85,
        189, 103, 214, 67, 228, 21, 96, 253, 219, 48, 116, 160, 184, 118, 81, 204, 29, 239, 236,
        67, 132, 241, 22, 105, 24, 54, 167, 251, 13, 52, 237, 120, 251, 93, 29, 62, 196, 111, 82,
        198, 96, 163, 129, 153, 48, 129, 150, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 235, 238, 46,
        48, 21, 41, 64, 174, 25, 152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19, 48, 15, 6, 3,
        85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1,
        134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6,
        1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20,
        124, 174, 169, 223, 185, 72, 1, 43, 19, 137, 11, 154, 230, 69, 133, 28, 57, 23, 7, 115, 48,
        17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 1, 48, 20, 6, 8, 42, 134, 72,
        206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68, 2, 32, 25, 179,
        65, 122, 103, 10, 143, 94, 3, 25, 33, 192, 203, 116, 108, 26, 245, 67, 64, 15, 37, 113,
        116, 42, 224, 81, 176, 80, 86, 1, 174, 146, 2, 32, 96, 199, 161, 188, 27, 210, 35, 236, 31,
        62, 203, 174, 72, 180, 178, 149, 114, 206, 202, 135, 164, 93, 158, 190, 127, 215, 48, 127,
        200, 252, 31, 174, 48, 130, 2, 148, 48, 130, 2, 49, 160, 3, 2, 1, 2, 2, 8, 35, 216, 181,
        181, 84, 142, 206, 251, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72,
        206, 61, 3, 1, 7, 48, 99, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 17, 48, 15, 6,
        3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89, 111, 114, 107, 49, 14, 48, 12, 6, 3, 85, 4, 11,
        19, 5, 67, 111, 114, 100, 97, 49, 22, 48, 20, 6, 3, 85, 4, 10, 19, 13, 82, 51, 32, 72, 111,
        108, 100, 67, 111, 32, 76, 76, 67, 49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 67, 111, 114,
        100, 97, 32, 68, 111, 111, 114, 109, 97, 110, 32, 67, 65, 48, 30, 23, 13, 49, 57, 49, 48,
        51, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48,
        90, 48, 59, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7,
        12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109, 49, 24, 48, 22, 6, 3, 85, 4, 10, 12, 15,
        86, 97, 108, 117, 101, 88, 32, 45, 32, 78, 111, 116, 97, 114, 121, 48, 89, 48, 19, 6, 7,
        42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 66, 0, 4, 231, 255, 44,
        3, 246, 17, 25, 253, 225, 250, 76, 83, 224, 54, 188, 212, 183, 179, 10, 83, 146, 166, 111,
        105, 115, 42, 179, 100, 173, 205, 212, 223, 28, 6, 83, 83, 184, 34, 29, 172, 153, 219, 36,
        144, 234, 124, 96, 232, 41, 28, 236, 138, 242, 103, 171, 3, 248, 215, 52, 35, 181, 237,
        232, 184, 163, 129, 236, 48, 129, 233, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 154, 131,
        149, 237, 125, 55, 168, 209, 149, 14, 185, 246, 181, 201, 230, 2, 40, 38, 15, 73, 48, 15,
        6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2,
        1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43,
        6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20,
        235, 238, 46, 48, 21, 41, 64, 174, 25, 152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19,
        48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 4, 48, 81, 6, 3, 85, 29, 30,
        1, 1, 255, 4, 71, 48, 69, 160, 65, 48, 63, 164, 61, 48, 59, 49, 11, 48, 9, 6, 3, 85, 4, 6,
        19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97,
        109, 49, 24, 48, 22, 6, 3, 85, 4, 10, 12, 15, 86, 97, 108, 117, 101, 88, 32, 45, 32, 78,
        111, 116, 97, 114, 121, 161, 0, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134,
        72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68, 2, 32, 120, 163, 128, 129, 110, 187, 254, 251, 179,
        57, 18, 144, 45, 225, 13, 132, 218, 8, 143, 168, 17, 131, 50, 192, 27, 64, 190, 205, 154,
        72, 229, 20, 2, 32, 33, 105, 226, 98, 42, 40, 79, 254, 154, 71, 62, 21, 206, 234, 161, 20,
        217, 2, 236, 235, 199, 34, 151, 230, 255, 2, 167, 122, 77, 219, 150, 115, 48, 130, 1, 234,
        48, 130, 1, 135, 160, 3, 2, 1, 2, 2, 8, 77, 235, 139, 63, 191, 57, 108, 135, 48, 20, 6, 8,
        42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 48, 59, 49, 11, 48, 9,
        6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116,
        101, 114, 100, 97, 109, 49, 24, 48, 22, 6, 3, 85, 4, 10, 12, 15, 86, 97, 108, 117, 101, 88,
        32, 45, 32, 78, 111, 116, 97, 114, 121, 48, 30, 23, 13, 49, 57, 49, 48, 51, 48, 48, 48, 48,
        48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 59, 49, 11,
        48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115,
        116, 101, 114, 100, 97, 109, 49, 24, 48, 22, 6, 3, 85, 4, 10, 12, 15, 86, 97, 108, 117,
        101, 88, 32, 45, 32, 78, 111, 116, 97, 114, 121, 48, 42, 48, 5, 6, 3, 43, 101, 112, 3, 33,
        0, 186, 172, 105, 246, 26, 136, 217, 8, 165, 28, 237, 74, 5, 108, 99, 126, 46, 184, 206,
        45, 77, 5, 248, 236, 37, 186, 77, 242, 235, 201, 139, 37, 163, 129, 153, 48, 129, 150, 48,
        29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 132, 146, 72, 145, 44, 30, 244, 165, 107, 6, 10, 219,
        114, 94, 93, 122, 138, 113, 215, 0, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1,
        255, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 2, 132, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26,
        6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31,
        6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 154, 131, 149, 237, 125, 55, 168, 209, 149, 14,
        185, 246, 181, 201, 230, 2, 40, 38, 15, 73, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1,
        1, 4, 3, 2, 1, 6, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61,
        3, 1, 7, 3, 71, 0, 48, 68, 2, 32, 10, 230, 172, 239, 255, 76, 46, 99, 96, 224, 37, 175,
        105, 149, 53, 255, 164, 58, 49, 119, 118, 3, 228, 72, 161, 94, 115, 176, 253, 111, 191,
        215, 2, 32, 32, 93, 78, 7, 228, 110, 114, 137, 149, 62, 198, 178, 124, 71, 128, 51, 184,
        163, 186, 225, 98, 2, 20, 69, 133, 47, 81, 34, 41, 63, 102, 111, 161, 5, 88, 46, 53, 48,
        57, 84, 4, 129, 0, 0, 1, 110, 28, 150, 173, 156, 0, 128, 197, 98, 0, 0, 0, 0, 0, 2, 208, 0,
        0, 4, 246, 0, 0, 0, 1, 208, 0, 0, 4, 237, 0, 0, 0, 7, 0, 128, 197, 98, 0, 0, 0, 0, 0, 5,
        192, 137, 5, 161, 36, 110, 101, 116, 46, 99, 111, 114, 100, 97, 46, 99, 111, 114, 101, 46,
        117, 116, 105, 108, 105, 116, 105, 101, 115, 46, 84, 114, 121, 36, 83, 117, 99, 99, 101,
        115, 115, 64, 69, 0, 128, 197, 98, 0, 0, 0, 0, 0, 3, 192, 38, 2, 163, 34, 110, 101, 116,
        46, 99, 111, 114, 100, 97, 58, 101, 43, 113, 115, 87, 47, 99, 74, 52, 97, 106, 71, 112, 98,
        56, 89, 107, 74, 87, 66, 49, 65, 61, 61, 64, 192, 44, 1, 0, 128, 197, 98, 0, 0, 0, 0, 0, 4,
        192, 31, 7, 161, 5, 118, 97, 108, 117, 101, 161, 16, 106, 97, 118, 97, 46, 108, 97, 110,
        103, 46, 79, 98, 106, 101, 99, 116, 69, 64, 64, 65, 66, 0, 128, 197, 98, 0, 0, 0, 0, 0, 6,
        192, 94, 6, 161, 32, 106, 97, 118, 97, 46, 117, 116, 105, 108, 46, 76, 105, 115, 116, 60,
        106, 97, 118, 97, 46, 108, 97, 110, 103, 46, 79, 98, 106, 101, 99, 116, 62, 64, 69, 161, 4,
        108, 105, 115, 116, 0, 128, 197, 98, 0, 0, 0, 0, 0, 3, 192, 38, 2, 163, 34, 110, 101, 116,
        46, 99, 111, 114, 100, 97, 58, 49, 66, 76, 80, 74, 103, 78, 118, 115, 120, 100, 118, 80,
        99, 98, 114, 73, 81, 100, 56, 55, 103, 61, 61, 64, 69, 0, 128, 197, 98, 0, 0, 0, 0, 0, 5,
        208, 0, 0, 1, 119, 0, 0, 0, 5, 161, 28, 110, 101, 116, 46, 99, 111, 114, 100, 97, 46, 99,
        111, 114, 101, 46, 110, 111, 100, 101, 46, 78, 111, 100, 101, 73, 110, 102, 111, 64, 69, 0,
        128, 197, 98, 0, 0, 0, 0, 0, 3, 192, 38, 2, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100,
        97, 58, 110, 99, 85, 99, 90, 122, 118, 84, 57, 89, 71, 110, 48, 73, 116, 100, 111, 87, 87,
        51, 81, 81, 61, 61, 64, 208, 0, 0, 1, 28, 0, 0, 0, 4, 0, 128, 197, 98, 0, 0, 0, 0, 0, 4,
        192, 83, 7, 161, 9, 97, 100, 100, 114, 101, 115, 115, 101, 115, 161, 1, 42, 192, 62, 1,
        161, 59, 106, 97, 118, 97, 46, 117, 116, 105, 108, 46, 76, 105, 115, 116, 60, 110, 101,
        116, 46, 99, 111, 114, 100, 97, 46, 99, 111, 114, 101, 46, 117, 116, 105, 108, 105, 116,
        105, 101, 115, 46, 78, 101, 116, 119, 111, 114, 107, 72, 111, 115, 116, 65, 110, 100, 80,
        111, 114, 116, 62, 64, 64, 65, 66, 0, 128, 197, 98, 0, 0, 0, 0, 0, 4, 192, 97, 7, 161, 23,
        108, 101, 103, 97, 108, 73, 100, 101, 110, 116, 105, 116, 105, 101, 115, 65, 110, 100, 67,
        101, 114, 116, 115, 161, 1, 42, 192, 62, 1, 161, 59, 106, 97, 118, 97, 46, 117, 116, 105,
        108, 46, 76, 105, 115, 116, 60, 110, 101, 116, 46, 99, 111, 114, 100, 97, 46, 99, 111, 114,
        101, 46, 105, 100, 101, 110, 116, 105, 116, 121, 46, 80, 97, 114, 116, 121, 65, 110, 100,
        67, 101, 114, 116, 105, 102, 105, 99, 97, 116, 101, 62, 64, 64, 65, 66, 0, 128, 197, 98, 0,
        0, 0, 0, 0, 4, 192, 30, 7, 161, 15, 112, 108, 97, 116, 102, 111, 114, 109, 86, 101, 114,
        115, 105, 111, 110, 161, 3, 105, 110, 116, 69, 161, 1, 48, 64, 65, 66, 0, 128, 197, 98, 0,
        0, 0, 0, 0, 4, 192, 22, 7, 161, 6, 115, 101, 114, 105, 97, 108, 161, 4, 108, 111, 110, 103,
        69, 161, 1, 48, 64, 65, 66, 0, 128, 197, 98, 0, 0, 0, 0, 0, 6, 192, 121, 6, 161, 59, 106,
        97, 118, 97, 46, 117, 116, 105, 108, 46, 76, 105, 115, 116, 60, 110, 101, 116, 46, 99, 111,
        114, 100, 97, 46, 99, 111, 114, 101, 46, 117, 116, 105, 108, 105, 116, 105, 101, 115, 46,
        78, 101, 116, 119, 111, 114, 107, 72, 111, 115, 116, 65, 110, 100, 80, 111, 114, 116, 62,
        64, 69, 161, 4, 108, 105, 115, 116, 0, 128, 197, 98, 0, 0, 0, 0, 0, 3, 192, 38, 2, 163, 34,
        110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 57, 120, 80, 84, 77, 65, 121, 103, 108, 49,
        112, 71, 84, 65, 88, 82, 112, 71, 86, 116, 105, 65, 61, 61, 64, 69, 0, 128, 197, 98, 0, 0,
        0, 0, 0, 5, 192, 164, 5, 161, 43, 110, 101, 116, 46, 99, 111, 114, 100, 97, 46, 99, 111,
        114, 101, 46, 117, 116, 105, 108, 105, 116, 105, 101, 115, 46, 78, 101, 116, 119, 111, 114,
        107, 72, 111, 115, 116, 65, 110, 100, 80, 111, 114, 116, 64, 69, 0, 128, 197, 98, 0, 0, 0,
        0, 0, 3, 192, 38, 2, 163, 34, 110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 73, 65, 43, 53,
        100, 55, 43, 85, 118, 79, 54, 121, 116, 115, 54, 119, 68, 122, 114, 56, 54, 81, 61, 61, 64,
        192, 64, 2, 0, 128, 197, 98, 0, 0, 0, 0, 0, 4, 192, 20, 7, 161, 4, 104, 111, 115, 116, 161,
        6, 115, 116, 114, 105, 110, 103, 69, 64, 64, 65, 66, 0, 128, 197, 98, 0, 0, 0, 0, 0, 4,
        192, 19, 7, 161, 4, 112, 111, 114, 116, 161, 3, 105, 110, 116, 69, 161, 1, 48, 64, 65, 66,
        0, 128, 197, 98, 0, 0, 0, 0, 0, 6, 192, 121, 6, 161, 59, 106, 97, 118, 97, 46, 117, 116,
        105, 108, 46, 76, 105, 115, 116, 60, 110, 101, 116, 46, 99, 111, 114, 100, 97, 46, 99, 111,
        114, 101, 46, 105, 100, 101, 110, 116, 105, 116, 121, 46, 80, 97, 114, 116, 121, 65, 110,
        100, 67, 101, 114, 116, 105, 102, 105, 99, 97, 116, 101, 62, 64, 69, 161, 4, 108, 105, 115,
        116, 0, 128, 197, 98, 0, 0, 0, 0, 0, 3, 192, 38, 2, 163, 34, 110, 101, 116, 46, 99, 111,
        114, 100, 97, 58, 97, 87, 101, 56, 106, 50, 97, 65, 100, 110, 47, 75, 50, 49, 97, 50, 65,
        51, 67, 75, 109, 65, 61, 61, 64, 69, 0, 128, 197, 98, 0, 0, 0, 0, 0, 5, 192, 158, 5, 161,
        43, 110, 101, 116, 46, 99, 111, 114, 100, 97, 46, 99, 111, 114, 101, 46, 105, 100, 101,
        110, 116, 105, 116, 121, 46, 80, 97, 114, 116, 121, 65, 110, 100, 67, 101, 114, 116, 105,
        102, 105, 99, 97, 116, 101, 64, 69, 0, 128, 197, 98, 0, 0, 0, 0, 0, 3, 192, 38, 2, 163, 34,
        110, 101, 116, 46, 99, 111, 114, 100, 97, 58, 71, 97, 80, 112, 113, 47, 114, 76, 57, 75,
        116, 102, 84, 79, 81, 68, 78, 57, 90, 67, 98, 65, 61, 61, 64, 192, 58, 1, 0, 128, 197, 98,
        0, 0, 0, 0, 0, 4, 192, 45, 7, 161, 8, 99, 101, 114, 116, 80, 97, 116, 104, 161, 27, 106,
        97, 118, 97, 46, 115, 101, 99, 117, 114, 105, 116, 121, 46, 99, 101, 114, 116, 46, 67, 101,
        114, 116, 80, 97, 116, 104, 69, 64, 64, 65, 66, 0, 128, 197, 98, 0, 0, 0, 0, 0, 9, 193, 1,
        0,
    ];

    let rsp = Envelope::<Try<amqp::List<NodeInfo>, ()>>::decode(&body).unwrap();
    let expected = Envelope {
        obj: Try::Success(Success {
            value: amqp::List(vec![
                NodeInfo {
                    addresses: amqp::List(vec![NetworkHostAndPort {
                        host: "localhost",
                        port: 10005,
                    }]),
                    legal_identities_and_certs: amqp::List(vec![PartyAndCertificate {
                        cert_path: CertPath {
                            data: &[
                                48, 130, 9, 2, 48, 130, 2, 9, 48, 130, 1, 176, 160, 3, 2, 1, 2, 2,
                                8, 112, 87, 180, 169, 203, 106, 74, 231, 48, 10, 6, 8, 42, 134, 72,
                                206, 61, 4, 3, 2, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3, 12, 18,
                                67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82, 111, 111,
                                116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2, 82, 51, 49,
                                14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100, 97, 49, 15,
                                48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111, 110, 49, 11,
                                48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13, 49, 55, 48,
                                53, 50, 50, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50,
                                48, 48, 48, 48, 48, 48, 48, 90, 48, 88, 49, 27, 48, 25, 6, 3, 85,
                                4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82,
                                111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2,
                                82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100,
                                97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111,
                                110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 89, 48, 19,
                                6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1,
                                7, 3, 66, 0, 4, 105, 102, 232, 177, 71, 173, 89, 51, 126, 229, 7,
                                138, 125, 250, 38, 185, 181, 105, 67, 26, 173, 127, 205, 81, 149,
                                240, 242, 127, 32, 74, 34, 102, 177, 147, 229, 80, 74, 101, 39,
                                231, 255, 151, 54, 76, 147, 46, 68, 100, 244, 237, 9, 200, 134,
                                223, 138, 103, 56, 253, 30, 136, 199, 127, 37, 48, 163, 100, 48,
                                98, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 124, 174, 169, 223,
                                185, 72, 1, 43, 19, 137, 11, 154, 230, 69, 133, 28, 57, 23, 7, 115,
                                48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29,
                                37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1,
                                5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 15, 6, 3, 85, 29, 19, 1, 1,
                                255, 4, 5, 48, 3, 1, 1, 255, 48, 10, 6, 8, 42, 134, 72, 206, 61, 4,
                                3, 2, 3, 71, 0, 48, 68, 2, 32, 13, 162, 248, 74, 11, 138, 176, 215,
                                147, 79, 180, 158, 82, 65, 93, 160, 32, 64, 9, 225, 188, 26, 163,
                                184, 51, 82, 165, 126, 41, 169, 133, 12, 2, 32, 40, 170, 211, 72,
                                67, 106, 116, 249, 91, 198, 211, 46, 161, 59, 24, 129, 174, 166,
                                87, 255, 59, 192, 0, 6, 90, 86, 173, 72, 58, 186, 37, 1, 48, 130,
                                2, 94, 48, 130, 1, 251, 160, 3, 2, 1, 2, 2, 8, 29, 86, 250, 193,
                                221, 209, 30, 18, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6,
                                8, 42, 134, 72, 206, 61, 3, 1, 7, 48, 88, 49, 27, 48, 25, 6, 3, 85,
                                4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82,
                                111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2,
                                82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100,
                                97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111,
                                110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13,
                                49, 56, 48, 55, 49, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55,
                                48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 99, 49, 11, 48, 9,
                                6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 17, 48, 15, 6, 3, 85, 4, 7, 19,
                                8, 78, 101, 119, 32, 89, 111, 114, 107, 49, 14, 48, 12, 6, 3, 85,
                                4, 11, 19, 5, 67, 111, 114, 100, 97, 49, 22, 48, 20, 6, 3, 85, 4,
                                10, 19, 13, 82, 51, 32, 72, 111, 108, 100, 67, 111, 32, 76, 76, 67,
                                49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 67, 111, 114, 100, 97, 32,
                                68, 111, 111, 114, 109, 97, 110, 32, 67, 65, 48, 89, 48, 19, 6, 7,
                                42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3,
                                66, 0, 4, 3, 203, 222, 160, 38, 225, 102, 102, 179, 151, 34, 5, 90,
                                49, 49, 5, 223, 43, 187, 147, 152, 116, 85, 189, 103, 214, 67, 228,
                                21, 96, 253, 219, 48, 116, 160, 184, 118, 81, 204, 29, 239, 236,
                                67, 132, 241, 22, 105, 24, 54, 167, 251, 13, 52, 237, 120, 251, 93,
                                29, 62, 196, 111, 82, 198, 96, 163, 129, 153, 48, 129, 150, 48, 29,
                                6, 3, 85, 29, 14, 4, 22, 4, 20, 235, 238, 46, 48, 21, 41, 64, 174,
                                25, 152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19, 48, 15, 6,
                                3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85,
                                29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48,
                                26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2,
                                6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128,
                                20, 124, 174, 169, 223, 185, 72, 1, 43, 19, 137, 11, 154, 230, 69,
                                133, 28, 57, 23, 7, 115, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138,
                                98, 1, 1, 4, 3, 2, 1, 1, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3,
                                2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68, 2, 32,
                                25, 179, 65, 122, 103, 10, 143, 94, 3, 25, 33, 192, 203, 116, 108,
                                26, 245, 67, 64, 15, 37, 113, 116, 42, 224, 81, 176, 80, 86, 1,
                                174, 146, 2, 32, 96, 199, 161, 188, 27, 210, 35, 236, 31, 62, 203,
                                174, 72, 180, 178, 149, 114, 206, 202, 135, 164, 93, 158, 190, 127,
                                215, 48, 127, 200, 252, 31, 174, 48, 130, 2, 155, 48, 130, 2, 55,
                                160, 3, 2, 1, 2, 2, 8, 18, 34, 64, 127, 5, 156, 72, 141, 48, 20, 6,
                                8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1,
                                7, 48, 99, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 17,
                                48, 15, 6, 3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89, 111, 114, 107,
                                49, 14, 48, 12, 6, 3, 85, 4, 11, 19, 5, 67, 111, 114, 100, 97, 49,
                                22, 48, 20, 6, 3, 85, 4, 10, 19, 13, 82, 51, 32, 72, 111, 108, 100,
                                67, 111, 32, 76, 76, 67, 49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16,
                                67, 111, 114, 100, 97, 32, 68, 111, 111, 114, 109, 97, 110, 32, 67,
                                65, 48, 30, 23, 13, 49, 57, 49, 48, 51, 48, 48, 48, 48, 48, 48, 48,
                                90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48,
                                62, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16,
                                6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109,
                                49, 27, 48, 25, 6, 3, 85, 4, 10, 12, 18, 86, 97, 108, 117, 101, 88,
                                32, 45, 32, 68, 105, 114, 101, 99, 116, 111, 114, 121, 48, 89, 48,
                                19, 6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61,
                                3, 1, 7, 3, 66, 0, 4, 169, 76, 198, 30, 14, 219, 148, 134, 190,
                                174, 10, 230, 192, 22, 132, 137, 188, 173, 92, 210, 165, 52, 51,
                                66, 123, 6, 81, 187, 121, 51, 246, 96, 50, 35, 170, 165, 236, 117,
                                214, 23, 227, 25, 35, 132, 40, 126, 168, 34, 238, 193, 150, 201,
                                172, 230, 0, 156, 215, 52, 226, 97, 67, 240, 104, 219, 163, 129,
                                239, 48, 129, 236, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 91, 158,
                                182, 50, 210, 58, 215, 60, 146, 130, 224, 89, 244, 248, 194, 220,
                                81, 27, 114, 187, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3,
                                1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6,
                                3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8,
                                43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29,
                                35, 4, 24, 48, 22, 128, 20, 235, 238, 46, 48, 21, 41, 64, 174, 25,
                                152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19, 48, 17, 6, 10,
                                43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 4, 48, 84, 6, 3,
                                85, 29, 30, 1, 1, 255, 4, 74, 48, 72, 160, 68, 48, 66, 164, 64, 48,
                                62, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16,
                                6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109,
                                49, 27, 48, 25, 6, 3, 85, 4, 10, 12, 18, 86, 97, 108, 117, 101, 88,
                                32, 45, 32, 68, 105, 114, 101, 99, 116, 111, 114, 121, 161, 0, 48,
                                20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206,
                                61, 3, 1, 7, 3, 72, 0, 48, 69, 2, 33, 0, 200, 120, 214, 1, 175,
                                235, 86, 246, 185, 52, 166, 113, 42, 191, 85, 120, 57, 216, 16,
                                155, 143, 108, 164, 67, 222, 69, 80, 67, 192, 34, 1, 17, 2, 32,
                                118, 129, 25, 26, 56, 103, 24, 170, 222, 237, 143, 34, 201, 182,
                                21, 142, 51, 170, 112, 151, 5, 161, 216, 113, 207, 232, 245, 101,
                                146, 136, 91, 234, 48, 130, 1, 240, 48, 130, 1, 141, 160, 3, 2, 1,
                                2, 2, 8, 74, 176, 187, 161, 30, 162, 152, 112, 48, 20, 6, 8, 42,
                                134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 48,
                                62, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16,
                                6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109,
                                49, 27, 48, 25, 6, 3, 85, 4, 10, 12, 18, 86, 97, 108, 117, 101, 88,
                                32, 45, 32, 68, 105, 114, 101, 99, 116, 111, 114, 121, 48, 30, 23,
                                13, 49, 57, 49, 48, 51, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50,
                                55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 62, 49, 11, 48,
                                9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7,
                                12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109, 49, 27, 48, 25,
                                6, 3, 85, 4, 10, 12, 18, 86, 97, 108, 117, 101, 88, 32, 45, 32, 68,
                                105, 114, 101, 99, 116, 111, 114, 121, 48, 42, 48, 5, 6, 3, 43,
                                101, 112, 3, 33, 0, 231, 105, 89, 66, 81, 164, 219, 134, 117, 107,
                                156, 156, 32, 81, 60, 106, 124, 23, 254, 152, 154, 203, 163, 57,
                                168, 233, 72, 108, 206, 14, 197, 169, 163, 129, 153, 48, 129, 150,
                                48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 49, 60, 64, 155, 27, 23,
                                226, 152, 129, 212, 27, 106, 89, 240, 56, 249, 72, 155, 227, 1, 48,
                                15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6,
                                3, 85, 29, 15, 4, 4, 3, 2, 2, 132, 48, 35, 6, 3, 85, 29, 37, 4, 28,
                                48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3,
                                2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22,
                                128, 20, 91, 158, 182, 50, 210, 58, 215, 60, 146, 130, 224, 89,
                                244, 248, 194, 220, 81, 27, 114, 187, 48, 17, 6, 10, 43, 6, 1, 4,
                                1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 6, 48, 20, 6, 8, 42, 134, 72,
                                206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0,
                                48, 68, 2, 32, 69, 239, 48, 104, 93, 212, 203, 178, 187, 67, 63,
                                154, 210, 49, 85, 209, 76, 188, 21, 250, 253, 7, 96, 52, 170, 89,
                                36, 162, 78, 223, 155, 47, 2, 32, 124, 251, 64, 23, 22, 28, 147,
                                141, 122, 57, 19, 88, 60, 178, 189, 44, 180, 196, 80, 110, 116,
                                198, 103, 81, 244, 77, 56, 117, 245, 141, 34, 222,
                            ],
                            ty: "X.509",
                        },
                    }]),
                    platform_version: 6,
                    serial: 1_580_734_505_008,
                },
                NodeInfo {
                    addresses: amqp::List(vec![NetworkHostAndPort {
                        host: "localhost",
                        port: 10008,
                    }]),
                    legal_identities_and_certs: amqp::List(vec![PartyAndCertificate {
                        cert_path: CertPath {
                            data: &[
                                48, 130, 8, 211, 48, 130, 2, 9, 48, 130, 1, 176, 160, 3, 2, 1, 2,
                                2, 8, 112, 87, 180, 169, 203, 106, 74, 231, 48, 10, 6, 8, 42, 134,
                                72, 206, 61, 4, 3, 2, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3, 12,
                                18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82, 111, 111,
                                116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2, 82, 51, 49,
                                14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100, 97, 49, 15,
                                48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111, 110, 49, 11,
                                48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13, 49, 55, 48,
                                53, 50, 50, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50,
                                48, 48, 48, 48, 48, 48, 48, 90, 48, 88, 49, 27, 48, 25, 6, 3, 85,
                                4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82,
                                111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2,
                                82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100,
                                97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111,
                                110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 89, 48, 19,
                                6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1,
                                7, 3, 66, 0, 4, 105, 102, 232, 177, 71, 173, 89, 51, 126, 229, 7,
                                138, 125, 250, 38, 185, 181, 105, 67, 26, 173, 127, 205, 81, 149,
                                240, 242, 127, 32, 74, 34, 102, 177, 147, 229, 80, 74, 101, 39,
                                231, 255, 151, 54, 76, 147, 46, 68, 100, 244, 237, 9, 200, 134,
                                223, 138, 103, 56, 253, 30, 136, 199, 127, 37, 48, 163, 100, 48,
                                98, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 124, 174, 169, 223,
                                185, 72, 1, 43, 19, 137, 11, 154, 230, 69, 133, 28, 57, 23, 7, 115,
                                48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29,
                                37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1,
                                5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 15, 6, 3, 85, 29, 19, 1, 1,
                                255, 4, 5, 48, 3, 1, 1, 255, 48, 10, 6, 8, 42, 134, 72, 206, 61, 4,
                                3, 2, 3, 71, 0, 48, 68, 2, 32, 13, 162, 248, 74, 11, 138, 176, 215,
                                147, 79, 180, 158, 82, 65, 93, 160, 32, 64, 9, 225, 188, 26, 163,
                                184, 51, 82, 165, 126, 41, 169, 133, 12, 2, 32, 40, 170, 211, 72,
                                67, 106, 116, 249, 91, 198, 211, 46, 161, 59, 24, 129, 174, 166,
                                87, 255, 59, 192, 0, 6, 90, 86, 173, 72, 58, 186, 37, 1, 48, 130,
                                2, 94, 48, 130, 1, 251, 160, 3, 2, 1, 2, 2, 8, 29, 86, 250, 193,
                                221, 209, 30, 18, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6,
                                8, 42, 134, 72, 206, 61, 3, 1, 7, 48, 88, 49, 27, 48, 25, 6, 3, 85,
                                4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82,
                                111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2,
                                82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100,
                                97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111,
                                110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13,
                                49, 56, 48, 55, 49, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55,
                                48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 99, 49, 11, 48, 9,
                                6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 17, 48, 15, 6, 3, 85, 4, 7, 19,
                                8, 78, 101, 119, 32, 89, 111, 114, 107, 49, 14, 48, 12, 6, 3, 85,
                                4, 11, 19, 5, 67, 111, 114, 100, 97, 49, 22, 48, 20, 6, 3, 85, 4,
                                10, 19, 13, 82, 51, 32, 72, 111, 108, 100, 67, 111, 32, 76, 76, 67,
                                49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 67, 111, 114, 100, 97, 32,
                                68, 111, 111, 114, 109, 97, 110, 32, 67, 65, 48, 89, 48, 19, 6, 7,
                                42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3,
                                66, 0, 4, 3, 203, 222, 160, 38, 225, 102, 102, 179, 151, 34, 5, 90,
                                49, 49, 5, 223, 43, 187, 147, 152, 116, 85, 189, 103, 214, 67, 228,
                                21, 96, 253, 219, 48, 116, 160, 184, 118, 81, 204, 29, 239, 236,
                                67, 132, 241, 22, 105, 24, 54, 167, 251, 13, 52, 237, 120, 251, 93,
                                29, 62, 196, 111, 82, 198, 96, 163, 129, 153, 48, 129, 150, 48, 29,
                                6, 3, 85, 29, 14, 4, 22, 4, 20, 235, 238, 46, 48, 21, 41, 64, 174,
                                25, 152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19, 48, 15, 6,
                                3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85,
                                29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48,
                                26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2,
                                6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128,
                                20, 124, 174, 169, 223, 185, 72, 1, 43, 19, 137, 11, 154, 230, 69,
                                133, 28, 57, 23, 7, 115, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138,
                                98, 1, 1, 4, 3, 2, 1, 1, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3,
                                2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68, 2, 32,
                                25, 179, 65, 122, 103, 10, 143, 94, 3, 25, 33, 192, 203, 116, 108,
                                26, 245, 67, 64, 15, 37, 113, 116, 42, 224, 81, 176, 80, 86, 1,
                                174, 146, 2, 32, 96, 199, 161, 188, 27, 210, 35, 236, 31, 62, 203,
                                174, 72, 180, 178, 149, 114, 206, 202, 135, 164, 93, 158, 190, 127,
                                215, 48, 127, 200, 252, 31, 174, 48, 130, 2, 132, 48, 130, 2, 31,
                                160, 3, 2, 1, 2, 2, 8, 4, 193, 40, 158, 84, 33, 190, 222, 48, 20,
                                6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3,
                                1, 7, 48, 99, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 17,
                                48, 15, 6, 3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89, 111, 114, 107,
                                49, 14, 48, 12, 6, 3, 85, 4, 11, 19, 5, 67, 111, 114, 100, 97, 49,
                                22, 48, 20, 6, 3, 85, 4, 10, 19, 13, 82, 51, 32, 72, 111, 108, 100,
                                67, 111, 32, 76, 76, 67, 49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16,
                                67, 111, 114, 100, 97, 32, 68, 111, 111, 114, 109, 97, 110, 32, 67,
                                65, 48, 30, 23, 13, 49, 57, 49, 48, 51, 48, 48, 48, 48, 48, 48, 48,
                                90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48,
                                50, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16,
                                6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97, 109,
                                49, 15, 48, 13, 6, 3, 85, 4, 10, 12, 6, 83, 101, 108, 108, 101,
                                114, 48, 89, 48, 19, 6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42,
                                134, 72, 206, 61, 3, 1, 7, 3, 66, 0, 4, 67, 28, 133, 95, 43, 216,
                                205, 81, 61, 140, 73, 20, 27, 161, 42, 13, 242, 109, 92, 2, 57, 79,
                                103, 168, 60, 193, 68, 7, 120, 101, 55, 73, 161, 205, 172, 50, 26,
                                180, 137, 34, 223, 52, 236, 42, 175, 208, 38, 191, 26, 222, 245,
                                179, 176, 182, 7, 158, 136, 104, 138, 252, 227, 133, 217, 92, 163,
                                129, 227, 48, 129, 224, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20,
                                168, 88, 110, 18, 26, 134, 51, 194, 208, 119, 70, 226, 234, 168,
                                129, 90, 220, 76, 235, 97, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4,
                                5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1, 134,
                                48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7,
                                3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31,
                                6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 235, 238, 46, 48, 21, 41,
                                64, 174, 25, 152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19, 48,
                                17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 4, 48,
                                72, 6, 3, 85, 29, 30, 1, 1, 255, 4, 62, 48, 60, 160, 56, 48, 54,
                                164, 52, 48, 50, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49,
                                18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114,
                                100, 97, 109, 49, 15, 48, 13, 6, 3, 85, 4, 10, 12, 6, 83, 101, 108,
                                108, 101, 114, 161, 0, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2,
                                6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 73, 0, 48, 70, 2, 33, 0,
                                216, 218, 93, 78, 59, 31, 39, 40, 53, 73, 71, 131, 159, 143, 33,
                                206, 75, 10, 54, 221, 62, 96, 183, 127, 155, 113, 115, 247, 193,
                                91, 57, 27, 2, 33, 0, 176, 50, 185, 180, 60, 225, 75, 111, 117, 80,
                                118, 183, 84, 182, 69, 24, 251, 191, 207, 173, 219, 57, 150, 97,
                                171, 45, 134, 136, 125, 0, 50, 96, 48, 130, 1, 216, 48, 130, 1,
                                117, 160, 3, 2, 1, 2, 2, 8, 82, 143, 125, 134, 40, 127, 21, 57, 48,
                                20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206,
                                61, 3, 1, 7, 48, 50, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76,
                                49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114,
                                100, 97, 109, 49, 15, 48, 13, 6, 3, 85, 4, 10, 12, 6, 83, 101, 108,
                                108, 101, 114, 48, 30, 23, 13, 49, 57, 49, 48, 51, 48, 48, 48, 48,
                                48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48,
                                48, 90, 48, 50, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49,
                                18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114,
                                100, 97, 109, 49, 15, 48, 13, 6, 3, 85, 4, 10, 12, 6, 83, 101, 108,
                                108, 101, 114, 48, 42, 48, 5, 6, 3, 43, 101, 112, 3, 33, 0, 63,
                                120, 115, 97, 250, 206, 11, 217, 65, 199, 136, 105, 13, 17, 36,
                                105, 206, 236, 173, 211, 22, 213, 174, 119, 165, 236, 203, 169, 34,
                                140, 1, 178, 163, 129, 153, 48, 129, 150, 48, 29, 6, 3, 85, 29, 14,
                                4, 22, 4, 20, 153, 126, 14, 158, 38, 185, 194, 178, 44, 167, 168,
                                64, 215, 179, 57, 178, 236, 168, 242, 241, 48, 15, 6, 3, 85, 29,
                                19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4,
                                4, 3, 2, 2, 132, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43,
                                6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29,
                                37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 168, 88,
                                110, 18, 26, 134, 51, 194, 208, 119, 70, 226, 234, 168, 129, 90,
                                220, 76, 235, 97, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138, 98, 1,
                                1, 4, 3, 2, 1, 6, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6,
                                8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68, 2, 32, 37, 7,
                                247, 198, 67, 28, 180, 5, 62, 115, 22, 223, 130, 171, 177, 27, 194,
                                184, 151, 48, 27, 132, 243, 113, 33, 5, 2, 20, 126, 147, 11, 175,
                                2, 32, 61, 228, 82, 18, 136, 29, 221, 50, 48, 32, 119, 14, 197, 91,
                                208, 33, 6, 243, 123, 231, 65, 22, 17, 29, 205, 107, 124, 123, 219,
                                37, 197, 155,
                            ],
                            ty: "X.509",
                        },
                    }]),
                    platform_version: 4,
                    serial: 1_572_437_668_928,
                },
                NodeInfo {
                    addresses: amqp::List(vec![NetworkHostAndPort {
                        host: "localhost",
                        port: 10002,
                    }]),
                    legal_identities_and_certs: amqp::List(vec![PartyAndCertificate {
                        cert_path: CertPath {
                            data: &[
                                48, 130, 8, 245, 48, 130, 2, 9, 48, 130, 1, 176, 160, 3, 2, 1, 2,
                                2, 8, 112, 87, 180, 169, 203, 106, 74, 231, 48, 10, 6, 8, 42, 134,
                                72, 206, 61, 4, 3, 2, 48, 88, 49, 27, 48, 25, 6, 3, 85, 4, 3, 12,
                                18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82, 111, 111,
                                116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2, 82, 51, 49,
                                14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100, 97, 49, 15,
                                48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111, 110, 49, 11,
                                48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13, 49, 55, 48,
                                53, 50, 50, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50,
                                48, 48, 48, 48, 48, 48, 48, 90, 48, 88, 49, 27, 48, 25, 6, 3, 85,
                                4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82,
                                111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2,
                                82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100,
                                97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111,
                                110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 89, 48, 19,
                                6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1,
                                7, 3, 66, 0, 4, 105, 102, 232, 177, 71, 173, 89, 51, 126, 229, 7,
                                138, 125, 250, 38, 185, 181, 105, 67, 26, 173, 127, 205, 81, 149,
                                240, 242, 127, 32, 74, 34, 102, 177, 147, 229, 80, 74, 101, 39,
                                231, 255, 151, 54, 76, 147, 46, 68, 100, 244, 237, 9, 200, 134,
                                223, 138, 103, 56, 253, 30, 136, 199, 127, 37, 48, 163, 100, 48,
                                98, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 124, 174, 169, 223,
                                185, 72, 1, 43, 19, 137, 11, 154, 230, 69, 133, 28, 57, 23, 7, 115,
                                48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29,
                                37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1,
                                5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 15, 6, 3, 85, 29, 19, 1, 1,
                                255, 4, 5, 48, 3, 1, 1, 255, 48, 10, 6, 8, 42, 134, 72, 206, 61, 4,
                                3, 2, 3, 71, 0, 48, 68, 2, 32, 13, 162, 248, 74, 11, 138, 176, 215,
                                147, 79, 180, 158, 82, 65, 93, 160, 32, 64, 9, 225, 188, 26, 163,
                                184, 51, 82, 165, 126, 41, 169, 133, 12, 2, 32, 40, 170, 211, 72,
                                67, 106, 116, 249, 91, 198, 211, 46, 161, 59, 24, 129, 174, 166,
                                87, 255, 59, 192, 0, 6, 90, 86, 173, 72, 58, 186, 37, 1, 48, 130,
                                2, 94, 48, 130, 1, 251, 160, 3, 2, 1, 2, 2, 8, 29, 86, 250, 193,
                                221, 209, 30, 18, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6,
                                8, 42, 134, 72, 206, 61, 3, 1, 7, 48, 88, 49, 27, 48, 25, 6, 3, 85,
                                4, 3, 12, 18, 67, 111, 114, 100, 97, 32, 78, 111, 100, 101, 32, 82,
                                111, 111, 116, 32, 67, 65, 49, 11, 48, 9, 6, 3, 85, 4, 10, 12, 2,
                                82, 51, 49, 14, 48, 12, 6, 3, 85, 4, 11, 12, 5, 99, 111, 114, 100,
                                97, 49, 15, 48, 13, 6, 3, 85, 4, 7, 12, 6, 76, 111, 110, 100, 111,
                                110, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 75, 48, 30, 23, 13,
                                49, 56, 48, 55, 49, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 55,
                                48, 53, 50, 48, 48, 48, 48, 48, 48, 48, 90, 48, 99, 49, 11, 48, 9,
                                6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 17, 48, 15, 6, 3, 85, 4, 7, 19,
                                8, 78, 101, 119, 32, 89, 111, 114, 107, 49, 14, 48, 12, 6, 3, 85,
                                4, 11, 19, 5, 67, 111, 114, 100, 97, 49, 22, 48, 20, 6, 3, 85, 4,
                                10, 19, 13, 82, 51, 32, 72, 111, 108, 100, 67, 111, 32, 76, 76, 67,
                                49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 67, 111, 114, 100, 97, 32,
                                68, 111, 111, 114, 109, 97, 110, 32, 67, 65, 48, 89, 48, 19, 6, 7,
                                42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3,
                                66, 0, 4, 3, 203, 222, 160, 38, 225, 102, 102, 179, 151, 34, 5, 90,
                                49, 49, 5, 223, 43, 187, 147, 152, 116, 85, 189, 103, 214, 67, 228,
                                21, 96, 253, 219, 48, 116, 160, 184, 118, 81, 204, 29, 239, 236,
                                67, 132, 241, 22, 105, 24, 54, 167, 251, 13, 52, 237, 120, 251, 93,
                                29, 62, 196, 111, 82, 198, 96, 163, 129, 153, 48, 129, 150, 48, 29,
                                6, 3, 85, 29, 14, 4, 22, 4, 20, 235, 238, 46, 48, 21, 41, 64, 174,
                                25, 152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19, 48, 15, 6,
                                3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 11, 6, 3, 85,
                                29, 15, 4, 4, 3, 2, 1, 134, 48, 35, 6, 3, 85, 29, 37, 4, 28, 48,
                                26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5, 7, 3, 2,
                                6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128,
                                20, 124, 174, 169, 223, 185, 72, 1, 43, 19, 137, 11, 154, 230, 69,
                                133, 28, 57, 23, 7, 115, 48, 17, 6, 10, 43, 6, 1, 4, 1, 131, 138,
                                98, 1, 1, 4, 3, 2, 1, 1, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3,
                                2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0, 48, 68, 2, 32,
                                25, 179, 65, 122, 103, 10, 143, 94, 3, 25, 33, 192, 203, 116, 108,
                                26, 245, 67, 64, 15, 37, 113, 116, 42, 224, 81, 176, 80, 86, 1,
                                174, 146, 2, 32, 96, 199, 161, 188, 27, 210, 35, 236, 31, 62, 203,
                                174, 72, 180, 178, 149, 114, 206, 202, 135, 164, 93, 158, 190, 127,
                                215, 48, 127, 200, 252, 31, 174, 48, 130, 2, 148, 48, 130, 2, 49,
                                160, 3, 2, 1, 2, 2, 8, 35, 216, 181, 181, 84, 142, 206, 251, 48,
                                20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206,
                                61, 3, 1, 7, 48, 99, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83,
                                49, 17, 48, 15, 6, 3, 85, 4, 7, 19, 8, 78, 101, 119, 32, 89, 111,
                                114, 107, 49, 14, 48, 12, 6, 3, 85, 4, 11, 19, 5, 67, 111, 114,
                                100, 97, 49, 22, 48, 20, 6, 3, 85, 4, 10, 19, 13, 82, 51, 32, 72,
                                111, 108, 100, 67, 111, 32, 76, 76, 67, 49, 25, 48, 23, 6, 3, 85,
                                4, 3, 19, 16, 67, 111, 114, 100, 97, 32, 68, 111, 111, 114, 109,
                                97, 110, 32, 67, 65, 48, 30, 23, 13, 49, 57, 49, 48, 51, 48, 48,
                                48, 48, 48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48,
                                48, 48, 48, 90, 48, 59, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78,
                                76, 49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101,
                                114, 100, 97, 109, 49, 24, 48, 22, 6, 3, 85, 4, 10, 12, 15, 86, 97,
                                108, 117, 101, 88, 32, 45, 32, 78, 111, 116, 97, 114, 121, 48, 89,
                                48, 19, 6, 7, 42, 134, 72, 206, 61, 2, 1, 6, 8, 42, 134, 72, 206,
                                61, 3, 1, 7, 3, 66, 0, 4, 231, 255, 44, 3, 246, 17, 25, 253, 225,
                                250, 76, 83, 224, 54, 188, 212, 183, 179, 10, 83, 146, 166, 111,
                                105, 115, 42, 179, 100, 173, 205, 212, 223, 28, 6, 83, 83, 184, 34,
                                29, 172, 153, 219, 36, 144, 234, 124, 96, 232, 41, 28, 236, 138,
                                242, 103, 171, 3, 248, 215, 52, 35, 181, 237, 232, 184, 163, 129,
                                236, 48, 129, 233, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 154,
                                131, 149, 237, 125, 55, 168, 209, 149, 14, 185, 246, 181, 201, 230,
                                2, 40, 38, 15, 73, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48,
                                3, 1, 1, 255, 48, 11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 1, 134, 48, 35,
                                6, 3, 85, 29, 37, 4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6,
                                8, 43, 6, 1, 5, 5, 7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85,
                                29, 35, 4, 24, 48, 22, 128, 20, 235, 238, 46, 48, 21, 41, 64, 174,
                                25, 152, 30, 216, 111, 227, 125, 127, 7, 162, 194, 19, 48, 17, 6,
                                10, 43, 6, 1, 4, 1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 4, 48, 81, 6,
                                3, 85, 29, 30, 1, 1, 255, 4, 71, 48, 69, 160, 65, 48, 63, 164, 61,
                                48, 59, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49, 18, 48,
                                16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114, 100, 97,
                                109, 49, 24, 48, 22, 6, 3, 85, 4, 10, 12, 15, 86, 97, 108, 117,
                                101, 88, 32, 45, 32, 78, 111, 116, 97, 114, 121, 161, 0, 48, 20, 6,
                                8, 42, 134, 72, 206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1,
                                7, 3, 71, 0, 48, 68, 2, 32, 120, 163, 128, 129, 110, 187, 254, 251,
                                179, 57, 18, 144, 45, 225, 13, 132, 218, 8, 143, 168, 17, 131, 50,
                                192, 27, 64, 190, 205, 154, 72, 229, 20, 2, 32, 33, 105, 226, 98,
                                42, 40, 79, 254, 154, 71, 62, 21, 206, 234, 161, 20, 217, 2, 236,
                                235, 199, 34, 151, 230, 255, 2, 167, 122, 77, 219, 150, 115, 48,
                                130, 1, 234, 48, 130, 1, 135, 160, 3, 2, 1, 2, 2, 8, 77, 235, 139,
                                63, 191, 57, 108, 135, 48, 20, 6, 8, 42, 134, 72, 206, 61, 4, 3, 2,
                                6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 48, 59, 49, 11, 48, 9, 6, 3,
                                85, 4, 6, 19, 2, 78, 76, 49, 18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65,
                                109, 115, 116, 101, 114, 100, 97, 109, 49, 24, 48, 22, 6, 3, 85, 4,
                                10, 12, 15, 86, 97, 108, 117, 101, 88, 32, 45, 32, 78, 111, 116,
                                97, 114, 121, 48, 30, 23, 13, 49, 57, 49, 48, 51, 48, 48, 48, 48,
                                48, 48, 48, 90, 23, 13, 50, 55, 48, 53, 50, 48, 48, 48, 48, 48, 48,
                                48, 90, 48, 59, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 78, 76, 49,
                                18, 48, 16, 6, 3, 85, 4, 7, 12, 9, 65, 109, 115, 116, 101, 114,
                                100, 97, 109, 49, 24, 48, 22, 6, 3, 85, 4, 10, 12, 15, 86, 97, 108,
                                117, 101, 88, 32, 45, 32, 78, 111, 116, 97, 114, 121, 48, 42, 48,
                                5, 6, 3, 43, 101, 112, 3, 33, 0, 186, 172, 105, 246, 26, 136, 217,
                                8, 165, 28, 237, 74, 5, 108, 99, 126, 46, 184, 206, 45, 77, 5, 248,
                                236, 37, 186, 77, 242, 235, 201, 139, 37, 163, 129, 153, 48, 129,
                                150, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 132, 146, 72, 145, 44,
                                30, 244, 165, 107, 6, 10, 219, 114, 94, 93, 122, 138, 113, 215, 0,
                                48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48,
                                11, 6, 3, 85, 29, 15, 4, 4, 3, 2, 2, 132, 48, 35, 6, 3, 85, 29, 37,
                                4, 28, 48, 26, 6, 8, 43, 6, 1, 5, 5, 7, 3, 1, 6, 8, 43, 6, 1, 5, 5,
                                7, 3, 2, 6, 4, 85, 29, 37, 0, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48,
                                22, 128, 20, 154, 131, 149, 237, 125, 55, 168, 209, 149, 14, 185,
                                246, 181, 201, 230, 2, 40, 38, 15, 73, 48, 17, 6, 10, 43, 6, 1, 4,
                                1, 131, 138, 98, 1, 1, 4, 3, 2, 1, 6, 48, 20, 6, 8, 42, 134, 72,
                                206, 61, 4, 3, 2, 6, 8, 42, 134, 72, 206, 61, 3, 1, 7, 3, 71, 0,
                                48, 68, 2, 32, 10, 230, 172, 239, 255, 76, 46, 99, 96, 224, 37,
                                175, 105, 149, 53, 255, 164, 58, 49, 119, 118, 3, 228, 72, 161, 94,
                                115, 176, 253, 111, 191, 215, 2, 32, 32, 93, 78, 7, 228, 110, 114,
                                137, 149, 62, 198, 178, 124, 71, 128, 51, 184, 163, 186, 225, 98,
                                2, 20, 69, 133, 47, 81, 34, 41, 63, 102, 111,
                            ],
                            ty: "X.509",
                        },
                    }]),
                    platform_version: 4,
                    serial: 1_572_437_667_228,
                },
            ]),
        }),
        schema: Schema {
            types: amqp::List(vec![
                TypeNotation::CompositeType(CompositeType {
                    name: "net.corda.core.utilities.Try$Success",
                    label: None,
                    provides: amqp::List(vec![]),
                    descriptor: Descriptor {
                        name: Some(amqp::Symbol("net.corda:e+qsW/cJ4ajGpb8YkJWB1A==")),
                        code: None,
                    },
                    fields: amqp::List(vec![Field {
                        name: "value",
                        ty: "java.lang.Object",
                        requires: amqp::List(vec![]),
                        default: None,
                        label: None,
                        mandatory: true,
                        multiple: false,
                    }]),
                }),
                TypeNotation::RestrictedType(RestrictedType {
                    name: "java.util.List<java.lang.Object>",
                    label: None,
                    provides: amqp::List(vec![]),
                    source: "list",
                    descriptor: Descriptor {
                        name: Some(amqp::Symbol("net.corda:1BLPJgNvsxdvPcbrIQd87g==")),
                        code: None,
                    },
                    choices: amqp::List(vec![]),
                }),
                TypeNotation::CompositeType(CompositeType {
                    name: "net.corda.core.node.NodeInfo",
                    label: None,
                    provides: amqp::List(vec![]),
                    descriptor: Descriptor {
                        name: Some(amqp::Symbol("net.corda:ncUcZzvT9YGn0ItdoWW3QQ==")),
                        code: None,
                    },
                    fields: amqp::List(vec![
                        Field {
                            name: "addresses",
                            ty: "*",
                            requires: amqp::List(vec![
                                "java.util.List<net.corda.core.utilities.NetworkHostAndPort>",
                            ]),
                            default: None,
                            label: None,
                            mandatory: true,
                            multiple: false,
                        },
                        Field {
                            name: "legalIdentitiesAndCerts",
                            ty: "*",
                            requires: amqp::List(vec![
                                "java.util.List<net.corda.core.identity.PartyAndCertificate>",
                            ]),
                            default: None,
                            label: None,
                            mandatory: true,
                            multiple: false,
                        },
                        Field {
                            name: "platformVersion",
                            ty: "int",
                            requires: amqp::List(vec![]),
                            default: Some("0"),
                            label: None,
                            mandatory: true,
                            multiple: false,
                        },
                        Field {
                            name: "serial",
                            ty: "long",
                            requires: amqp::List(vec![]),
                            default: Some("0"),
                            label: None,
                            mandatory: true,
                            multiple: false,
                        },
                    ]),
                }),
                TypeNotation::RestrictedType(RestrictedType {
                    name: "java.util.List<net.corda.core.utilities.NetworkHostAndPort>",
                    label: None,
                    provides: amqp::List(vec![]),
                    source: "list",
                    descriptor: Descriptor {
                        name: Some(amqp::Symbol("net.corda:9xPTMAygl1pGTAXRpGVtiA==")),
                        code: None,
                    },
                    choices: amqp::List(vec![]),
                }),
                TypeNotation::CompositeType(CompositeType {
                    name: "net.corda.core.utilities.NetworkHostAndPort",
                    label: None,
                    provides: amqp::List(vec![]),
                    descriptor: Descriptor {
                        name: Some(amqp::Symbol("net.corda:IA+5d7+UvO6yts6wDzr86Q==")),
                        code: None,
                    },
                    fields: amqp::List(vec![
                        Field {
                            name: "host",
                            ty: "string",
                            requires: amqp::List(vec![]),
                            default: None,
                            label: None,
                            mandatory: true,
                            multiple: false,
                        },
                        Field {
                            name: "port",
                            ty: "int",
                            requires: amqp::List(vec![]),
                            default: Some("0"),
                            label: None,
                            mandatory: true,
                            multiple: false,
                        },
                    ]),
                }),
                TypeNotation::RestrictedType(RestrictedType {
                    name: "java.util.List<net.corda.core.identity.PartyAndCertificate>",
                    label: None,
                    provides: amqp::List(vec![]),
                    source: "list",
                    descriptor: Descriptor {
                        name: Some(amqp::Symbol("net.corda:aWe8j2aAdn/K21a2A3CKmA==")),
                        code: None,
                    },
                    choices: amqp::List(vec![]),
                }),
                TypeNotation::CompositeType(CompositeType {
                    name: "net.corda.core.identity.PartyAndCertificate",
                    label: None,
                    provides: amqp::List(vec![]),
                    descriptor: Descriptor {
                        name: Some(amqp::Symbol("net.corda:GaPpq/rL9KtfTOQDN9ZCbA==")),
                        code: None,
                    },
                    fields: amqp::List(vec![Field {
                        name: "certPath",
                        ty: "java.security.cert.CertPath",
                        requires: amqp::List(vec![]),
                        default: None,
                        label: None,
                        mandatory: true,
                        multiple: false,
                    }]),
                }),
            ]),
        },
        transforms_schema: Some(TransformsSchema {}),
    };

    assert_eq!(rsp, expected);
}
