use std::path::PathBuf;

use actix_web::{middleware::Logger, web, App, HttpServer};
use diesel::{r2d2::ConnectionManager, Connection, SqliteConnection};
use diesel_migrations::{embed_migrations, EmbeddedMigrations, MigrationHarness};
use utoipa_swagger_ui::SwaggerUi;

//To make the schema available
mod api;
pub mod schema;

type DbPool = r2d2::Pool<diesel::r2d2::ConnectionManager<SqliteConnection>>;

include!(concat!(env!("OUT_DIR"), "/generated.rs"));

const MIGRATIONS: EmbeddedMigrations = embed_migrations!();

pub struct Config {
    pub db_path: String,
}

pub async fn gen_docs(path: PathBuf) -> std::io::Result<()> {
    let docs = api::docs();
    tokio::fs::write(&path, docs.to_pretty_json().unwrap().as_bytes()).await
}

pub async fn run(config: Config) -> std::io::Result<()> {
    let manager = ConnectionManager::new(&config.db_path);
    let pool = r2d2::Pool::new(manager).expect("Failed to create db pool");

    if let Ok(mut conn) = SqliteConnection::establish(&config.db_path) {
        match conn.run_pending_migrations(MIGRATIONS) {
            Ok(_) => println!("Migration run success"),
            Err(err) => panic!("Migration run failed: {}", err),
        }
    }

    let docs = api::docs();

    HttpServer::new(move || {
        let generated = generate();
        App::new()
            .app_data(web::Data::<DbPool>::new(pool.clone()))
            .service(SwaggerUi::new("/swagger-ui/{_:.*}").url("/api", docs.clone()))
            .service(api::routes())
            .service(actix_web_static_files::ResourceFiles::new("/", generated))
            .wrap(Logger::default())
    })
    .bind(("127.0.0.1", 9000))?
    .run()
    .await
}
