use crate::DbPool;
use actix_web::{
    get, post,
    web::{self, Data},
    HttpResponse, Responder, Scope,
};

mod models;
use models::{get_recipes, insert_recipy, Recipy};
use utoipa::OpenApi;

#[utoipa::path(
    context_path =  "/api",
    responses(
        (status=200, description="Get all recipes", body=[Recipy])
    )
)]
#[get("/recipes")]
async fn recipes(pool: Data<DbPool>) -> impl Responder {
    if let Ok(conn) = &mut pool.get() {
        if let Ok(recipes) = get_recipes(conn) {
            return HttpResponse::Ok().json(recipes);
        }
    }
    HttpResponse::InternalServerError().finish()
}

#[utoipa::path(
    context_path =  "/api",
    request_body = Recipy,
    responses(
        (status=200, description="Insert new recipy", body=usize)
    )
)]
#[post("/recipy")]
async fn recipy(pool: Data<DbPool>, new_recipy: web::Json<Recipy>) -> impl Responder {
    if let Ok(conn) = &mut pool.get() {
        if let Ok(size) = insert_recipy(conn, new_recipy.into_inner()) {
            return HttpResponse::Ok().body(size.to_string());
        }
    }
    HttpResponse::InternalServerError().finish()
}

pub fn routes() -> Scope {
    web::scope("/api").service(recipes).service(recipy)
}

pub fn docs() -> utoipa::openapi::OpenApi {
    #[derive(OpenApi)]
    #[openapi(handlers(recipes, recipy), components(Recipy))]
    struct ApiDoc;

    ApiDoc::openapi()
}
