use anyhow::{bail, Result};
use serde_json::Value;
use std::env;
use std::fs::File;
use std::io::BufReader;

#[derive(Debug)]
pub(crate) struct Configuration {
    pub image: String,
    pub mountpoint: String,
    pub engine: String,
}

pub(crate) fn build_configuration(basename: &str) -> Result<Configuration> {
    let filename = get_config_filename()?;
    let file = File::open(filename)?;
    let reader = BufReader::new(file);
    let v: Value = serde_json::from_reader(reader)?;

    let profile = profile_for_basename(&v, basename);
    debug!("profile = {:?}", &profile);
    let image = image_for_profile(&v, &profile)?;
    debug!("image = {:?}", &image);
    let mountpoint = mountpoint_for_profile(&v, basename, &profile)?;
    debug!("mountpoint = {:?}", &mountpoint);
    let engine = engine_for_profile(&v, &profile);
    debug!("engine = {:?}", &engine);

    Ok(Configuration {
        image,
        mountpoint,
        engine,
    })
}

fn engine_for_profile(value: &Value, profile: &str) -> String {
    let defaults = &value["defaults"]["engine"];
    let profile = &value["profiles"][&profile]["engine"];
    match profile.is_string() {
        true => profile.as_str().unwrap_or("").trim().to_string(),
        false => match defaults.is_string() {
            true => defaults.as_str().unwrap_or("").trim().to_string(),
            false => String::new(),
        },
    }
}

fn get_config_filename() -> Result<String> {
    let home_key = "HOME";
    let home = env::var(home_key)?;

    let possible_file_paths = [
        String::from("./.container-run.conf"),
        [home.as_str(), ".container-run.conf"].join("/"),
        [home.as_str(), ".config/container-run/container-run.conf"].join("/"),
    ];

    for p in possible_file_paths {
        let f = File::open(&p);
        match f {
            Ok(_) => {
                debug!("found configuration file {:?}", p);
                return Ok(p);
            }
            _ => debug!("configuration file {:?}, not found", p),
        }
    }

    bail!("no configuration file found");
}

fn profile_for_basename(value: &Value, basename: &str) -> String {
    let profile = &value["basenames"][basename]["profile"];
    match profile.is_string() {
        true => String::from(profile.as_str().unwrap_or("defaults")),
        _ => String::from("defaults"),
    }
}

fn image_for_profile(value: &Value, profile: &str) -> Result<String> {
    let default_image = &value["defaults"]["image"];
    let image = &value["profiles"][profile]["image"];
    match image.is_string() {
        true => match image.as_str() {
            Some(i) => Ok(i.to_string()),
            None => bail!("unable to read image value from config"),
        },
        _ => match default_image.is_string() {
            true => match default_image.as_str() {
                Some(i) => Ok(i.to_string()),
                None => bail!("unable to read default image value from config"),
            },
            _ => bail!("no image found for profile {:?}, or default", profile),
        },
    }
}

fn mountpoint_for_profile(value: &Value, basename: &str, profile: &str) -> Result<String> {
    let default_mountpoint = &value["defaults"]["mountpoint"];
    let mountpoint = &value["profiles"][profile]["mountpoint"];
    match mountpoint.is_string() {
        true => match mountpoint.as_str() {
            Some(m) => Ok(m.to_string().replace("{basename}", basename)),
            None => bail!("unable to read mountpoint from config"),
        },
        _ => match default_mountpoint.is_string() {
            true => match default_mountpoint.as_str() {
                Some(m) => Ok(m.to_string().replace("{basename}", basename)),
                None => bail!("unable to read default mountpoint from config"),
            },
            _ => bail!("no mountpoint found for profile {:?}, or default", profile),
        },
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use serde_json::json;

    fn create_valid_configdata() -> Value {
        json!({
            "basenames": {
                "testproject": {"profile": "golang-github"}
            },
            "defaults": {
                "engine": "podman",
                "image": "docker.io/library/bash",
                "mountpoint": "/src/{basename}"
            },
            "profiles": {
                "golang-github": {
                    "engine": "docker",
                    "image": "docker.io/library/golang:latest",
                    "mountpoint": "/go/src/github.com/{basename}"
                }
            }
        })
    }

    #[test]
    fn test_engine_for_profile_found() {
        let configdata = create_valid_configdata();
        let expected = "docker";
        let observed = engine_for_profile(&configdata, "golang-github");
        assert_eq!(observed, expected);
    }

    #[test]
    fn test_engine_for_profile_not_found() {
        let configdata = create_valid_configdata();
        let expected = "podman";
        let observed = engine_for_profile(&configdata, "noprofile");
        assert_eq!(observed, expected);
    }

    #[test]
    fn test_image_for_profile_found() {
        let configdata = create_valid_configdata();
        let expected = "docker.io/library/golang:latest";
        let observed = image_for_profile(&configdata, "golang-github").unwrap();
        assert_eq!(observed, expected);
    }

    #[test]
    fn test_image_for_profile_not_found() {
        let configdata = create_valid_configdata();
        let expected = "docker.io/library/bash";
        let observed = image_for_profile(&configdata, "noprofile").unwrap();
        assert_eq!(observed, expected);
    }

    #[test]
    fn test_mountpoint_for_profile_found() {
        let configdata = create_valid_configdata();
        let expected = "/go/src/github.com/noproject";
        let observed = mountpoint_for_profile(&configdata, "noproject", "golang-github").unwrap();
        assert_eq!(observed, expected);
    }

    #[test]
    fn test_mountpoint_for_profile_not_found() {
        let configdata = create_valid_configdata();
        let expected = "/src/noproject";
        let observed = mountpoint_for_profile(&configdata, "noproject", "nonexistent").unwrap();
        assert_eq!(observed, expected);
    }

    #[test]
    fn test_profile_for_basename_found() {
        let configdata = create_valid_configdata();
        let expected = "golang-github";
        let observed = profile_for_basename(&configdata, "testproject");
        assert_eq!(observed, expected);
    }

    #[test]
    fn test_profile_for_basename_not_found() {
        let configdata = create_valid_configdata();
        let expected = "defaults";
        let observed = profile_for_basename(&configdata, "noproject");
        assert_eq!(observed, expected);
    }
}
